/* JListLogCellRenderer.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.logging;

import java.awt.Color;
import java.awt.Component;
import java.util.logging.Formatter;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import javax.swing.DefaultListCellRenderer;
import javax.swing.Icon;
import javax.swing.JList;

/**
 * Renders log records for use with a {@link JListLogHandler}.
 */
public class JListLogCellRenderer extends DefaultListCellRenderer {
    
    
    //
    private final Formatter formatter;
    
    /**
     * Create a cell renderer of this type which uses the given formatter.
     */
    public JListLogCellRenderer (Formatter formatter) {
        this.formatter = formatter;
    }

    /**
     * Determines the text to be displayed.<p>
     * This implementation uses the formatter to format the message
     * stored in the record.
     */
    protected String toText (LogRecord record) {
        return formatter.formatMessage (record);
    }
    
    /**
     * Determines the foreground color for the text to be displayed.<p>
     * This implementation uses the standard foreground color for messages
     * that are less than warnings, orange when less than severe and red otherwise 
     */
    protected Color toForeground (LogRecord record) {
        int val = record.getLevel().intValue ();
        if (val < Level.WARNING.intValue ())
            return getForeground ();
        else if (val < Level.SEVERE.intValue ())
            return Color.ORANGE;
        else
            return Color.RED;
    }
    
    /**
     * Determines the icon to be displayed next to the message.<p>
     * This implementation always returns null.
     */
    protected Icon toIcon (LogRecord record) {
        return null;
    }
    
    //
    public Component getListCellRendererComponent (JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
        super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
        LogRecord record = (LogRecord)value;
        setText (toText (record));
        setForeground (toForeground (record));
        setIcon (toIcon (record));
        return this;
    }
    
    
    
}
