/* FaceBuilder.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.commands;

import java.util.ArrayList;
import java.util.List;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;
import org.pietschy.command.CommandManager;
import org.pietschy.command.DefaultFaceBuilder;
import org.pietschy.command.Face;
import org.pietschy.command.FaceId;
import org.pietschy.command.FaceManager;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Extension of {@link DefaultFaceBuilder} enabling the following features:
 * <ul>
 * <li>Easy internationalization. Allows
 * for specifying the accelerator key as part of the <tt>text</tt>element as in
 * the following example:
 * <pre>
 *   &lt;text&gt;Javadoc inde_x search [shift F1]&lt;/text&gt;
 * </pre>
 * Accelerator keys, if present, are written between square brackets as the
 * last part of the {@code}-text string and use the conventions of method
 * {@link javax.swing.KeyStroke#getKeyStroke(String)}.<p></li>
 * <li>Dynamic inheritance of faces. When a face property is changed it is now
 * propagated to all extensions of that property.</li>
 * </ul>
 * To use this face builder it should be registered with the face manager by calling
 * {@link #register} before any commands are configured.<p>
 *
 * <b>Important note:</b> work on this part of the Swirl library will be 
 * discontinued because unfortunately newer versions
 * of the <i>GUI commands</i> framework will no longer be available as <i>open source</i>.
 * In due course this class will become deprecated and will eventually be
 * removed from the library.
 *
 */
public class FaceBuilder extends DefaultFaceBuilder {
    
    // should not be instantiated by clients
    private FaceBuilder() {
    }
    
    // singleton
    private static final FaceBuilder SINGLETON = new FaceBuilder();
    
    /**
     * Register this face builder with the command system instead of the
     * default.
     */
    public static void register(CommandManager commandManager) {
        commandManager.getFaceManager().setFaceBuilder(SINGLETON);
    }
    
    /**
     * Process the accelerator key part of the given text and return the
     * text without that part.
     */
    protected String processAcceleratorKey(Face face, String newText) {
        String text = newText.trim();
        int len = text.length();
        if (len == 0)
            return text;
        
        // determine accelerator key, if any
        if (text.charAt(len - 1) == ']') {
            int pos = text.lastIndexOf('[', len - 1);
            if (len == pos + 2) // ends in '[]'
                return text.substring(0, pos).trim();
            else if (pos >= 0) {
                String key = text.substring(pos + 1, len - 1).trim();
                KeyStroke ks = KeyStroke.getKeyStroke(key);
                if (ks == null)
                    throw new IllegalArgumentException("'" + key + "' isn't a valid KeyStroke specification");
                
                face.setAccelerator(ks);
                return text.substring(0, pos).trim();
            } else
                throw new IllegalArgumentException("Invalid format of text element");
            
        } else
            return text; // no accelerator specified
    }
    
    // constants copied verbatim from package private org.pietschy.command.Name
    private static final String TEXT_ELEMENT = "text";
    private static final String HORIZONTAL_TEXT_POSITION_ATTRIBUTE = "horizontal-position";
    private static final String VERTICAL_TEXT_POSITION_ATTRIBUTE = "vertical-position";
    private static final String TEXT_POSITION_LEADING = "leading";
    private static final String TEXT_POSITION_TRAILING = "trailing";
    private static final String TEXT_POSITION_CENTRE = "centre";
    private static final String TEXT_POSITION_TOP = "top";
    private static final String TEXT_POSITION_BOTTOM = "bottom";
    private static final String MNEMONIC_ATTRIBUTE = "mnemonic";
    private static final String MNEMONIC_INDEX_ATTRIBUTE = "mnemonic-index";
    
    // Overrides buildText (copied almost verbatim from source)
    protected void buildText(Face face, Element faceElement) {
        
        // copy of package private org.pietschy.command.Util.getElementsByName
        List<Element> elements = new ArrayList<Element>();
        NodeList nodes = faceElement.getElementsByTagName(TEXT_ELEMENT);
        for (int i = 0; i < nodes.getLength(); i++) {
            Node n = nodes.item(i);
            if (n.getNodeType() == Node.ELEMENT_NODE)
                elements.add((Element)n);
        }
        
        boolean textConfigured = false;
        
        for (Element textElement : elements) {
            
            if (isIncluded(textElement)) {
                
                // copy of private barfIfAlreadyConfigured
                if (textConfigured)
                    throw new RuntimeException("Element configured twice:" + getElementPath(textElement));
                
                textConfigured = true;
                
                if (isEmptyElement(textElement)) {
                    face.setText(null);
                    face.setMnemonic(null);
                    face.setMnemonicIndex(null);
                    face.setTextInherited(false);
                } else {
                    // [kc] new code
                    face.setText(processAcceleratorKey(face, getElementText(textElement)));
                    //face.setText(getElementText(textElement));
                    String horizontalPosition = getAttribute(textElement, HORIZONTAL_TEXT_POSITION_ATTRIBUTE);
                    
                    if (horizontalPosition != null) {
                        if (TEXT_POSITION_LEADING.equals(horizontalPosition.trim())) {
                            face.setHorizontalTextPosition(Integer.valueOf(SwingConstants.LEADING));
                        } else if (TEXT_POSITION_TRAILING.equals(horizontalPosition.trim())) {
                            face.setHorizontalTextPosition(Integer.valueOf(SwingConstants.TRAILING));
                        } else if (TEXT_POSITION_CENTRE.equals(horizontalPosition.trim())) {
                            face.setHorizontalTextPosition(Integer.valueOf(SwingConstants.CENTER));
                        }
                    }
                    
                    String verticalPosition = getAttribute(textElement, VERTICAL_TEXT_POSITION_ATTRIBUTE);
                    
                    if (verticalPosition != null) {
                        if (TEXT_POSITION_TOP.equals(verticalPosition.trim())) {
                            face.setVerticalTextPosition(Integer.valueOf(SwingConstants.TOP));
                        } else if (TEXT_POSITION_BOTTOM.equals(verticalPosition.trim())) {
                            face.setVerticalTextPosition(Integer.valueOf(SwingConstants.BOTTOM));
                        } else if (TEXT_POSITION_CENTRE.equals(verticalPosition.trim())) {
                            face.setVerticalTextPosition(Integer.valueOf(SwingConstants.CENTER));
                        }
                    }
                    
                    String mnemonicString = getAttribute(textElement, MNEMONIC_ATTRIBUTE);
                    
                    
                    if (mnemonicString != null) {
                        if (mnemonicString.length() != 1)
                            throw new RuntimeException("Mnemonic '" + mnemonicString + "' in face " + face.getId() + " is not a single character");
                        
                        KeyStroke ks = KeyStroke.getKeyStroke(mnemonicString.toUpperCase());
                        if (ks != null)
                            face.setMnemonic(Integer.valueOf(ks.getKeyCode()));
                    }
                    
                    try {
                        String mnemonicIndexString = getAttribute(textElement, MNEMONIC_INDEX_ATTRIBUTE);
                        if (mnemonicIndexString != null && mnemonicIndexString.length() != 0)
                            face.setMnemonicIndex(Integer.valueOf(mnemonicIndexString));
                    } catch (NumberFormatException e1) {
                        throw new RuntimeException("Unexpected non integer mnemonic", e1);
                    }
                }
            }
        }
    }
    
    // overrides DefaultFaceBuilder
    protected void buildAccelerator(Face face, Element commandElement) {
        KeyStroke ks = face.getAccelerator();
        super.buildAccelerator(face, commandElement);
        KeyStroke ks2 = face.getAccelerator();
        if (ks2 == null || ks2 == ks) // NOPMD
            face.setAccelerator(ks);
        else if (ks != null)
            throw new RuntimeException("Accelerator specified twice in face " + face.getId());
    }
    
    // overrides DefaultFaceBuilder
    public DynamicFace createFace(FaceId id, FaceManager faceManager) {
        return new DynamicFace(id, faceManager);
    }
    
    
}
