/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDRECORDING_H
#define ISDRECORDING_H

/**
 * Main class that loads and represents a recording.
 * This is the main access class to the library.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>

#include <glibmm/ustring.h>

#include <libinstrudeo/isdobject.h>
#include <libinstrudeo/isdexporter.h>

using namespace std;

//Note: -u means: overwrite existing files with newer versions
//Note: -o means: don't query when overwriting
#define UNPACK(zipFile, unpackFolder) \
     system(("unzip -u -o " + zipFile + " -d " + unpackFolder).c_str())

//Note: -j means: don't insert the directories in the filenames
#define PACK(zipFile, xmlFile, dataFile) \
     system(("zip -j " + zipFile + " " + xmlFile + " " + dataFile).c_str())

#define EXPORT_EXT(sourceFile, targetFile, language, format) \
     system(("isdexport " + sourceFile + " " + targetFile + " " + language + " " + format).c_str())

#define RELATIVE_UNPACK_FOLDER ".unpack"
#define RELATIVE_META_FILE_NAME "meta.xml"
#define RELATIVE_DATA_FILE_NAME "data.isd"

class ISDXmlFile;
class ISDImporter;
class ISDDataFile;
class ISDImage;
class ISDVideoProperties;
class ISDVideoCanvas;
class ISDProgressCallback;

class ISDRecording : public ISDObject
{
    
 public:
    //-----CONSTANTS-----
    enum versionNumber {
	MAJOR_VERSION_NUMBER = 0,
	MINOR_VERSION_NUMBER = 1,
	MICRO_VERSION_NUMBER = 3,
	NANO_VERSION_NUMBER = 0
    };

    //-----CONSTRUCTORS-----
    /**
     * Loads a recording from the location indicated by fileName
     *
     * @param fileName The filename of the file.
     */
    ISDRecording(string fileName);

    /**
     * Loads a foreign file format and imports it in a new recording.
     *
     * @param importer The importer for the foreign file format
     * @param callbackClass Can be set to monitor the progress or NULL to discard that.
     */
    ISDRecording(ISDImporter& importer, ISDProgressCallback* callbackClass = NULL);
    
    ~ISDRecording();

    //-----METHODS-----
    /**
     * Compresses (zips) the two files (meta and data) together to the file specified.
     * If the target file exists, it gets deleted first, so handle backups or warnings
     * BEFORE this method. (if something goes wrong here, the file can still get deleted)
     *
     * @param fileName The filename of the packed file.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode packRecording(string fileName);

    /**
     * Exports this recording to the specified file.
     * The fileformat is specified by the format flag and defaults
     * to auto, which means the fileName extension is used to determine
     * the format.
     * If the target file exists, it gets deleted first, so handle backups or warnings
     * BEFORE this method. (if something goes wrong here, the file can still get deleted)
     *
     * Note: The OpenGL context must be initialised (and will be used) before this is called.
     *
     * Note 2: The optional parameter specifies the pixelbuffer that should be used instead
     *         of grabbing the frames from the framebuffer. If offers the possibility
     *         to use an offscreen rendering context that doesn't support glReadPixels()
     *         (like OSMESA).
     *
     * Note 3: DON'T USE THIS METHOD FOR EXPORTING YOUR RECORDINGS, unless you know what you are doing.
     *         As of libinstrudeo 0.1.2, the isdexport program takes care of exporting a recording,
     *         using this method to really export it, but as there were a lot of GL-context-switch issues
     *         in libinstrudeo < 0.1.2, the isdexport program was introduced.
     *
     * @param fileName The target export file name.
     * @param lang The comment-language to use for the comments, set to NULL to disable comments.
     * @param callbackClass Can be set to monitor the progress or NULL to discard that.
     * @param pixelBuffer An optional offscreen pixel buffer.
     * @param format The export format.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode exportRecording(string fileName, Glib::ustring* lang,
				 ISDProgressCallback* callbackClass,
				 char* pixelBuffer = NULL,
				 ISDExporter::ISDExportFormat format = ISDExporter::ISD_EXPORT_AUTO_FORMAT);

    /**
     * Positions the current video at position pos and updates the video at that position,
     * if necessary. The position is the time in milliseconds and must be
     * between the videos time-bounds.
     *
     * @param pos The position in the videostream.
     * @param lang The current comment-language or NULL to disable comments.
     * @param noVideoChange Is set to true if the image didn't change since last update.
     * @param noCommentChange Is set to true if the commentboxes didn't change since last update.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode update(int pos, Glib::ustring* lang, bool& noVideoChange, bool& noCommentChange);

    /**
     * Returns the xmlFile object.
     *
     * @return The xml file object, NULL when an error occurred.
     */
    ISDXmlFile* getXmlFile();

    /**
     * Returns the videoCanvas object.
     *
     * @return The videoCanvas object, NULL when an error occurred.
     */
    ISDVideoCanvas* getVideoCanvas();

    /**
     * Returns the properties of the video. (length, number of rects, etc.)
     *
     * @return The properties-object, NULL when an error occurred.
     */
    ISDVideoProperties* getVideoProperties();

    /**
     * Returns the dataFile object.
     *
     * @return The datafile-object, NULL when an error occurred.
     */
    ISDDataFile* getDataFile();

 protected:
    //-----METHODS-----
    /**
     * Unpacks the file and initialises the following variables accordingly:
     * xmlFile: the extracted, initialised XML meta file
     * binaryFileName: the path to the extracted binary data file
     * unpackDirectory: the directory where both files were extracted to
     * 
     * @param fileName The filename of the recording to unpack.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode unpackRecordingFile(string fileName);

    /**
     * Removes the unpack directory and it's content.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode deleteUnpackDirectory();
        
    //-----VARIABLES-----
    ISDDataFile* dataFile;
    ISDXmlFile* xmlFile;
    ISDVideoProperties* videoProperties;
    ISDVideoCanvas* videoCanvas;
    string unpackDirectory;
};

#endif
