/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#include <cstdio>
#include <cstdlib>
#include <iostream>

#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/glm.h>
#include <libinstrudeo/isdglobjfile.h>

#undef LOG_HEADER
#define LOG_HEADER "Error while initialising a .obj file: \n"
#include <libinstrudeo/isdloggermacros.h>

//-----CONSTRUCTORS-----
ISDGLObjFile::ISDGLObjFile(string fileName, string meshObjName, string textboxObjName)
    : ISDObject(),
      model(NULL),
      smoothingAngle(DEFAULT_SMOOTHING_ANGLE),
      modelList(0),
      facetNormal(GL_FALSE),
      meshObject(NULL),
      textPlaneObject(NULL),
      textboxWidth(0.0),
      textboxHeight(0.0)
{
    //read and init the model
    initModel(fileName);

    //init references to both objects
    meshObject = glmFindObject(model, (char*)meshObjName.c_str());
    if (!meshObject){
	LOG_WARNING("Could not find the main mesh object in the .obj file.");
	lastError = ISD_INIT_ERROR;
	return;
    }
    textPlaneObject = glmFindObject(model, (char*)textboxObjName.c_str());
    if (!textPlaneObject){
	LOG_WARNING("Could not find the text plane object in the .obj file.");
	lastError = ISD_INIT_ERROR;
	return;
    }

    //extract some useful textbox data
    if (initTextbox() != ISD_SUCCESS) {
	lastError = ISD_INIT_ERROR;
	return;
    }
}

//-----DESTRUCTOR-----
ISDGLObjFile::~ISDGLObjFile()
{
    if (model!=NULL) {
	glmDelete(model);
    }
    
    //these are just convenient references
    meshObject = NULL;
    textPlaneObject = NULL;
}

//-----PUBLIC METHODS-----
GLMmodel* ISDGLObjFile::getMainModel()
{
    //check for errors
    if (error() || model==NULL) {
	return NULL;
    }
    else {
	return model;
    }
}
GLMobject* ISDGLObjFile::getMeshObject()
{
    //check for errors
    if (error() || meshObject==NULL) {
	return NULL;
    }
    else {
	return meshObject;
    }
}
GLuint ISDGLObjFile::getMeshObjectList()
{
    //check for errors
    if (error() || meshObject==NULL) {
	return 0;
    }
    else {
	return modelList;
    }
}
GLMobject* ISDGLObjFile::getTextPlaneObject()
{
    //check for errors
    if (error() || textPlaneObject==NULL) {
	return NULL;
    }
    else {
	return textPlaneObject;
    }
}
ISDObject::ISDErrorCode ISDGLObjFile::getTextPlaneOrigin(float orig[3])
{
    //check for errors
    if (error() || textPlaneObject==NULL || textPlaneObject==NULL) {
	RETURN_ERROR(ISD_INIT_ERROR);
    }

    orig[X] = textboxOrigin[X];
    orig[Y] = textboxOrigin[Y];
    orig[Z] = textboxOrigin[Z];

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDGLObjFile::getTextPlaneDimensions(float* width, float* height)
{
    //check for errors
    if (error() || textPlaneObject==NULL) {
	RETURN_ERROR(ISD_INIT_ERROR);
    }

    *width = textboxWidth;
    *height = textboxHeight;

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDGLObjFile::getDimensions(float* width, float* height, float* depth)
{
    if (meshObject==NULL) {
	return ISD_INIT_ERROR;
    }

    float dim[3];
    glmDimensions(meshObject, (GLfloat*)dim);

    *width = dim[0];
    *height = dim[1];
    *depth = dim[2];

    RETURN_SUCCESS;
}
void ISDGLObjFile::drawModel(GLuint mode)
{
    //check for errors
    if (error() || model==NULL) {
	lastError = ISD_INIT_ERROR;
	return;
    }
    else {
	glmDraw(model, mode);
    }
}

//-----PROTECTED METHODS-----
ISDObject::ISDErrorCode ISDGLObjFile::initModel(string fileName)
{
    //read in the .obj file
    model = glmReadOBJ((char*)fileName.c_str());
    if (!model){
	LOG_WARNING("Error while reading in the .obj file.");
	RETURN_ERROR(ISD_INIT_ERROR);
    }
    
    //scales the model inside a unit cube around the center
    glmUnitize(model);
    //scale one more time to a more comfortable value
    glmScale(model, MODEL_UNIT_SCALE);
    glmFacetNormals(model);
    glmVertexNormals(model, smoothingAngle);

    //check if the model has a material or just a color
    if (model->nummaterials > 0)
	createLists(model, MATERIAL_OBJECT);
    else
	createLists(model, MATERIAL_COLOR);

    RETURN_SUCCESS;
}

ISDObject::ISDErrorCode ISDGLObjFile::createLists(GLMmodel* model, GLuint materialMode)
{
    //delete the list if it exists
    if (modelList)
	glDeleteLists(modelList, 1);

    //generate the list
    if (materialMode == MATERIAL_NONE) { 
	if (facetNormal) {
	    modelList = glmList(model, GLM_FLAT);
	}
	else {
	    modelList = glmList(model, GLM_SMOOTH);
	}
    }
    else if (materialMode == MATERIAL_COLOR) {
	if (facetNormal) {
	    modelList = glmList(model, GLM_FLAT | GLM_COLOR);
	}
	else {
	    modelList = glmList(model, GLM_SMOOTH | GLM_COLOR);
	}
    }
    else if (materialMode == MATERIAL_OBJECT) {
	if (facetNormal) {
	    modelList = glmList(model, GLM_FLAT | GLM_MATERIAL);
	}
	else {
	    modelList = glmList(model, GLM_SMOOTH | GLM_MATERIAL);
	}
    }
    else {
	LOG_WARNING("Invalid or unrecognised option while creating lists.");
	return ISD_INIT_ERROR;
    }
    
    RETURN_SUCCESS;
}

ISDObject::ISDErrorCode ISDGLObjFile::initTextbox()
{
    if (textPlaneObject==NULL) {
	LOG_WARNING("Trying to extract textbox-data without initialising it.");
	RETURN_ERROR(ISD_INIT_ERROR);
    }

    textboxOrigin[X] = textPlaneObject->vertices[TOP_LEFT+X];
    textboxOrigin[Y] = textPlaneObject->vertices[TOP_LEFT+Y];
    textboxOrigin[Z] = textPlaneObject->vertices[TOP_LEFT+Z];
	
    if (textPlaneObject->vertices[TOP_LEFT+X] > textPlaneObject->vertices[TOP_RIGHT+X]) {
	textboxWidth = textPlaneObject->vertices[TOP_LEFT+X] - textPlaneObject->vertices[TOP_RIGHT+X];
    }
    else {
	textboxWidth = textPlaneObject->vertices[TOP_RIGHT+X] - textPlaneObject->vertices[TOP_LEFT+X];
    }
	
    if (textPlaneObject->vertices[TOP_LEFT+Y] > textPlaneObject->vertices[BOTTOM_LEFT+Y]) {
	textboxHeight = textPlaneObject->vertices[TOP_LEFT+Y] - textPlaneObject->vertices[BOTTOM_LEFT+Y];
    }
    else {
	textboxHeight = textPlaneObject->vertices[BOTTOM_LEFT+Y] - textPlaneObject->vertices[TOP_LEFT+Y];
    }

    RETURN_SUCCESS;
}
