/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.
   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



-----------------------------------------------------------------------
FILENAME:                valid.c
LIBRARY:                 libgpiv
EXTERNAL FUNCTIONS:      gpiv_valid_residu
                         gpiv_valid_peaklck
                         gpiv_valid_residu_stats
                         gpiv_valid_errvec
			 gpiv_valid_gradient
                         gpiv_valid_threshold
                         gpiv_cumhisto_eqdatbin_gnuplot

LAST MODIFICATION DATE: $Id: valid.c,v 1.19 2006/01/31 13:30:13 gerber Exp $
--------------------------------------------------------------------- */
#include <gpiv.h>

#define SNR_ERR 99.0
#define MIN_VECTORS4MEDIAN 3    /* Minimum number of valid vectors needed 
                                   to calculate median */
#define RESIDU_EPSI 0.1

/*
 * Local functions
 */

static int
compare_float (const void * a,
               const void * b)
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Compares two float numbers
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     a:              first float number
 *     b:              second float number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     int:            -1, 0 or +1 for a < b, a = b or a > b respectively
 *---------------------------------------------------------------------------*/
{
    float *la = (float *) a, *lb = (float *) b;

    if (*la > *lb)
       return 1;
    else if (*la < *lb)
       return -1;
    else
       return 0;
}


static float
median_residu(gint i, 
              gint j, 
              gint neighbors, 
              GpivPivData data, 
              gint *i_median_x, 
              gint *j_median_x,
              gint *i_median_y, 
              gint *j_median_y,
              gboolean incl_point,
              gboolean norm
              )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     If incl_point. calculates residu value of point i, j from a NxN data 
 *     array out of data[i][j] and the indices of the median velocity
 *     If not incl_point, returns the median residu from NxN data array, 
 *     calculated from dx(i) - dx(median), i,= 1, ..,8
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     i:              horizontal index of data to be investigated
 *     j:              verical index of data to be investigated
 *     neighbors:      number of neighboring PIV data (typically 3x3, 5x5)
 *     data:           piv dataset
 *     incl_point:     flag to include data point under investigation
 *     norm:           flag to return residu for normalizing
 *
 * OUTPUTS:
 *    i_median_x:      i-index of median for x displacement
 *    j_median_x:      j-index of median for x displacement
 *    i_median_y:      i-index of median for y displacement
 *    j_median_y:      j-index of median for y displacement
 *
 * RETURNS:
 *    median residu
 *---------------------------------------------------------------------------*/
{
    int k, l, m = 0;
    gint vlength = 0, median_index;
    const gint N = neighbors - 2;
    float *dx_m, *dy_m, *dx_org, *dy_org;
    float *r_m;
    float residu;
/*
 * Obtain length of array
 */
    for (k = i - N; k <= i + N; k++) {
	if ((k >= 0) && (k < data.ny)) {
	    for (l = j - N; l <= j + N; l++) {
		if ((l >= 0) && (l < data.nx)) {
                    if ((incl_point
                         && data.peak_no[k][l] != -1)
                        || (!incl_point 
                            && (k != i || l != j)
                            && data.peak_no[k][l] != -1) ) {
			vlength++;
		    }
		}
	    }
	}
    }

    if (data.peak_no[i][j] != -1 && vlength >= MIN_VECTORS4MEDIAN ) {
        dx_m = gpiv_vector(vlength);
        dy_m = gpiv_vector(vlength);
        dx_org = gpiv_vector(vlength);
        dy_org = gpiv_vector(vlength);
        if (norm) {
            r_m = gpiv_vector(vlength);
        }
/*
 * Write the absolute neighbouring velocities and its components to a 
 * 1-dimensional array 
 */
        m = 0;
        for (k = i - N; k <= i + N; k++) {
            if ((k >= 0) && (k < data.ny)) {
                for (l = j - N; l <= j + N; l++) {
                    if ((l >= 0) && (l < data.nx)) {
                        if ((incl_point
                             && data.peak_no[k][l] != -1)
                            || (!incl_point 
                                && (k != i || l != j)
                                && data.peak_no[k][l] != -1) ) {
                            dx_m[m] = data.dx[k][l];
                            dx_org[m] = data.dx[k][l];
                            dy_m[m] = data.dy[k][l];
                            dy_org[m] = data.dy[k][l];
                            m++;
                        }
                    }
                }
            }
        }
        
/*
 * Sorting dx_m and dy_m arrays and searching median index and value
 */
        qsort(dx_m, vlength, sizeof(float), compare_float);
        qsort(dy_m, vlength, sizeof(float), compare_float);
        if (incl_point) {
            median_index = (int) (vlength - 1) / 2;
        } else {
            median_index = (int) (vlength) / 2;
        }        

        if (norm) {
/*
 * Obtain all residus from surrounding data, sorting and picking the median
 */
            for (i = 0; i < vlength; i++) {
                r_m[i] = sqrt((dx_m[i] - dx_m[median_index]) *
                            (dx_m[i] - dx_m[median_index]) +
                            (dy_m[i] - dy_m[median_index]) *
                            (dy_m[i] - dy_m[median_index]));
            }
            qsort(r_m, vlength, sizeof(float), compare_float);
            residu = r_m[median_index];
        } else {
/*
 * Obtain residu from difference between current displacement at (i,j) 
 * and median displacement from the surroundings.
 */
            residu = sqrt((data.dx[i][j] - dx_m[median_index]) *
                          (data.dx[i][j] - dx_m[median_index]) +
                          (data.dy[i][j] - dy_m[median_index]) *
                          (data.dy[i][j] - dy_m[median_index]));
        }

/*
 * Search the indexes for the 2-dim arrays dx and dy belonging 
 * to mediaan_index
 */
        m = 0;
        for (k = i - N; k <= i + N; k++) {
            if ((k >= 0) && (k < data.ny)) {
                for (l = j - N; l <= j + N; l++) {
                    if ((l >= 0) && (l < data.nx)) {
                        if ((incl_point
                             && data.peak_no[k][l] != -1)
                            || (!incl_point 
                                && (k != i || l != j)
                                && data.peak_no[k][l] != -1) ) {
                            if (dx_org[m] == dx_m[median_index]) {
                                *i_median_x = k - i;
                                *j_median_x = l - j;
                            }
                            if (dy_org[m] == dy_m[median_index]) {
                                *i_median_y = k - i;
                                *j_median_y = l - j;
                            }
                            m++;
                        }
                    }
                }
            }
        }

        gpiv_free_vector (dx_m);
        gpiv_free_vector (dx_org);
        gpiv_free_vector (dy_m);
        gpiv_free_vector (dy_org);
        if (norm) {
            gpiv_free_vector (r_m);
        }
        
    } else {
        residu = SNR_ERR;
    }
    
    return residu;
}



void 
gpiv_valid_residu(GpivPivData in_data, 
                  GpivPivData * out_data,
                  GpivValidPar piv_valid_par,
                  gboolean incl_point
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculates residu values (at the inner points) of a PIV data set and 
 *     applies to snr member of out_data
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     in_data:        piv dataset
 *     piv_valid_par:  validation parameters
 *     incl_point:     flag to include data point under investigation
 *
 * OUTPUTS:
 *     out_data:       piv dataset containing residu values in snr
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    gint i, j, i_mx, j_mx, i_my, j_my;

    if (piv_valid_par.residu_type == GPIV_VALID_RESIDUTYPE__SNR) {
        for (i = 1; i < in_data.ny - 1; i++) {
            for (j = 1; j < in_data.nx - 1; j++) {
                out_data->snr[i][j] = in_data.snr[i][j];
            }
        }

    } else if (piv_valid_par.residu_type == GPIV_VALID_RESIDUTYPE__MEDIAN) {
        for (i = 1; i < in_data.ny - 1; i++) {
            for (j = 1; j < in_data.nx - 1; j++) {
                out_data->snr[i][j] = median_residu(i, j, 
                                                    piv_valid_par.neighbors,
                                                    in_data, &i_mx, &j_mx,
                                                    &i_my, &j_my,
                                                    incl_point, FALSE);
                if (out_data->snr[i][j] == SNR_ERR) 
                    out_data->peak_no[i][j] = -1;
            }
        }

    } else if (piv_valid_par.residu_type == GPIV_VALID_RESIDUTYPE__NORMMEDIAN) {
        gfloat residu_from_median_dxdy = 1.111, residu_norm = 5.555;
        for (i = 0; i < in_data.ny; i++) {
            for (j = 0; j < in_data.nx; j++) {
                 residu_norm = 
                     median_residu(i, j, piv_valid_par.neighbors, in_data, 
                                   &i_mx, &j_mx, &i_my, &j_my, FALSE, TRUE);
                 residu_from_median_dxdy = 
                     median_residu(i, j, piv_valid_par.neighbors, in_data, 
                                   &i_mx, &j_mx, &i_my, &j_my, FALSE, FALSE);
                 if (residu_norm + RESIDU_EPSI != 0.0) {
                     out_data->snr[i][j] = 
                         residu_from_median_dxdy / 
                         (residu_norm + RESIDU_EPSI);
                 } else {
                     out_data->snr[i][j] = SNR_ERR;
                     out_data->peak_no[i][j] = -1;
                 }
            }
        }

    } else {
        gpiv_warning("GPIV VALID RESIDU: should no arrive here");
    }

}



static void
interrogate_point(int index_y, 
                  int index_x,
                  guint16 **img1, 
                  guint16 **img2, 
                  GpivImagePar image_par, 
                  GpivEvalPar piv_eval_par,
                  GpivPivData * piv_data
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Interrogates the image(pair) on a single point
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     img1:           first image
 *     img2:           second image
 *     image_par:      image parameters
 *     piv_eval_par:   evaluation parameters
 *
 * OUTPUTS:
 *     piv_data:       piv dataset containing particle image displacements
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL, c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    int nc_lines = 0;

    float **intreg1;
    float **intreg2;
    int int_size_0;

    GpivImagePar image_par_ACT;
    GpivEvalPar piv_eval_par_ACT;
    Covariance cov, w_k;
    int sweep = 1, sweep_last = 0;
    int return_val;
    guint16 **img1_ACT = NULL, **img2_ACT = NULL;   
    int cmpr = 1, cmpr_fact = 1;


/*
 * Local (actualized) parameters
 */
    image_par_ACT.ncolumns = image_par.ncolumns;
    image_par_ACT.nrows = image_par.nrows;
    image_par_ACT.depth = image_par.depth;
    image_par_ACT.x_corr = image_par.x_corr;

    image_par_ACT.ncolumns_logic = TRUE;
    image_par_ACT.nrows_logic = TRUE;
    image_par_ACT.x_corr_logic = TRUE;


    gpiv_piv_cp_parameters(piv_eval_par, &piv_eval_par_ACT, TRUE, FALSE);

    if (piv_eval_par_ACT.ad_int == 1) {
        piv_eval_par_ACT.int_size_1 = piv_eval_par.int_size_2;
    } else {
        piv_eval_par_ACT.int_size_1 = piv_eval_par.int_size_1;
    }

    if (piv_eval_par.int_scheme == GPIV_ZERO_OFF_FORWARD
        || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL
        || piv_eval_par.ad_int) {
        sweep_last = 0;
        piv_eval_par_ACT.ifit = 0;
    } else {
        sweep_last = 1;
        piv_eval_par_ACT.ifit = piv_eval_par.ifit;
    }

    img1_ACT = &img1[0];
    img2_ACT = &img2[0];
 
/*
 * Reads eventually existing fftw wisdom
 */
    gpiv_fread_fftw_wisdom(1);
    gpiv_fread_fftw_wisdom(-1);

    while (((piv_eval_par_ACT.int_scheme != GPIV_ZERO_OFF_FORWARD
             && piv_eval_par_ACT.int_scheme != GPIV_ZERO_OFF_CENTRAL)
            && sweep <= 1) 
	   || ((piv_eval_par_ACT.int_scheme == GPIV_ZERO_OFF_FORWARD
                || piv_eval_par_ACT.int_scheme == GPIV_ZERO_OFF_CENTRAL)
               && piv_eval_par_ACT.ad_int != 1 
               && sweep <= GPIV_MAX_EVAL_SWEEP )
           || (piv_eval_par_ACT.ad_int == 1 
               && piv_eval_par_ACT.int_size_2 >= 
               (piv_eval_par.int_size_1 - GPIV_DIFF_ISI) / cmpr_fact )
           ) {
/*
 * Memory allocation of interrogation area's
 */
	int_size_0 = 2 * piv_eval_par_ACT.int_size_2;
	intreg1 = gpiv_matrix(int_size_0, int_size_0);
	intreg2 = gpiv_matrix(int_size_0, int_size_0);
	memset(intreg1[0], 0, (sizeof(float)) * int_size_0 * int_size_0);
	memset(intreg2[0], 0, (sizeof(float)) * int_size_0 * int_size_0);
        

/*
 * Memory allocation of the packed interrogation area arrays.
 * Define weight kernel values
 */
	/*  After resizing piv_eval_par_ACT.int_size_2?? */
	gpiv_piv_bounds_cov(&cov, int_size_0, image_par);
	gpiv_piv_bounds_cov(&w_k, int_size_0, image_par);
	cov.z = gpiv_matrix_index(cov.z_rl, cov.z_rh, cov.z_cl, cov.z_ch);
	w_k.z = gpiv_matrix_index(w_k.z_rl, w_k.z_rh, w_k.z_cl, w_k.z_ch);

	if (piv_eval_par_ACT.int_scheme == GPIV_LK_WEIGHT) {
	    gpiv_piv_weight_kernel_lin(&w_k, int_size_0);
	} else {
	    gpiv_piv_weight_kernel_1(&w_k);
	}


/*
 * Interrogation at a single point
 */
        if ((err_msg = 
             gpiv_piv_interr_reg(index_y, index_x, img1_ACT, img2_ACT, 
                                 intreg1, intreg2, &cov, &w_k, piv_data,
                                 sweep, sweep_last, image_par_ACT, 
                                 piv_eval_par_ACT))
            != NULL) gpiv_error ("LIBGPIV valid: interrogate_point: %s", err_msg);

/*
 * Freeing memory: smaller sizes are needed for eventually next sweep
 */
	gpiv_free_matrix(intreg1);
	gpiv_free_matrix(intreg2);
	gpiv_free_matrix_index(cov.z, cov.z_rl, cov.z_rh, cov.z_cl, cov.z_ch);
	gpiv_free_matrix_index(w_k.z, w_k.z_rl, w_k.z_rh, w_k.z_cl, w_k.z_ch);

/*
 * Adjust int_size_2_ACT in case int_size_1
 */
	if (piv_eval_par_ACT.ad_int == 1) {
            if (piv_eval_par_ACT.int_size_2 * cmpr_fact != 
                piv_eval_par.int_size_1 
                && sweep_last == 0
                && (piv_eval_par_ACT.int_size_2 * cmpr_fact) / 2 <= 
                piv_eval_par.int_size_1) { 
                sweep_last = 1;
                piv_eval_par_ACT.int_size_1 = 
                    (piv_eval_par.int_size_1 - GPIV_DIFF_ISI) / cmpr_fact;
                piv_eval_par_ACT.int_size_2 = 
                    piv_eval_par.int_size_1 / cmpr_fact;
                piv_eval_par_ACT.ifit = piv_eval_par.ifit;
	     } else {
                 piv_eval_par_ACT.int_size_1 = piv_eval_par_ACT.int_size_2 / 2;
                 piv_eval_par_ACT.int_size_2 = piv_eval_par_ACT.int_size_2 / 2;
	     }
	}

        if ((piv_eval_par.int_scheme == GPIV_ZERO_OFF_FORWARD
             || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL)
            && piv_eval_par_ACT.ad_int != 1) {
            sweep_last = 1;
            piv_eval_par_ACT.ifit = piv_eval_par.ifit;
            piv_eval_par_ACT.int_size_1 = (piv_eval_par.int_size_1 - GPIV_DIFF_ISI)
                / cmpr_fact;
            piv_eval_par_ACT.int_size_2 = piv_eval_par.int_size_1/ cmpr_fact;
       }


       sweep++;
    }

/*
 * Writes existing fftw wisdom
 */
    gpiv_fwrite_fftw_wisdom(1);
    gpiv_fwrite_fftw_wisdom(-1);

}



static void
interr_reg_nhpeak(int index_y, 
                  int index_x,
                  guint16 **img1, 
                  guint16 **img2, 
                  GpivImagePar image_par, 
                  GpivEvalPar piv_eval_par,
                  GpivPivData * piv_data
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Interrogates the image(pair) at a single region at the next higher
 *     correlation peak from a previous analysis
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     img1:           first image
 *     img2:           second image
 *     image_par:      image parameters
 *     piv_eval_par:   evaluation parameters
 *
 * OUTPUTS:
 *     piv_data:       piv dataset containing particle image displacements
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    int nc_lines = 0;

    float **intreg1;
    float **intreg2;
    int int_size_0;

    GpivEvalPar piv_eval_par_ACT;
    Covariance cov, w_k;
    int sweep = 1, sweep_last = 1;
    int return_val;
    int cmpr = 1, cmpr_fact = 1;

/*
 * Checking for memory allocation of input variables
 */
    assert(img1[0] != NULL);
    assert(img2[0] != NULL);

/*
 * Local (actualized) parameters
 */
    gpiv_piv_cp_parameters(piv_eval_par, &piv_eval_par_ACT, TRUE, FALSE);
    piv_eval_par_ACT.peak = piv_data->peak_no[index_y][index_x] + 1;

/*
 * Reads eventually existing fftw wisdom
 */
    gpiv_fread_fftw_wisdom(1);
    gpiv_fread_fftw_wisdom(-1);

/*
 * Memory allocation of interrogation area's and packed interrogation area 
 * arrays. Define weight kernel values
 */
    int_size_0 = 2 * piv_eval_par_ACT.int_size_2;
    intreg1 = gpiv_matrix(int_size_0, int_size_0);
    intreg2 = gpiv_matrix(int_size_0, int_size_0);
    memset(intreg1[0], 0, (sizeof(float)) * int_size_0 * int_size_0);
    memset(intreg2[0], 0, (sizeof(float)) * int_size_0 * int_size_0);

    gpiv_piv_bounds_cov(&cov, int_size_0, image_par);
    gpiv_piv_bounds_cov(&w_k, int_size_0, image_par);
    cov.z = gpiv_matrix_index(cov.z_rl, cov.z_rh, cov.z_cl, cov.z_ch);
    w_k.z = gpiv_matrix_index(w_k.z_rl, w_k.z_rh, w_k.z_cl, w_k.z_ch);

    if (piv_eval_par_ACT.int_scheme == GPIV_LK_WEIGHT) {
        gpiv_piv_weight_kernel_lin(&w_k, int_size_0);
    } else {
        gpiv_piv_weight_kernel_1(&w_k);
    }

/*
 * Interrogate at a single point
 */
    gpiv_piv_interr_reg(index_y, index_x, img1, img2, intreg1, intreg2, &cov, 
                        &w_k, piv_data,	sweep, sweep_last, image_par, 
                        piv_eval_par_ACT);

/*
 * Freeing memory
 */
    gpiv_free_matrix(intreg1);
    gpiv_free_matrix(intreg2);
    gpiv_free_matrix_index(cov.z, cov.z_rl, cov.z_rh, cov.z_cl, cov.z_ch);
    gpiv_free_matrix_index(w_k.z, w_k.z_rl, w_k.z_rh, w_k.z_cl, w_k.z_ch);

/*
 * Writes existing fftw wisdom
 */
/*     gpiv_fwrite_fftw_wisdom(1); */
/*     gpiv_fwrite_fftw_wisdom(-1); */

}



static void
nh_corrpeak(int index_y,
           int index_x,
           GpivImagePar image_par,
           GpivEvalPar *piv_eval_par,
           GpivPivData in_data,
           GpivPivData * out_data,
           guint16 **img1, 
           guint16 **img2 
           )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 * Validates piv image on next highest correlation peak
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     index_y:        y-index
 *     index_x:        x-index
 *     image_par:      image parameters
 *     piv_eval_par:   evaluation parameters
 *     in_data:        piv dataset containing particle image displacements
 *     img1:           first image
 *     img2:           second image
 *
 * OUTPUTS:
 *     out_data:       piv dataset containing particle image displacements
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char fname_parameter[GPIV_MAX_CHARS], fname_img[GPIV_MAX_CHARS];
    int print_par = 0;
    
    piv_eval_par->int_geo = GPIV_POINT;
    piv_eval_par->int_geo_logic = TRUE;
    piv_eval_par->int_point_col_logic = TRUE;
    piv_eval_par->int_point_col = out_data->point_x[index_y][index_x];
    piv_eval_par->int_point_row = out_data->point_y[index_y][index_x];
    piv_eval_par->int_point_row_logic = TRUE;
    piv_eval_par->peak_logic = TRUE;
    
    piv_eval_par->peak = in_data.peak_no[index_y][index_x] + 1;
    interrogate_point(index_y,
                      index_x,
                      img1, 
                      img2, 
                      image_par, 
                      *piv_eval_par, 
                      out_data);
}



static gboolean
subst_vector(GpivPivData in_data, 
             GpivPivData * out_data,
             GpivImagePar image_par, 
             GpivEvalPar piv_eval_par, 
             GpivValidPar piv_valid_par, 
             guint16 **img1, 
             guint16 **img2 
             )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Substitutes data with residu values higher than threshold
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     in_data:        piv dataset containing particle image displacements
 *     image_par:      image parameters
 *     piv_eval_par:   evaluation parameters
 *     piv_valid_par:  validation parameters
 *     img1:           first image
 *     img2:           second image
 *
 * OUTPUTS:
 *     out_data:       piv dataset containing particle image displacements
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    gboolean data_change = FALSE;
    int i, j, i_mx, j_mx, i_my, j_my, k, l;
    const gint N = piv_valid_par.neighbors - 2;
    char line[GPIV_MAX_CHARS], command[GPIV_MAX_CHARS];
    int count;
    int peak_no;
    float row = 0, col = 0;
    float dx, dy, snr, dx_sum, dy_sum;
    FILE *fp;

    for (i = 0; i < in_data.ny; i++) {
	 for (j = 0; j < in_data.nx; j++) {
             if (in_data.peak_no[i][j] != -1 
                 && out_data->snr[i][j] > piv_valid_par.residu_max) {
                     data_change = TRUE;
/*                  } */
/*
 * no substitution, only sets peak_no to 0
 */
                     if (piv_valid_par.subst_type == 
                       GPIV_VALID_SUBSTYPE__NONE) {
			out_data->dx[i][j] = in_data.dx[i][j];
			out_data->dy[i][j] = in_data.dy[i][j];
			out_data->snr[i][j] = in_data.snr[i][j];
                        if (out_data->snr[i][j] == SNR_ERR) {
                            out_data->peak_no[i][j] = -1;
                        } else {
                            out_data->peak_no[i][j] = 0;
                        }

                     } else if (piv_valid_par.subst_type == 
                                GPIV_VALID_SUBSTYPE__L_MEAN) {
                         count = 0;
                         dx_sum = 0;
                         dy_sum = 0;
                         for (k = i - N; k <= i + N; k++) {
			     if ((k >= 0) && (k < in_data.ny)) {
                                 for (l = j - N; l <= j + N; l++) {
                                     if ((l >= 0) && (l < in_data.nx)) {
/*
 * Exclude the point under investigation for calculating the mean
 */
					    if ((k != 0) || (l != 0)) {
						 dx_sum += in_data.dx[k][l];
						 dy_sum += in_data.dy[k][l];
						 count++;
					    }
				       }
				  }
			     }
			}
			out_data->dx[i][j] = dx_sum / count;
			out_data->dy[i][j] = dy_sum / count;
			out_data->snr[i][j] = in_data.snr[i][j];
                        out_data->peak_no[i][j] = 0;



/*
 * Substitution with  median particle displacements
 */
                   } else if (piv_valid_par.subst_type == 
                              GPIV_VALID_SUBSTYPE__MEDIAN) {
                       out_data->snr[i][j] = 
                           median_residu(i, j, piv_valid_par.neighbors,
                                         in_data, &i_mx, &j_mx, &i_my, &j_my, 
                                         FALSE, FALSE);
                       out_data->dx[i][j] = in_data.dx[i + i_mx][j + j_mx];
                       out_data->dy[i][j] = in_data.dy[i + i_my][j + j_my];
                       if (out_data->snr[i][j] == SNR_ERR) {
                           out_data->peak_no[i][j] = -1;
                       } else {
                           out_data->peak_no[i][j] = 0;
                       }

/*
 * substitution with image analyse of next higher correlation
 * peak (or higher)
 */
                     } else if (piv_valid_par.subst_type == 
                                GPIV_VALID_SUBSTYPE__COR_PEAK) {
/*                        nh_corrpeak(i, j, image_par, &piv_eval_par,  */
/*                                   in_data, out_data, img1, img2); */
                         interr_reg_nhpeak(i, j, img1, img2, image_par, 
                                           piv_eval_par, out_data);
                       
/*
 * Copy input data to output if disabled data or no outlier (snr <= snr_max)
 */
                   }
             } else {
                 out_data->dx[i][j] = in_data.dx[i][j];
                 out_data->dy[i][j] = in_data.dy[i][j];
                 out_data->peak_no[i][j] = in_data.peak_no[i][j];
/*                  if (set == 0) data_change = 0; */
             }


         }
    }

    return data_change;  
}


static void 
cumhisto_eqdatbin(GpivPivData * data,
                  GpivBinData * klass
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculating cumulative histogram from GpivPivData (NOT from GpivScalarData!) 
 *     with an equal number of date per bin of klass
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     data:           piv dataset
 *
 * OUTPUTS:
 *     klass:          histogram dataset
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    gint i, j, k, nresidus = 0;
    gint nx = data->nx, ny = data->ny, **peak_no = data->peak_no;
    gfloat **snr = data->snr;
    gfloat delta, *residu;
    gfloat *bound = klass->bound, *centre = klass->centre;
    gdouble fract, yval;
    gint *count = klass->count, nbins = klass->nbins;
    gint total_ndata = 0, nresidus_bin = 0;

    assert(data->point_x != NULL);
    assert(data->point_y != NULL);
    assert(data->dx != NULL);
    assert(data->dy != NULL);
    assert(data->snr != NULL);
    assert(data->peak_no != NULL);
    
    assert(klass->count != NULL);
    assert(klass->bound != NULL);
    assert(klass->centre != NULL);
  

   for (i = 0, k = 0; i < ny; i++) {
        for (j = 0; j < nx; j++) {
            if (peak_no[i][j] != -1 && snr[i][j] != 0) k++;
       }
    }
    nresidus = k;
    nresidus_bin = nresidus / nbins;

    residu = gpiv_vector(nresidus);
    for (i = 0, k = 0; i < ny; i++) {
        for (j = 0; j < nx; j++) {
            if (peak_no[i][j] != -1 && snr[i][j] != 0) {
                  residu[k] = snr[i][j];
                  k++;
            }
        }
    }


/*
 * sorting snr data
 */
    qsort(residu, nresidus, sizeof(float), compare_float);

/*
 * find lower boundaries of bins
 */

    for (i = 0; i < nbins; i++) {
        for (j = 0; j < nresidus_bin; j++) {
             if (j == 0) {
                klass->bound[i] = log (1.0/(1.0 - (double) i / (double) nbins));
/*                 klass->bound[i] = (double) i / (double) nbins; */
                klass->centre[i] = residu[(i+1) * nresidus_bin];
            }
        }
    }

    klass->min = bound[0];
    klass->max = bound[nbins - 1];
    gpiv_free_vector(residu);
}



void 
gpiv_valid_peaklck(GpivPivData in_data, 
                   GpivBinData * klass
                   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculating histogram of sub-pixel displacements to check on peaklocking
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i, j, k;
    float delta, fract;
    float *bound = klass->bound, *centre = klass->centre;
    int *count = klass->count, nbins = klass->nbins;

    assert(in_data.point_x != NULL);
    assert(in_data.point_y != NULL);
    assert(in_data.dx != NULL);
    assert(in_data.dy != NULL);
    assert(in_data.snr != NULL);
    assert(in_data.peak_no != NULL);

    assert(klass->count != NULL);
    assert(klass->bound != NULL);
    assert(klass->centre != NULL);

    delta = 1. / nbins;
    for (i = 0; i < nbins; i++) {
	centre[i] = (float) i *delta;
	bound[i] = -delta / 2.0 + (float) i *delta;
	count[i] = 0;
    }

/*
 * Subdividing fractional particle displacements in bins
 */
    for (i = 0; i < in_data.ny; i++) {
	for (j = 0; j < in_data.nx; j++) {
	    fract = fabs(in_data.dx[i][j] - (int) in_data.dx[i][j]);
	    for (k = 0; k < nbins; k++) {
		if ((fract >= bound[k]) && (fract < bound[k + 1])) {
		    count[k] = count[k] + 1;
		}
	    }
	    fract = fabs(in_data.dy[i][j] - (int) in_data.dy[i][j]);
	    for (k = 0; k < nbins; k++) {
		if ((fract >= bound[k]) && (fract < bound[k + 1])) {
		    count[k] = count[k] + 1;
		}
	    }
	}
    }


}



char *
gpiv_valid_residu_stats(GpivLinRegData * linreg, 
                        GpivPivData * out_data, 
                        GpivBinData * klass
                        )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculates cumulative histogram of residus
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     linreg:         linear regression data structure
 *     klass:          histogram data
 *
 * OUTPUTS:
 *     out_data:       piv dataset containing residu values in snr
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, return_val;
    double * x, * y;
            
    cumhisto_eqdatbin(out_data, klass);
    x = gpiv_dvector(klass->nbins);
    y = gpiv_dvector(klass->nbins);

    for (i = 0; i < klass->nbins; i++) {
        x[i] = (double) klass->bound[i];
        y[i] = (double) klass->centre[i];
    }

    if (return_val = 
        gsl_fit_linear (x, 1, y, 1, klass->nbins, &linreg->c0, 
                        &linreg->c1, &linreg->cov00, &linreg->cov01, 
                        &linreg->cov11, &linreg->sumsq) == 1) {
        err_msg = "GPIV_VALID_RESIDU_STATS: error from gsl_fit_linear";
        g_warning("%s", err_msg);
        return err_msg;
    }
    gpiv_free_dvector(x);
    gpiv_free_dvector(y);
   
    return err_msg;
}   



char *
gpiv_valid_errvec(GpivImagePar image_par,
                  GpivEvalPar piv_eval_par,
                  GpivValidPar piv_valid_par, 
                  GpivPivData in_data, 
                  GpivPivData * out_data, 
                  GpivBinData * klass,
                  GpivLinRegData * linreg,
                  guint16 **img1,
                  guint16 **img2,
                  gboolean interrogate_valid
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Searches the erroneous vectors in a PIV data set and
 *     substitutes with new values, if possible
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     fname:          base name of image and piv dataset
 *     image_par:      image parameters
 *     piv_eval_par:   evaluation parameters
 *     piv_valid_par:  validation parameters
 *     in_data:        piv dataset containing particle image displacements
 *     img1:           first image
 *     img2:           second image
 *     interrogate_valid: validation during (iterative) interrogation process
 *
 * OUTPUTS:
 *     out_data:       piv dataset containing particle image displacements
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    gchar *err_msg = NULL;
    gint i, j;
    gboolean data_change = TRUE;
    gint count = 0;
    enum SubstitutionType l_subst_type = piv_valid_par.subst_type;
    gint l_peak = piv_eval_par.peak;

    assert(in_data.point_x != NULL);
    assert(in_data.point_y != NULL);
    assert(in_data.dx != NULL);
    assert(in_data.dy != NULL);
    assert(in_data.snr != NULL);
    assert(in_data.peak_no != NULL);

    assert(out_data->snr != NULL);

    if (piv_valid_par.res_stats == 1) {
        assert(klass->count != NULL);
        assert(klass->bound != NULL);
        assert(klass->centre != NULL);

    } else {
        assert(out_data->point_x != NULL);
        assert(out_data->point_y != NULL);
        assert(out_data->dx != NULL);
        assert(out_data->dy != NULL);
        assert(out_data->peak_no != NULL);
        
        for (i = 0; i < in_data.ny; i++) {
            for (j = 0; j < in_data.nx; j++) {
                out_data->point_x[i][j] = in_data.point_x[i][j];
                out_data->point_y[i][j] = in_data.point_y[i][j];
            }
        }
    }

    while (data_change && count < GPIV_VALID_MAX_SWEEP) {
        gpiv_valid_residu(in_data, out_data, piv_valid_par, TRUE);
/*
 * Sorting residus in bins
 */
        if (piv_valid_par.res_stats == TRUE) {
            data_change = FALSE;
            if ((err_msg = gpiv_valid_residu_stats(linreg, out_data, klass)) 
                != NULL) {
                err_msg = "GPIV_VALID_ERRVEC: error from residu_stats";
                g_warning("%s", err_msg);
                return err_msg;
            }
            
        } else {
/*
 * Calculates and substitutes snr data with residu values higher than threshold
 */
            if (piv_valid_par.subst_type != GPIV_VALID_SUBSTYPE__NONE
                && interrogate_valid) {
/*
 * Test data with different types of substitutions if used during
 * (iterative) image interrogation
 */
                if (count <= 1) {
                    piv_valid_par.subst_type = GPIV_VALID_SUBSTYPE__COR_PEAK;
                    piv_eval_par.peak = count + 2;
                } else  {
                    piv_valid_par.subst_type = l_subst_type;
                    piv_eval_par.peak = l_peak;
                }
            }

            data_change = subst_vector(in_data, out_data, image_par, 
                                       piv_eval_par, piv_valid_par, 
                                       img1, img2);

            if (piv_valid_par.subst_type == GPIV_VALID_SUBSTYPE__NONE) {
                data_change = FALSE;
            }
            
            if (data_change) {
                count++;
                gpiv_cp_pivdata(out_data, &in_data);
            }
        }
    }


    return err_msg;
}



void
gpiv_valid_gradient(GpivValidPar piv_valid_par, 
                    GpivEvalPar piv_eval_par, 
                    GpivPivData * piv_data
                    )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Searches vectors in a PIV data set that exceed the maximum gradient 
 *     (dU x dt/int_size > 0.05)
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     piv_eval_par:   evaluation parameters
 *     piv_valid_par:  validation parameters
 *
 * OUTPUTS:
 *     piv_data:       piv dataset containing peak_no = -1 for exceeded maxima
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i, j, diff_order = 1;
    double grad_dx, delta_dx, grad_dy, delta_dy;

    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);
    assert(piv_data->dx != NULL);
    assert(piv_data->dy != NULL);
    assert(piv_data->snr != NULL);
    assert(piv_data->peak_no != NULL);

/* BUGFIX: test op patch. Gerber */
    for (i = diff_order; i < piv_data->ny - diff_order; i++) {
	for (j = diff_order; j < piv_data->nx - diff_order; j++) {

            if(piv_data->peak_no[i][j] != -1 
/*                && piv_data->peak_no[i-1][j] != -1 && */
/*                piv_data->peak_no[i][j-1] != -1 && */
/*                piv_data->peak_no[i][j+1] != -1 */
               ) {
                grad_dx = (piv_data->dx[i+1][j] - piv_data->dx[i-1][j]) / 
                    (2 * piv_eval_par.int_shift);
                delta_dx = fabs(grad_dx) * piv_eval_par.int_size_1;
                
                piv_data->snr[i][j] = delta_dx;
                grad_dy = (piv_data->dy[i][j+1] - piv_data->dy[i][j-1]) / 
                    (2 * piv_eval_par.int_shift); 
                delta_dy = fabs(grad_dy) * piv_eval_par.int_size_1;
                if (delta_dx > GPIV_GRADIENT_THRESHOLD || 
                    delta_dy > GPIV_GRADIENT_THRESHOLD) 
                    piv_data->peak_no[i][j] = -1;
            }
        }
    }



/*
 * exclude all data near the boundaries of the dataset
 */
     for (i=0; i < diff_order; i++) {
	  for (j=0; j < piv_data->nx; j++) {
              piv_data->peak_no[i][j] = 0;
	  }
     }

     for (i=0; i < piv_data->ny; i++) {
	  for (j=0; j < diff_order; j++) {
              piv_data->peak_no[i][j] = 0;
	  }
     }

     for (i=piv_data->ny - diff_order; i < piv_data->ny; i++) {
	  for (j=0; j < piv_data->nx; j++) {
              piv_data->peak_no[i][j] = 0;
	  }
     }

     for (i=0; i < piv_data->ny; i++) {
	  for (j=piv_data->nx - diff_order; j < piv_data->nx; j++) {
              piv_data->peak_no[i][j] = 0;
	  }
     }


}





void
gpiv_cumhisto_eqdatbin_gnuplot(char *fname_out, 
                               char *title, 
                               const char *GNUPLOT_DISPLAY_COLOR,
                               const int GNUPLOT_DISPLAY_SIZE,
                               GpivLinRegData * linreg
                               )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Plots data on screen with gnuplot
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/

{
  FILE *fp_cmd;
  char *fname_cmd="/tmp/gpiv_gnuplot.cmd";
  char *function_name="gpiv_histo_gnuplot";
  char command[GPIV_MAX_CHARS];

  if((fp_cmd=fopen(fname_cmd,"w"))==NULL) { 
    fprintf (stderr,"\n%s:%s error: Failure opening %s for output\n",
	     LIBNAME, function_name, fname_cmd); 
    exit(1);
  }

  fprintf (fp_cmd,"\nset xlabel \"-ln(1-i/nbins)\"");
  fprintf (fp_cmd,"\nset ylabel \"residu (pixels)\"");
  fprintf (fp_cmd,"\nplot \"%s\" title \"%s\" with boxes, %f + %f * x", /* with boxes */
	   fname_out, title, linreg->c0, linreg->c1);
  fprintf (fp_cmd,"\npause -1 \"Hit return to exit\"");
  fprintf (fp_cmd,"\nquit");

  fclose (fp_cmd);
  
  
  snprintf(command, GPIV_MAX_CHARS, "gnuplot -bg %s -geometry %dx%d %s",
	   GNUPLOT_DISPLAY_COLOR, GNUPLOT_DISPLAY_SIZE, 
	   GNUPLOT_DISPLAY_SIZE, fname_cmd);

  if (system (command) != 0) {
    fprintf (stderr,"\n%s:%s could not exec shell command\n", 
	     LIBNAME, function_name);
    exit(1);
  }


}


float 
gpiv_valid_threshold(GpivEvalPar piv_eval_par,
                     GpivValidPar piv_valid_par,
                     GpivLinRegData linreg
                     )

/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculates threshold value (residu_max) from residus. 
 *     Will need the int_size_1 from the GpivEvalPar struct!
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    float search_radius = (float) piv_eval_par.int_size_1 / 4.0;
    float residu_max = - linreg.c1 * log((1.0 - piv_valid_par.data_yield) / 
                                        piv_valid_par.data_yield * 
                                         linreg.c1 / search_radius);
    return residu_max;
}

#undef SNR_ERR
#undef MIN_VECTORS4MEDIAN
#undef RESIDU_EPSI
