/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.
   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



-------------------------------------------------------------------------------
FILENAME:                imgproc_par.c
LIBRARY:                 libgpiv:
EXTERNAL FUNCTIONS:
                        gpiv_imgproc_read_parameters
                        gpiv_imgproc_check_parameters_read
                        gpiv_imgproc_test_parameters
                        gpiv_imgproc_print_parameters
                        gpiv_imgproc_fprint_parameters


LAST MODIFICATION DATE:  $Id: imgproc_par.c,v 1.2 2006/01/31 13:30:13 gerber Exp $
 --------------------------------------------------------------------------- */

#include <gpiv.h>



void
gpiv_imgproc_read_parameters(FILE *fp_par, 
			     GpivImageProcPar * image_proc_par, 
			     int print_par
			     )
/*----------------------------------------------------------------------------
  DESCRIPTION:
      Reads each line of the file and looks for image parameters parameters

  PROTOTYPE LOCATATION:
      gpiv_io.h

  INPUTS:
      fp_par:           file pointer to input file
      print_par         flag to print parameter to stdout

  OUTPUTS:
---------------------------------------------------------------------------- */
{
  char line[GPIV_MAX_CHARS], par_name[GPIV_MAX_CHARS];

    while (fgets(line, GPIV_MAX_CHARS, fp_par) != NULL) {
	if (line[0] != '#' && line[0] != '\n' && line[0] != ' '
	    && line[0] != '\t') {
	    sscanf(line, "%s", par_name);

	    if (image_proc_par->first_dir_logic == FALSE) {
		image_proc_par->first_dir_logic =
		    gpiv_scan_iph(GPIV_IMAGEPROC_PAR_KEY, line, par_name, 
                                  "FirstDir:",
				  &image_proc_par->first_dir, print_par, 0);
	    }

	    if (image_proc_par->last_dir_logic == FALSE) {
		image_proc_par->last_dir_logic =
		    gpiv_scan_iph(GPIV_IMAGEPROC_PAR_KEY, line, par_name, 
                                  "LastDir:",
				  &image_proc_par->last_dir, print_par, 0);
	    }

	    if (image_proc_par->dir_prefix_logic == FALSE) {
		image_proc_par->dir_prefix_logic =
		    gpiv_scan_iph(GPIV_IMAGEPROC_PAR_KEY, line, par_name, 
                                  "DirPrefix:",
				  &image_proc_par->dir_prefix, print_par, 0);
	    }


	    if (image_proc_par->first_file_logic == FALSE) {
		image_proc_par->first_file_logic =
		    gpiv_scan_iph(GPIV_IMAGEPROC_PAR_KEY, line, par_name, 
                                  "FirstFile:",
				  &image_proc_par->first_file, print_par, 0);
	    }

	    if (image_proc_par->last_file_logic == FALSE) {
		image_proc_par->last_file_logic =
		    gpiv_scan_iph(GPIV_IMAGEPROC_PAR_KEY, line, par_name, 
                                  "LastFile:",
				  &image_proc_par->last_file, print_par, 0);
	    }

	    if (image_proc_par->file_prefix_logic == FALSE) {
		image_proc_par->file_prefix_logic =
		    gpiv_scan_iph(GPIV_IMAGEPROC_PAR_KEY, line, par_name, 
                                  "FilePrefix:",
				  &image_proc_par->file_prefix, print_par, 0);
	    }

	    if (image_proc_par->operator_logic == FALSE) {
		image_proc_par->operator_logic =
		    gpiv_scan_iph(GPIV_IMAGEPROC_PAR_KEY, line, par_name, 
                                  "Operator:",
				  &image_proc_par->operator, print_par, 0);
	    }

	    if (image_proc_par->window_logic == FALSE) {
		image_proc_par->window_logic =
		    gpiv_scan_iph(GPIV_IMAGEPROC_PAR_KEY, line, par_name, 
                                  "Window:",
				  &image_proc_par->window, print_par, 0);
	    }

	    if (image_proc_par->threshold_logic == FALSE) {
		image_proc_par->threshold_logic =
		    gpiv_scan_iph(GPIV_IMAGEPROC_PAR_KEY, line, par_name, 
                                  "Threshold:",
				  &image_proc_par->threshold, print_par, 0);
	    }

	    if (image_proc_par->bit_logic == FALSE) {
		image_proc_par->bit_logic =
		    gpiv_scan_iph(GPIV_IMAGEPROC_PAR_KEY, line, par_name, 
                                  "Bit:",
				  &image_proc_par->bit, print_par, 0);
	    }



	}
    }


}



char *
gpiv_imgproc_check_parameters_read(GpivImageProcPar image_proc_par
				   )
/*----------------------------------------------------------------------------
  DESCRIPTION:
      Check out if all image processing parameters have been read

  INPUTS:
      image_proc_par:   structure to image processing parameters

  RETURNS:
      NULL on success or *err_msg on failure
---------------------------------------------------------------------------- */
{
    char *err_msg = NULL;

    if (image_proc_par.first_dir_logic == FALSE) {
	err_msg = "GPIV_IMGPROC_CHECK_PARAMETERS_READ: FirstDir not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (image_proc_par.first_dir_logic == FALSE) {
	err_msg = "GPIV_IMGPROC_CHECK_PARAMETERS_READ: LastDir not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (image_proc_par.first_dir_logic == FALSE) {
	err_msg = "GPIV_IMGPROC_CHECK_PARAMETERS_READ: FirstFile not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (image_proc_par.first_dir_logic == FALSE) {
	err_msg = "GPIV_IMGPROC_CHECK_PARAMETERS_READ: LastFile not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (image_proc_par.operator_logic == FALSE) {
	err_msg = "GPIV_IMGPROC_CHECK_PARAMETERS_READ: Operator not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (image_proc_par.operator_logic == FALSE) {
	err_msg = "GPIV_IMGPROC_CHECK_PARAMETERS_READ: Operator not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (image_proc_par.window_logic == FALSE) {
	err_msg = "GPIV_IMGPROC_CHECK_PARAMETERS_READ: Window not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (image_proc_par.threshold_logic == FALSE) {
	err_msg = "GPIV_IMGPROC_CHECK_PARAMETERS_READ: Threshold not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (image_proc_par.bit_logic == FALSE) {
	err_msg = "GPIV_IMGPROC_CHECK_PARAMETERS_READ: Bit not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }


    return err_msg;
}



char *
gpiv_imgproc_test_parameters(GpivImageProcPar image_proc_par
			     )
/* ----------------------------------------------------------------------------
   Test parameters information on validity */
{
    char *err_msg = NULL;

    if (image_proc_par.window > GPIV_MAX_IMG_SIZE) {
        err_msg = "GPIV_IMGPROC_TEST_PARAMETERS: window larger than GPIV_MAX_IMG_SIZE";
         gpiv_warning("%s", err_msg);
         return err_msg;
     }

    if (image_proc_par.threshold < 0 
        || image_proc_par.threshold > pow((double)2, GPIV_MAX_IMG_DEPTH)) {
        err_msg = "GPIV_IMGPROC_TEST_PARAMETERS: threshold larger than GPIV_MAX_IMG_DEPTH";
         gpiv_warning("%s", err_msg);
         return err_msg;
     }

    if (image_proc_par.bit > GPIV_MAX_IMG_DEPTH) {
        err_msg = "GPIV_IMGPROC_TEST_PARAMETERS: bit larger than GPIV_MAX_IMG_DEPTH";
         gpiv_warning("%s", err_msg);
         return err_msg;
     }

    if (image_proc_par.window > GPIV_MAX_IMG_SIZE) {
        err_msg = "GPIV_IMGPROC_TEST_PARAMETERS: window larger than GPIV_MAX_IMG_SIZE";
         gpiv_warning("%s", err_msg);
         return err_msg;
     }
}



void 
gpiv_imgproc_print_parameters(GpivImageProcPar image_proc_par
			      )
/* ----------------------------------------------------------------------------
 * prints image parameters parameters to stdout
 */
{
    if (image_proc_par.first_dir_logic)
	printf("%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "FirstDir:", 
               image_proc_par.first_dir);

    if (image_proc_par.last_dir_logic)
	printf("%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "LastDir:", 
               image_proc_par.last_dir);

    if (image_proc_par.first_file_logic)
	printf("%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "FirstFile:", 
               image_proc_par.first_file);

    if (image_proc_par.last_file_logic)
	printf("%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "LastFile:", 
               image_proc_par.last_file);

    if (image_proc_par.filter_logic)
	printf("%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "Filter:",
               image_proc_par.filter);
 
   if (image_proc_par.operator_logic)
	printf("%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "Operator:",
               image_proc_par.operator);

   if (image_proc_par.window_logic)
	printf("%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "Window:",
               image_proc_par.window);

   if (image_proc_par.threshold_logic)
	printf("%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "Threshold:",
               image_proc_par.threshold);

   if (image_proc_par.bit_logic)
	printf("%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "Bit:",
               image_proc_par.bit);
}



void
gpiv_imgproc_fprint_parameters(FILE * fp_par_out, 
			       GpivImageProcPar image_proc_par
			       )
/*  ---------------------------------------------------------------------------
 * prints image parameters parameters to filename.par
 */
{

    if (image_proc_par.first_dir_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "FirstDir:", 
                image_proc_par.first_dir);

    if (image_proc_par.last_dir_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "LastDir:", 
                image_proc_par.last_dir);

    if (image_proc_par.first_file_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "FirstFile:", 
                image_proc_par.first_file);

    if (image_proc_par.last_file_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "LastFile:", 
                image_proc_par.last_file);

    if (image_proc_par.filter_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "Filter", 
                image_proc_par.filter);

    if (image_proc_par.operator_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "Operator:", 
                image_proc_par.operator);

   if (image_proc_par.window_logic)
       fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "Window:",
               image_proc_par.window);

   if (image_proc_par.threshold_logic)
       fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "Threshold:",
               image_proc_par.threshold);

   if (image_proc_par.bit_logic)
       fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGEPROC_PAR_KEY, "Bit:",
               image_proc_par.bit);
}
