/* ex: set tabstop=4 noet: */
/* all network-based structures and definitions */

#ifndef PROTOCOLS_H
#define PROTOCOLS_H

#ifdef WIN32
	#include "types.h"
	#include <sys/types.h> /* time_t */
#else
	#include <stdint.h> /* uint16_t and friends */
	#include <sys/types.h> /* u_char and friends */
#endif
#include <stdlib.h> /* NULL */
#include "types.h"

enum {
	CONN_UNKNOWN = 0,
	CONN_ETHERNET,
	CONN_MODEM_DIALUP,
	CONN_MODEM_RADIO,
	CONN_MODEM_CABLE,
	CONN_DSL,
#if 0 /* let's start simple */
	CONN_PIX,
	CONN_SMC,
	CONN_WIFI,
	CONN_T1,
	CONN_TUNNEL_IPIP,
	CONN_VTUN,
	CONN_ISDN_PPP,
	CONN_GRE,
	CONN_PPPOE_DSL,
	CONN_VLAN,
	CONN_IRDA,
	CONN_LOOPBACK,
	CONN_TOKEN_RING,
#endif
	CONN_COUNT
};

/**
 * all supported protocol ids (internal)
 */
/*
seen:
	udp port 42: Microsoft WINS
*/
enum {
	PROT_UNKNOWN = 0,
	PROT_LOGIC, /* Logical frame */
	PROT_EXTRA_JUNK, /* unecessary trailing garbage */
	PROT_LINUX_SLL,
	PROT_ETHER2, /* Ethernet 2 */
	PROT_IEEE_802_3, /* IEEE 802.3 Ethernet */
	PROT_ARP, /* Address Resolution Protocol */
	PROT_LLC, /* Link Layer Control */
	PROT_CDP, /* Cisco Discovery Protocol */
	PROT_STP, /* Spanning Tree Protocol */
	PROT_AARP, /* Apple Address Resolution Protocol */
	PROT_IP,
	PROT_IPV6,
	PROT_ICMP,
	PROT_ICMPV6,
	PROT_UDP,
	PROT_TCP,
	PROT_IGMP,
	PROT_IPX,
	PROT_BOOTP,
	PROT_DNS,
	PROT_NB_DGM,
	PROT_NB_NS,
	PROT_NB_SSN,
	PROT_SMB,
	PROT_DDP,
	PROT_SAP, /* Service Advertisement Protocol */
	PROT_HTTP,
	PROT_HTTPS,
	PROT_IAPP, /* Inter-Access Point Protocol */
	PROT_LOOP, /* Configuration Test Protocol (loopback) */
	PROT_GRE, /* Generic Route Protocol */
	PROT_SMTP, /* Simple Mail Transport Protocol */
	PROT_AIM,
	PROT_IRC,
	PROT_JABBER,
	PROT_GNUTELLA,
	PROT_RTSP,
	PROT_PPP,
	PROT_SNMP,
	PROT_TEREDO,
	PROT_CSTRIKE,
	PROT_SSDP,
	PROT_SYMBOL_WIFI,
	PROT_CISCOWL, /* Cisco Wireless 2 */
	PROT_FTP,
	PROT_SSH,
	PROT_TELNET,
	PROT_TFTP,
	PROT_COUNT /* total number of supported protocols */
};


#define PROT_DESCR_BUFLEN				32 /* human-readable protocol name */

/* * * * MTUs * * * * */
enum {
	MTU_ETHER = 1500
};


/* * * * * Ethernet * * * * */
#define MAC_ADDR_BYTES					6
#define MAC_ADDR_BUFLEN					(6 * 3) /* human-readable */

#define ETHER_TYPE_BYTES				2
#define ETHER_HEAD_BYTES				(MAC_ADDR_BYTES + MAC_ADDR_BYTES + ETHER_TYPE_BYTES)
#define ETHER_LEN_MIN					(ETHER_HEAD_BYTES)
#define ETHER_LEN_PADTO					60 /* minimum total size, must be padded to */
#define ETHER_FRAME_MAX_BYTES			(MTU_ETHER + ETHER_HEAD_BYTES)


/* protocol types under ethernet */
#define ETHER2_TYPE_UNKNOWN				0x0000
#define ETHER2_TYPE_802_2_LLC			0x0004 /* used via LLC... just a guess */
#define ETHER2_TYPE_IP					0x0800
#define ETHER2_TYPE_ARP					0x0806
#define ETHER2_TYPE_LOOP				0x9000
#define ETHER2_TYPE_					0x0000

#define LINUX_SLL_LEN_MIN				16
enum {
	SLL_TYPE_WHOOPS_TOO_LOW = -1,
	SLL_TYPE_UCAST_2_US = 0,
	SLL_TYPE_BCAST_2_US,
	SLL_TYPE_MCAST,
	SLL_TYPE_SENT_BY_OTHERS,
	SLL_TYPE_SENT_BY_US,
	SLL_TYPE_WHOOPS_TOO_HIGH
};

#define SLL_ETHERNET_TYPE_802_2_LLC		0x0004

/**
 * IP versions 
 */
enum {
	IPV4,
	IPV6
};

/* ARP - Address Resolution Protocol */
#define ARP_LEN_MIN						28
#define ARP_LEN_MAX						28

#define ARP_OPCODE_REQ					0x0001
#define ARP_OPCODE_RESP					0x0002


/* LLC - Link Layer Control */
#define LLC_LEN_MIN						8
#define LLC_LEN_MAX						8

#define LLC_DSAP_SNAP					0xAA
#define LLC_DSAP_ST_BDPU				0x42
#define LLC_DSAP_NETWARE				0xE0

#define LLC_PID_CISCOWL					0x0000
#define LLC_PID_CDP						0x2000
#define LLC_PID_APPLETALK				0x809B
#define LLC_PID_AARP					0x80F3

enum {
	LLC_CTRL_XID = 0x2B
};

/**
 * entry for an LLC organization code
 */
struct llc_org {
	uint32_t id;
	const char *name;
};

/* STP - Spanning Tree Protocol */
#define STP_LEN_MIN						35
#define STP_LEN_MAX						35


/* CDP - Cisco Discovery Protocol */
#define CDP_LEN_MIN						1
#define CDP_LEN_MAX						1

/* data type */
#define CDP_TYPE_DEV					0x0001
	#define CDP_DEV_DESCR_BUFLEN		16
#define CDP_TYPE_ADDR					0x0002
#define CDP_ADDR_MAX					8
	/* protocol type */
	#define CDP_ADDR_PROT_TYPE_NLPID	1
	#define CDP_ADDR_PROT_TYPE_802_2	2
	/* protocol */
	/* NOTE: check lLAST N bytes of version */
	#define CDP_ADDR_PROT_CLNS			0x81
	#define CDP_ADDR_PROT_IP			0xCC
	#define CDP_ADDR_PROT_PV6			0x0800
	#define CDP_ADDR_PROT_DECNET_PHASE4	0x6003
	#define CDP_ADDR_PROT_APPLETALK		0x809B
	#define CDP_ADDR_PROT_IPX			0x8137
	#define CDP_ADDR_PROT_VINES			0x80C4
	#define CDP_ADDR_PROT_XNS			0x0600
	#define CDP_ADDR_PROT_APOLLO_DOMAIN	0x8019
	#define CDP_ADDR_BUFLEN				16
#define CDP_TYPE_PORT					0x0003
	/* ascii description of interfacem i.e. "FastEthernet0/14" */
	#define CDP_PORT_BUFLEN				24
#define CDP_TYPE_CAPAB					0x0004
	#define CDP_CAPAB_ROUTER			0x01
	#define CDP_CAPAB_TRANS_BRIDGE		0x02
	#define CDP_CAPAB_SRCRT_BRIDGE		0x04
	#define CDP_CAPAB_SWITCH			0x08
	#define CDP_CAPAB_HOST				0x10
	#define CDP_CAPAB_IGMP				0x20
	#define CDP_CAPAB_REPEATER			0x40
	#define CDP_CAPAB_UNUSED			0xFFFFFF80
#define CDP_TYPE_VERSION				0x0005
	/* ascii representation of software, long! */
	#define CDP_VERSION_BUFLEN			256
#define CDP_TYPE_PLATFORM				0x0006
	#define CDP_PLATFORM_BUFLEN			32
#define CDP_TYPE_IPPREFIX				0x0007


/* AARP - Apple Address Resolution Protocol */
#define AARP_LEN_MIN					28
#define AARP_LEN_MAX					28

#define AARP_OPCODE_REQ					0x0001
#define AARP_OPCODE_RESP				0x0002

/* CiscoWL */
#define CISCOWL_LEN_MIN					32


/* * * *  LOOPBACK * * * */
#define LOOP_LEN_MIN					6

/* * * *  IPX - Internetwork Packet eXchange * * * */
#define IPX_LEN_MIN						30 /* FIXME: check */
#define IPX_LEN_MAX						30 /* FIXME: check */

#define IPX_TYPE_IPX					0x00
#define IPX_TYPE_NETBIOS_BCAST			0x14

/* IPX socket types */
#define IPX_SOCK_SAP					0x0452
#define IPX_SOCK_NETBIOS				0x0455
#define IPX_SOCK_NMP					0x0553 /* Name Management Protocol over IPX */

/* * * * SAP - Service Advertisement Protocol * * * */
/* SAP advertises services over IPX */
#define SAP_LEN_MIN						4 /* FIXME: check */

#define SAP_SERVER_NAME_MAXLEN			48
#define SAP_SERVER_NAME_BUFLEN			SAP_SERVER_NAME_MAXLEN + 1

#define SAP_SERVER_TYPE_PRINTER			0x030C /* FIXME: Intel Netport 2 or HP JetDirect or HP Quicksilver */
#define SAP_SERVER_TYPE_FILESERVER		0x0004

/*  */
#define SAP_SERVER_SOCKET_HP_LJ_OR_QS	0x400C

/* * * * * DDP - Data Delivery Protocol * * * * * */

#define DDP_LEN_MIN						13 /* FIXME: verify */


/* * * * IP - Internet Protocol * * * */

/* NOTE: for both IP and IPv6... confusing? */
#define IP_ADDR_BUFLEN					52 /* a human-readable ip address, IPv[46] */

/* RFC 791 */
/* lengths of an entire IP */
#define IP_LEN_MIN						20
#define IP_LEN_MAX						60 /* TODO: double-check */

#define IP_OPT_MAX_LEN					40 /*  */

/* RFC 790 (???) */
/* RFC 1700 */
enum {
	IP_PROT_0			= 0x00,
	IP_PROT_ICMP		= 0x01,
	IP_PROT_IGMP		= 0x02,
	IP_PROT_GGP			= 0x03,
	IP_PROT_IP			= 0x04,
	IP_PROT_ST			= 0x05,
	IP_PROT_TCP			= 0x06,
	IP_PROT_UDP			= 0x11,
	IP_PROT_IPV6		= 0x29,
	IP_PROT_GRE			= 0x2F
};

/* * * IPv6 - Internet Protocol v6 * * */
/* RFC 2460 */
/* lengths of an entire IP */
#define IPV6_LEN_MIN					40
#define IPV6_LEN_MAX					40 /* FIXME */

#define IPV6_NEXT_ICMPV6				0x3A
#define IPV6_NEXT_NONE					0x3B /* no next header */


/* * * GRE - Generic Route Protocol * * */
#define GRE_LEN_MIN						12 /* FIXME: check! */
#define GRE_LEN_MAX						12 /* FIXME: check! */

enum {
	GRE_PROT_PPP		= 0x880B
};

/* * * IGMP * * */
#define IGMP_LEN_MIN					8
#define IGMP_LEN_MAX					IGMP_LEN_MIN /* FIXME! */

enum {
	IGMP_TYPE_UNKNOWN = 0,
	IGMP_TYPE_MEMB_REPORT = 16,
	IGMP_TYPE_LEAVE_GROUP = 17
};


/* * * UDP - User Datagram Protocol * * */
#define UDP_LEN_MIN						8
#define UDP_LEN_MAX						8


/* Teredo Tunneling */
/* see http://www.ietf.org/internet-drafts/draft-huitema-v6ops-teredo-05.txt */

#define TEREDO_LEN_MIN					13 /*  */


/* * * TCP - Transfer Control Protocol * * */
#define TCP_LEN_MIN						20
#define TCP_LEN_MAX						60

#define TCP_MSS_TO_MTU(mss)				((mss + IP_LEN_MIN + TCP_LEN_MIN))

/* SYN fingerprint stuff... bad place for it :/ */
#define TCP_MULT_MSS	(1 << 16)
#define TCP_MULT_MTU	(1 << 17)
#define TCP_MULT_MOD	(1 << 18)

#define TCP_OPTS_MAX					20 /* 20 bytes max / smallest opt 1 byte */

/**
 * http://www.iana.org/assignments/tcp-parameters
 */
enum {
	TCP_OPT_END = 0,
	TCP_OPT_NOP = 1,
	TCP_OPT_MSS = 2,
	TCP_OPT_WS = 3,
	TCP_OPT_SACK = 4,
	TCP_OPT_SACKP = 5, /* wtf is this? RTFM RFC 2018 */
	TCP_OPT_ECHO_REQ = 6,
	TCP_OPT_ECHO_RESP = 7,
	TCP_OPT_TS = 8,
	TCP_OPT_POC = 9,
	TCP_OPT_POS = 10,
	TCP_OPT_CC = 11,
	TCP_OPT_CC_NEW = 12,
	TCP_OPT_CC_ECHO = 13,
	TCP_OPT_CHK_REQ = 14,
	TCP_OPT_CHK_DATA = 15,
	TCP_OPT_SKEETER = 16,
	TCP_OPT_BUBBA = 17,
	TCP_OPT_TRAILER = 18,
	TCP_OPT_MD5 = 19,
	TCP_OPT_SCPS = 20,
	TCP_OPT_SNA = 21,
	TCP_OPT_RB = 22,
	TCP_OPT_CORR = 23,
	TCP_OPT_SNAP = 24,
	TCP_OPT_UNASSIGNED = 25,
	TCP_OPT_COMPFILT = 26,
	TCP_OPT_WHOOPS_TOO_HIGH /* too high */
};

/* * * HTTP - HyperText Transfer Protocol * * */
#define HTTP_LEN_MIN					1 /* FIXME: arbitrary */

/* * * * * * * Ports * * * * * * * */
/* TCP and UDP, both under IP, have ports! */

/* TODO: move this out externally */
enum {
	PORT_ECHO					= 7 /*  */
	,PORT_DISCARD				= 9 /*  */
	,PORT_SYSTAT				= 11 /*  */
	,PORT_DAYTIME				= 13 /*  */
	,PORT_NETSTAT				= 15 /*  */
	,PORT_QOTD					= 17 /*  */
	,PORT_CHARGEN				= 19 /*  */
	,PORT_FTPDATA				= 20 /*  */
	,PORT_FTP					= 21 /*  */
	,PORT_SSH					= 22 /*  */
	,PORT_TELNET				= 23 /*  */
	,PORT_SMTP					= 25 /*  */
	,PORT_TIME					= 37 /*  */
	,PORT_RLP					= 39 /*  */
	,PORT_WHOIS					= 43 /*  */
	,PORT_DNS					= 53 /*  */
	,PORT_SQLNET				= 66 /*  */
	,PORT_BOOTPS				= 67 /* BOOTP Server */
	,PORT_BOOTPC				= 68 /* BOOTP Client */
	,PORT_TFTP					= 69 /*  */
	,PORT_GOPHER				= 70 /*  */
	,PORT_FINGER				= 79 /*  */
	,PORT_HTTP					= 80 /*  */
	,PORT_HTTP81				= 81 /*  */
	,PORT_KERBEROS				= 88 /*  */
	,PORT_POP2					= 109 /*  */
	,PORT_POP3					= 110 /*  */
	,PORT_SUNRPC				= 111 /*  */
	,PORT_IDENT					= 113 /*  */
	,PORT_NNTP					= 119 /* Usenet! */
	,PORT_NTP					= 123 /* Network Time Protocol */
	,PORT_NB_NS					= 137 /* NetBIOS Nameservice */
	,PORT_NB_DGM				= 138 /* NetBIOS Datagram */
	,PORT_NB_SSN				= 139 /* NetBIOS Session Service */	
	,PORT_IMAP					= 143 /* */
	,PORT_SQLNET150				= 150 /*  */
	,PORT_SQLSERV				= 156 /*  */
	,PORT_SNMP					= 161 /*  */
	,PORT_SNMPTRAP				= 162 /*  */
/* AppleTalk should go here... */
	,PORT_IMAP3					= 220 /*  */
	,PORT_HTTPS					= 443
	,PORT_RTSP					= 554 /* Real-time Streaming Protocol */
	,PORT_SSDP					= 1900
	,PORT_TEREDO				= 3544 /* Teredo tunneling over UDP */
	,PORT_AIM					= 5190
	,PORT_XMPP					= 5222
	,PORT_GNUTELLA				= 6346
	,PORT_IRC					= 6667
	,PORT_HTTP8080				= 8080
	,PORT_HTTP8081				= 8081
	,PORT_CSTRIKE_CLIENT		= 27014 /* Counter-Strike Client */
	,PORT_CSTRIKE				= 27015 /* Counter-Strike Server */
};

#define PORT_DESCR_BUFLEN				64 /* enough for an human-readable port descr */


/* ICMP - Internet Control Message Protocol */
/* RFC 792 */
#define ICMP_LEN_MIN					8
#define ICMP_LEN_MAX					40 /* FIXME: check! */

/* ICMPV6 - Internet Control Message Protocol v6 */
/* RFC ??? */
#define ICMPV6_LEN_MIN					8
#define ICMPV6_LEN_MAX					40 /* FIXME: check! */

#define ICMP_TYPE_UNREACH				0x03
	#define ICMP_UNREACH_NET			0x0
	#define ICMP_UNREACH_HOST	 		0x1
	#define ICMP_UNREACH_PROT			0x2
	#define ICMP_UNREACH_PORT			0x3
	#define ICMP_UNREACH_FRAG_DF		0x4
	#define ICMP_UNREACH_SRC_RT			0x5
	#define ICMP_UNREACH_ADM_FILT		0xD
#define ICMP_TYPE_TIMEOUT				0x0B
	#define ICMP_TIMEOUT_TRANSIT		0x0
	#define ICMP_TIMEOUT_FRAG			0x1
#define ICMP_TYPE_PARAM					0x0C
	/* TODO */
#define ICMP_TYPE_SRC_QUENCH			0x04
	/* TODO */
#define ICMP_TYPE_SRC_QUENCH			0x04
#define ICMP_TYPE_REDIRECT				0x05
	#define ICMP_REDIRECT_NET			0x0
	#define ICMP_REDIRECT_HOST			0x1
	#define ICMP_REDIRECT_SRV_NET		0x2
	#define ICMP_REDIRECT_SRV_HOST		0x3
#define ICMP_TYPE_ECHO_REQ				0x08
#define ICMP_TYPE_ECHO_RESP				0x00
#define ICMP_TYPE_TIMESTAMP_REQ			0x0D
#define ICMP_TYPE_TIMESTAMP_RESP		0x0E
#define ICMP_TYPE_INFO_REQ				0x0F
#define ICMP_TYPE_INFO_RESP				0x10
#define ICMP_TYPE_ADMIN_FILTERED		0x13
#define ICMP_TYPE_WHOOPS_TOO_HIGH		0x14 /* upper boundary */

/* shorter aliases... */
#define PING_REQ						ICMP_TYPE_ECHO_REQ
#define PING_RESP						ICMP_TYPE_ECHO_RESP

/* * * * * DNS - Domain Name System * * * * */
/* RFC ??? */
#define DNS_LEN_MIN						12 /* FIXME: is this right?!? */
#define DNS_LEN_MAX						DNS_LEN_MIN /* FIXME! */

#define DNS_TYPE_REQ					0x0
#define DNS_TYPE_RESP					0x1

#define DNS_OP_DYN_UPD					0x5

#define DNS_HOST_TYPE_HOST				0x01
#define DNS_HOST_TYPE_NAMESERVER		0x02
#define DNS_HOST_TYPE_IPV6				0x0A /* FIXME: */

#define DNS_HOST_BYTES					83
#define DNS_HOST_BUFLEN					DNS_HOST_BYTES + 1

#define DNS_REC_TYPE_A					0x0001
#define DNS_REC_TYPE_NS					0x0002
#define DNS_REC_TYPE_CNAME				0x0005
#define DNS_REC_TYPE_SOA				0x0006
#define DNS_REC_TYPE_PTR				0x000C
#define DNS_REC_TYPE_HINFO				0x000D
#define DNS_REC_TYPE_MX					0x000F
#define DNS_REC_TYPE_AAAA				0x001C

#define DNS_HINFO_CPU_BUFLEN			128
#define DNS_HINFO_OS_BUFLEN				128

/* NOTE: arbitrary limits, check RFCs (there's too many!) */
#define DNS_QUERIES_MAX					16
#define DNS_ANS_RR_MAX					16
#define DNS_AUTH_RR_MAX					16
#define DNS_ADD_RR_MAX					16


/* * * * BOOTP - Boostrap Protocol * * * */
/* RFC ??? */
#define BOOTP_LEN_MIN					206 /* FIXME: verify */
#define BOOTP_LEN_MAX					1024 /* FIXME: verify */

#define BOOTP_TYPE_REQ					0x01
#define BOOTP_TYPE_RESP					0x02

#define BOOTP_OPTS_MAX					64 /* FIXME: verify */

/* NOTE: decimal! */
#define BOOTP_OPT_SUBNET_MASK			 1
#define BOOTP_OPT_ROUTER				 3
#define BOOTP_OPT_DNS_SERVER			 6
#define BOOTP_OPT_HOSTNAME				12
#define BOOTP_OPT_DOMAIN				15
#define BOOTP_OPT_IP_REQ				50
#define BOOTP_OPT_LEASE_TIME			51
#define BOOTP_OPT_MSG_TYPE				53
#define BOOTP_OPT_SERVER_ID				54
#define BOOTP_OPT_PARAM_REQ_LIST		55
#define BOOTP_OPT_TIME_RENEW			58
#define BOOTP_OPT_TIME_REBIND			59
#define BOOTP_OPT_VENDOR_CLASS			60

#define BOOTP_SERVER_HOST_BUFLEN		64 + 1
#define BOOTP_BOOT_FILE_BUFLEN			128 + 1
#define BOOTP_PARAM_REQ_MAX				64
#define BOOTP_VENDOR_CLASS_BUFLEN		64 + 1
#define BOOTP_DOMAIN_BUFLEN				64 + 1
#define BOOTP_HOSTNAME_BUFLEN			64 + 1

enum {
	BOOTP_DISC = 1,
	BOOTP_OFF = 2,
	BOOTP_REQ = 3,
	BOOTP_DECL = 4,
	BOOTP_ACK = 5,
	BOOTP_NAK = 6,
	BOOTP_REL = 7,
	BOOTP_INF = 8,
	BOOTP_COUNT, /* mine, WRONG?! */
};

#define BOOTP_OPTS_END_BYTE 			0xFF /* "stop" byte */


/* * * * * * NetBIOS * * * * * * */

#define NB_DGM_LEN_MIN				82 /* FIXME: */

#define NB_DGM_TYPE_UNKNOWN			0x00
#define NB_DGM_TYPE_GROUP_DGM		0x11

#define NB_DGM_SRC_NAME_MAXLEN		63
#define NB_DGM_SRC_NAME_BUFLEN		NB_DGM_SRC_NAME_MAXLEN + 1
#define NB_DGM_DEST_NAME_MAXLEN		63
#define NB_DGM_DEST_NAME_BUFLEN		NB_DGM_DEST_NAME_MAXLEN + 1

#define NB_SSN_LEN_MIN				4 /* FIXME: ??? */


typedef struct protonode_ protonode_t;
/**
 * big, ugly type that can store any single packet of information for any
 * supported protocol
 */
struct protonode_ {
	u_int prot_id;
	size_t bytes; /* raw bytes used */
	protonode_t *parent, *child;

	union { /* protocol-specific data */

		struct {
			size_t len;
			const u_char *data; /* just a pointer to original, not allocated */
		} unknown;

		/* Logical Frame */
		struct {
			uint32_t frame;
			uint32_t len;
			time_t when;
			int type;
		} logic;

		/* SLL - Linux Cooked Capture */
		struct {
			uint16_t type;
			uint16_t dev_type;
			uint16_t ll_len;
			uint16_t ll_type;
			u_char ll_addr[8];
			uint16_t ethernet_type;
		} linux_sll;

		/* Ethernet II */
		/* IEEE 802.3 Ethernet */
		struct {
			struct mac src;
			struct mac dest;
			union {
				uint16_t type; /* ethernet 2 */
				uint16_t len; /* 802.3 */
			} data;
			uint16_t pad_len;
			const u_char *pad_bytes;
		} ether;

		/* loopback - Configuration Test Protocol */
		struct {
			uint16_t skip;
			uint16_t func;
			uint16_t receiptno;
			uint32_t datalen;
			const u_char *data;
		} loop;

		/* ARP - Address Resolution Protocol */
		/* AARP - Apple Address Resolution Protocol */
		struct {
			uint16_t hw; /* hardware type */
			uint16_t prot;
			u_char hw_size;
			u_char prot_size;
			uint16_t opcode;
			struct mac src_mac;
			struct ip src_ip;
			struct mac dest_mac;
			struct ip dest_ip;
		} arp, aarp;

		/* Logical Link Control */
		struct {
			struct {
				uint8_t ig:1, addr:7;
				uint8_t whole;
			} dsap; /* destination service access point */
			struct {
				uint8_t cr:1, addr:7;
				uint8_t whole;
			} ssap; /* source service access point */
			u_char cmd:6, frame_type:2;
			union {
				struct {
					uint32_t org_id;
					uint16_t pid;
				} snap;
			} data;
		} llc;

		/* Internetwork Packet Exchange */
		struct {
			uint16_t checksum;
			uint16_t len;
			u_char hops;
			u_char type;
			struct {
				uint32_t net;
				struct mac node;
				uint16_t socket;
			} dest;
			struct {
				uint32_t net;
				struct mac node;
				uint16_t socket;
			} src;
		} ipx;

		/* SAP - Service Advertisement Protocol */
		struct {
			uint16_t code;
			struct {
				uint16_t type;
				char name[SAP_SERVER_NAME_BUFLEN];
				uint32_t network;
				struct mac node;
				uint16_t socket;
				uint16_t intermed_nets;
			} server;
		} sap;

		/* Cisco Discovery Protocol */
		struct {
			u_char version;
			u_char ttl;
			uint16_t checksum;
			char dev_id[CDP_DEV_DESCR_BUFLEN];
			struct {
				uint32_t count;
				struct {
					char prot_type;
					u_char len;
					uint16_t protocol;
					uint16_t addr_len;
					u_char addr[CDP_ADDR_BUFLEN];
				} addr[CDP_ADDR_MAX];
			} addrs;
			char port[CDP_PORT_BUFLEN];
			char version_id[CDP_VERSION_BUFLEN];
			char platform[CDP_PLATFORM_BUFLEN];
			uint32_t capab_unused:25,
					 capab_router:1,
					 capab_trans_bridge:1,
					 capab_srcrt_bridge:1,
					 capab_switch:1,
					 capab_host:1,
					 capab_igmp:1,
					 capab_repeater:1;
			int todo_ip_prefix;
		} cdp;

		/* Spanning Tree Protocol */
		struct {
			uint16_t prot_id;
			u_char prot_ver;
			u_char bdpu_type;
			u_char topo_chg_ack:1,
				   flag_unused:6,
				   topo_chg:1;
			struct {
				uint16_t num;
				struct mac mac;
			} root_id;
			uint32_t path_cost;
			struct {
				uint16_t num;
				struct mac mac;
			} bridge_id;
			uint16_t port_id;
			uint16_t msg_age;
			uint16_t max_age;
			uint16_t hello_time;
			uint16_t fwd_delay;
		} stp;

		/* IP - Internet Protocol */
		struct {
			u_char version;
			u_char headlen;
			u_char prec:3,
				   lowdelay:1,
				   throughput:1,
				   reliable:1,
				   ect:1,
				   ece:1;
			uint16_t len;
			uint16_t id;
			u_char res:1,
				   dontfrag:1,
				   morefrag:1;
			uint16_t fragoff;
			u_char ttl;
			u_char prot;
			uint16_t checksum;
			struct ip src;
			struct ip dest;
			u_char opt_len;
			u_char opt_bytes[IP_OPT_MAX_LEN];
		} ip;

		/* IGMP - Internet Group Management Protocol */
		struct {
			u_char version;
			u_char type;
			u_char max_resp;
			uint16_t checksum;
			struct ip multicast;
		} igmp;

		/* IPv6 - Internet Protocol v6 */
		struct {
			u_char version; /* 4-bit version, always 6 */
			u_char traffic_class; /* 8-bit traffic class */
			uint32_t flow_label; /* 20-bit flow label */
			uint16_t payload_len; /* 16-bit payload length */
			u_char next; /* 8-bit type of encapsulated data */
			u_char hop_limit; /* 8-bit ttl */
			struct ip src; /* 128-bit src */
			struct ip dest; /* 128-bit dest */
		} ipv6;

		/* ICMP - Internet Control Message Protocol */
		struct {
			u_char type;
			u_char code;
			uint16_t checksum;
			/* 4 bytes that mean different things for different types */
			union {
				u_char unused[4];
				struct {
					u_char offset;
					u_char unused[3];
				} param;
				struct ip gateway; /* redirect */
				struct {
					uint16_t id;
					uint16_t seq;
					size_t data_len;
					const u_char *data; /* pointer to echo data, rest of buffer */
				} echo;
				struct {
					uint32_t orig;
					uint32_t recv;
					uint32_t trans;
				} timestamp;
				struct {
					uint16_t id;
					uint16_t seq;
				} info;
			} data;
		} icmp;

		/* Internet Control Message Protocol v6 */
		struct {
			u_char type;
			u_char code;
			uint16_t checksum;
			uint16_t id;
			uint16_t seq;
			union {
				int todo;
			} data;
		} icmpv6;

		/* UDP - User Datagram Protocol */
		struct {
			uint16_t src_port;
			uint16_t dest_port;
			uint16_t len;
			uint16_t checksum;
		} udp;

		/* DNS - Domain Name System */
		/* FIXME: incomplete, check the goddamn RFC(s) (there are 50 of them :/) */
		struct {
			uint16_t trans_id;
			uint32_t type:1,
					 opcode:4,
					 auth:1,
					 trunc:1,
					 rec_des:1,
					 rec_avail:1,
					 Z:1,
					 ans_auth:1,
					 unused:1,
					 reply:4;
			uint16_t q_cnt; /* query count */
			uint16_t ans_rr_cnt; /* answers */
			uint16_t auth_rr_cnt; /* authoratative nameservers */
			uint16_t add_rr_cnt; /* additional records */
			struct {
				char host[DNS_HOST_BUFLEN];
				uint16_t type;
				uint16_t class;
			} q[DNS_QUERIES_MAX];
			struct {
				char host[DNS_HOST_BUFLEN];
				uint16_t type;
				uint16_t cl;
				uint32_t ttl;
				uint16_t datalen;
				union {
					/* record-specific data, if any */
					struct {
						struct ip ip;
					} a, aaaa;
					struct {
						char cpu[DNS_HINFO_CPU_BUFLEN]; /* FIXME: check limits... use separate var */
						char os[DNS_HINFO_OS_BUFLEN];
					} hinfo;
					struct {
						char domain[DNS_HOST_BUFLEN];
					} ptr, in;
					struct {
						uint16_t priority;
						uint16_t weight;
						uint16_t port;
						char target[DNS_HOST_BUFLEN];
					} srv;
					struct {
						char txt[256]; /* FIXME: check len! */
					} txt;
				} data;
			} ans_rr[DNS_ANS_RR_MAX];
			struct {
				int todo;
			} auth_rr[DNS_AUTH_RR_MAX];
			struct {
				int todo;
			} add_rr[DNS_ADD_RR_MAX];
		} dns;

		/* BOOTP - Bootstrap Protocol */
		/* FIXME: incomplete, check the goddamn RFC */
		struct {
			u_char type;
			u_char hw_type;
			u_char hw_len;
			u_char hops;
			uint32_t trans_id;
			uint16_t secs;
			uint16_t flag_bcast:1,
					 flag_res:15;
			struct ip client_ip;
			struct ip your_ip;
			struct ip next_ip;
			struct ip relay_ip;
			struct mac client_mac;
			char server_host[BOOTP_SERVER_HOST_BUFLEN];
			char boot_file[BOOTP_BOOT_FILE_BUFLEN];
			uint32_t magic_cookie;
			u_char opt_count;
			/* yuck, a shitload of options possibilies... read the goddamn RFC */
			struct {
				u_char type;
				union {
					struct ip subnet_mask;
					struct ip router;
					struct ip ip_req;
					uint32_t lease_time;
					u_char msg_type;
					struct ip server_id;
					struct {
						u_char size;
						u_char req[BOOTP_PARAM_REQ_MAX];
					} param_req_list;
					uint16_t max_size;
					char domain[BOOTP_DOMAIN_BUFLEN];
					char hostname[BOOTP_HOSTNAME_BUFLEN];
					char vendor_class[BOOTP_VENDOR_CLASS_BUFLEN];
					struct {
						u_char type;
						union {
							struct mac mac;
						} data;
					} client_id;
					uint32_t secs; /* renew time, rebind time */
				} data;
			} opts[BOOTP_OPTS_MAX];
		} bootp;

		/* NETBIOS Datagram */
		/* RFC: ??? */
		struct {
			u_char type;
			uint8_t node_type:6,
					first_frag:1,
					more_frag:1;
			uint16_t id;
			struct ip src_ip;
			uint16_t src_port;
			uint16_t len;
			uint16_t offset;
			u_char src_name_len;
			char src_name[NB_DGM_SRC_NAME_BUFLEN];
			u_char dest_name_len;
			char dest_name[NB_DGM_DEST_NAME_BUFLEN];
		} nb_dgm;

		/* NETBIOS Session */
		struct {
			u_char type;
			u_char flags;
			uint16_t size;
		} nb_ssn;

		/* Server Message Block */
		struct {
			int todo;
		} smb;

		/* DDP - Data Delivery Protocol */
		struct {
			uint16_t hops:8, /* FIXME: verify sizes */
					 len:8;
			uint16_t checksum;
			uint16_t dest_net;
			uint16_t src_net;
			uint8_t dest_node;
			uint8_t src_node;
			uint8_t dest_socket;
			uint8_t src_socket;
			uint8_t prot_type;
		} ddp;

		/* TCP - Transfer Control Protocol */
		struct {
			uint16_t src_port;
			uint16_t dest_port;
			uint32_t seqno;
			uint32_t ackno;
			u_char headlen;
			uint16_t reserved:4,
					 cwr:1,
					 ecn:1,
					 urg:1,
					 ack:1,
					 psh:1,
					 rst:1,
					 syn:1,
					 fin:1;
			uint16_t window;
			uint16_t checksum;
			uint16_t urg_ptr;
			u_char opts;
			struct {
				u_char type;
				union {
					uint16_t mss;
					u_char ws;
					struct {
						uint32_t val;
						uint32_t ecr;
					} ts;
				} data;
			} opt[TCP_OPTS_MAX];
			const u_char *payload;
			size_t payload_bytes;
		} tcp;

		/* IAPP - Inter-Access Point Protocol */
		struct {
			u_char version;
			u_char cmd;
			uint16_t id;
			uint16_t len;
			union {
				struct {
					u_char addr_len;
					u_char res;
					struct mac mac;
					uint16_t seqno;
				} add_no;
#if 0 /* unimplemented */
				struct {
					u_char addr_len;
					u_char res_or_len;
					struct mac mac;
					uint16_t seqno;
					uint16_t context_len;
					u_char *context;
				} move_no, move_resp;
#endif
			} data;
		} iapp;

		/* GRE - Generic Route Encapsulation */
		struct {
			uint16_t checksum:1,
					 routing:1,
					 key:1,
					 seqno:1,
					 str_src_rt:1,
					 rec_ctrl:3,
					 has_ack_no:1,
					 flags:4,
					 version:3;
			uint16_t prot_type;
			uint16_t payload;
			uint16_t call_id;
			uint32_t ack_no;
		} gre;

		struct {
			int todo;
		} smtp;

		/* Teredo Tunneling */
		struct {
			uint16_t zero_one;
			uint8_t id_len;
			uint8_t au_len;
			u_char id[255];
			u_char au[255];
			u_char nonce[8];
			uint8_t conf;
			/* origin indication */
			int has_origin;
			struct {
				uint16_t zero_zero;
				uint16_t port;
				struct ip ip;
			} origin;
		} teredo;

		/* Counter-Strike */
		struct {
			int todo;
		} cstrike;

		/* Symbol Wifi */
		struct {
			struct ip ip;
			char ssid[38 + 1];
			char freq_descr[16 + 1];
			char ap_descr[32 + 1];
		} symw;

		/* CiscoWL */
		struct {
			uint16_t len;
			uint16_t type;
			struct mac dst;
			struct mac src;
			uint16_t unknown1;
			uint32_t unknown2;
		} ciscowl;

	} data;
};

void protocol_init(void);

const struct mac_prot * mac_addr_prot(const struct mac *);
#if 0
const char * mac_addr_vend_to_str(int);
#endif

char * port_descr(uint16_t, char *, size_t);
char * mac_addr_to_str(const struct mac *, char *, size_t);
char * ip_addr_to_str(const struct ip *, char *, size_t);

const char * mac_addr_descr(const struct mac *);

#define IPV4_ADDR_NONE			"\x00\x00\x00\x00"
#define IPV4_ADDR_BCAST			"\xFF\xFF\xFF\xFF"

#define IPV6_ADDR_NONE			"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"


protonode_t * node_new_parse(const u_char *, size_t, int);
void node_report(protonode_t *);
void node_free(protonode_t *);

size_t parse_linux_sll(const u_char *, size_t, protonode_t *);
size_t parse_unknown(const u_char *, size_t, protonode_t *);
size_t parse_padding(const u_char *, size_t, protonode_t *);
size_t parse_ethernet2(const u_char *, size_t, protonode_t *);
	size_t parse_loop(const u_char *, size_t, protonode_t *);
	size_t parse_symbol_wifi(const u_char *, size_t, protonode_t *);
size_t parse_ethernet802_3(const u_char *, size_t, protonode_t *);
	size_t parse_arp(const u_char *, size_t, protonode_t *);
	size_t parse_llc(const u_char *, size_t, protonode_t *);
		size_t parse_cdp(const u_char *, size_t, protonode_t *);
		size_t parse_stp(const u_char *, size_t, protonode_t *);
		size_t parse_ipx(const u_char *, size_t, protonode_t *);
			size_t parse_sap(const u_char *, size_t, protonode_t *);
		size_t parse_ddp(const u_char *, size_t, protonode_t *);
		size_t parse_aarp(const u_char *, size_t, protonode_t *);
		size_t parse_ciscowl(const u_char *, size_t, protonode_t *);
size_t parse_ip(const u_char *, size_t, protonode_t *);
	size_t parse_ipv6(const u_char *, size_t, protonode_t *);
		size_t parse_icmpv6(const u_char *, size_t, protonode_t *);
	size_t parse_icmp(const u_char *, size_t, protonode_t *);
	size_t parse_igmp(const u_char *, size_t, protonode_t *);
	size_t parse_udp(const u_char *, size_t, protonode_t *);
		size_t parse_dns(const u_char *, size_t, protonode_t *);
		size_t parse_bootp(const u_char *, size_t, protonode_t *);
		size_t parse_netbios_ns(const u_char *, size_t, protonode_t *);
		size_t parse_netbios_dgm(const u_char *, size_t, protonode_t *);
		size_t parse_netbios_ssn(const u_char *, size_t, protonode_t *);
			size_t parse_smb(const u_char *, size_t, protonode_t *);
		size_t parse_snmp(const u_char *, size_t, protonode_t *);
		size_t parse_teredo(const u_char *, size_t, protonode_t *);
		size_t parse_cstrike(const u_char *, size_t, protonode_t *);
		size_t parse_ssdp(const u_char *, size_t, protonode_t *);
	size_t parse_tcp(const u_char *, size_t, protonode_t *);
		size_t parse_http(const u_char *, size_t, protonode_t *);
		size_t parse_https(const u_char *, size_t, protonode_t *);
		size_t parse_smtp(const u_char *, size_t, protonode_t *);
		size_t parse_aim(const u_char *, size_t, protonode_t *);
		size_t parse_irc(const u_char *, size_t, protonode_t *);
		size_t parse_jabber(const u_char *, size_t, protonode_t *);
		size_t parse_gnutella(const u_char *, size_t, protonode_t *);
		size_t parse_rtsp(const u_char *, size_t, protonode_t *);
	size_t parse_gre(const u_char *, size_t, protonode_t *);
size_t parse_ppp(const u_char *, size_t, protonode_t *);

void prot_dump(protonode_t *);

void ping_report(protonode_t *);
void report_tcp(protonode_t *);
void report_http_useragent(protonode_t *, const char *ua);
void bootp_report(protonode_t *);

int mac_is_special(const struct mac *);

int ip_is_special(const struct ip *);
int ip_is_private(const struct ip *);
int ip_is_broadcast(const struct ip *);
int ip_subnet_match(const struct ip *a, const struct ip *b, const u_char *mask);

const char * icmp_type_to_str(u_char);

#endif /* PROTOCOLS_H */


