// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptObjectClassDefinition"

#include "kvi_defines.h"
#include "kvi_locale.h"
#include "kvi_mirccntrl.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_window.h"

/**
 *  Script object class definitions.
 *  All are kept in an application-widget global list.
 */
KviScriptObjectClassDefinition::KviScriptObjectClassDefinition(
	const char *className, KviScriptObjectClassDefinition *inheritedClass, bool bBuiltin)
{
	m_pInheritedClass = inheritedClass;
	m_szClassName     = className;
	m_pChildClasses   = new QPtrList<KviScriptObjectClassDefinition>;
	m_pChildClasses->setAutoDelete(true);
	m_bBuiltin        = bBuiltin;
	m_pEventList      = new QPtrList<KviScriptEventStruct>;
	m_pEventList->setAutoDelete(true);
	m_pFunctionDict   = new QAsciiDict<KviScriptObjectFunctionStruct>(23, false);
	m_pFunctionDict->setAutoDelete(true);
	if( inheritedClass ) {
		QAsciiDictIterator<KviScriptObjectFunctionStruct> it(*(inheritedClass->m_pFunctionDict));
		while( it.current() ) {
			KviScriptObjectFunctionStruct *f = new KviScriptObjectFunctionStruct();
			f->fncHandler = it.current()->fncHandler;
			f->flags      = it.current()->flags | KVI_SCRIPTOBJECTFUNCTION_FLAG_INHERITED;
			f->szBuffer   = it.current()->szBuffer;
			m_pFunctionDict->replace(it.currentKey(), f);
			++it;
		}
		if( inheritedClass->isBuiltin() ) {
			addFunction("constructor", "setreturn 1", true);
			addFunction("destructor",  "setreturn 1", true);
		} else {
			KviStr tmp(KviStr::Format, "setreturn $this->$%s:constructor()", inheritedClass->getClass());
			addFunction("constructor", tmp.ptr(), true);
			tmp.sprintf("setreturn $this->$%s:destructor()", inheritedClass->getClass());
			addFunction("destructor", tmp.ptr(), true);
		}
	} else {
		addFunction("constructor", "setreturn 1", true);
		addFunction("destructor",  "setreturn 1", true);
	}
}

KviScriptObjectClassDefinition::~KviScriptObjectClassDefinition()
{
	delete m_pChildClasses; m_pChildClasses = 0;
	delete m_pFunctionDict; m_pFunctionDict = 0;
	delete m_pEventList;    m_pEventList    = 0;
}

QAsciiDict<KviScriptObjectFunctionStruct> *KviScriptObjectClassDefinition::functionDict()
{
	return m_pFunctionDict;
}

QPtrList<KviScriptEventStruct> *KviScriptObjectClassDefinition::eventList()
{
	return m_pEventList;
}

KviScriptObjectClassDefinition *KviScriptObjectClassDefinition::inheritedClass()
{
	return m_pInheritedClass;
}

QPtrList<KviScriptObjectClassDefinition> *KviScriptObjectClassDefinition::childClasses()
{
	return m_pChildClasses;
}

const char *KviScriptObjectClassDefinition::getClass()
{
	return m_szClassName.ptr();
}

bool KviScriptObjectClassDefinition::isBuiltin()
{
	return m_bBuiltin;
}

KviScriptObjectClassDefinition *KviScriptObjectClassDefinition::lookupParentClassDefinition(const char *szClass)
{
	if( m_pInheritedClass ) {
		if( kvi_strEqualCI(m_pInheritedClass->getClass(), szClass) )
			return m_pInheritedClass;
		return m_pInheritedClass->lookupParentClassDefinition(szClass);
	} else return 0;
}

KviScriptObjectClassDefinition *KviScriptObjectClassDefinition::lookupChildClassDefinition(const char *szClass)
{
	for( KviScriptObjectClassDefinition *d = m_pChildClasses->first(); d; d = m_pChildClasses->next() ) {
		if( kvi_strEqualCI(d->getClass(), szClass) )
			return d;
		KviScriptObjectClassDefinition *child = d->lookupChildClassDefinition(szClass);
		if( child ) return child;
	}
	return 0;
}

KviScriptObjectFunctionStruct *KviScriptObjectClassDefinition::lookupFunction(const char *szFncName)
{
	return m_pFunctionDict->find(szFncName);
}

void KviScriptObjectClassDefinition::addChildClassDefinition(KviScriptObjectClassDefinition *d)
{
	m_pChildClasses->append(d);
}

void KviScriptObjectClassDefinition::addDefaultEvent(KviScriptEventStruct *s)
{
	for( KviScriptEventStruct *str = m_pEventList->first(); str; str = m_pEventList->next() ) {
		if( kvi_strEqualCI(str->szName.ptr(), s->szName.ptr()) ) {
			str->szBuffer = s->szBuffer;
			delete s;
			return;
		}
	}
	m_pEventList->append(s);
}

void KviScriptObjectClassDefinition::addFunction(const char *fncName, const char *buffer, bool bBuiltin)
{
	KviScriptObjectFunctionStruct *f = new KviScriptObjectFunctionStruct();
	f->fncHandler = 0;
	f->flags      = bBuiltin ? KVI_SCRIPTOBJECTFUNCTION_FLAG_BUILTIN : 0;
	f->szBuffer   = buffer;
	if( m_pInheritedClass ) {
		KviScriptObjectFunctionStruct *old = m_pInheritedClass->m_pFunctionDict->find(fncName);
		if( old )
			f->flags |= KVI_SCRIPTOBJECTFUNCTION_FLAG_OVERRIDE;
	}
	m_pFunctionDict->replace(fncName, f);
}

void KviScriptObjectClassDefinition::addBuiltinFunction(const char *fncName, scriptObjectFunction fncHandler)
{
	KviScriptObjectFunctionStruct *f = new KviScriptObjectFunctionStruct();
	f->fncHandler = fncHandler;
	f->flags      = KVI_SCRIPTOBJECTFUNCTION_FLAG_BUILTIN;
	if( m_pInheritedClass ) {
		KviScriptObjectFunctionStruct *old = m_pInheritedClass->m_pFunctionDict->find(fncName);
		if( old )
			f->flags |= KVI_SCRIPTOBJECTFUNCTION_FLAG_OVERRIDE;
	}
	m_pFunctionDict->replace(fncName, f);
}

void KviScriptObjectClassDefinition::dump(KviWindow *wnd, const char *margin)
{
	KviStr tmp(KviStr::Format, _i18n_("%s%cClass definition: %s"), margin ? margin : "", KVI_TEXT_BOLD, m_szClassName.ptr());
	wnd->output(KVI_OUT_KVIRC, tmp.ptr());
	tmp.sprintf(
		_i18n_("%sInherited class: %s"),
		margin ? margin : "", m_pInheritedClass ? m_pInheritedClass->m_szClassName.ptr() : _i18n_("none")
	);
	wnd->output(KVI_OUT_KVIRC, tmp.ptr());
	tmp.sprintf(_i18n_("%sIs built-in: %s"), margin ? margin : "", isBuiltin() ? _i18n_("yes") : _i18n_("no"));
	wnd->output(KVI_OUT_KVIRC, tmp.ptr());
	for( KviScriptObjectClassDefinition *d = m_pChildClasses->first(); d ; d = m_pChildClasses->next() ) {
		KviStr mrgn = margin;
		mrgn.append("    ");
		d->dump(wnd, mrgn.ptr());
	}
}
