/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* rendererbase.h
 */
#ifndef rendererbase_h
#define rendererbase_h

#include "project.h"
#include "renderimage.h"
#include "rendertext.h"
#include "effect.h"

#include <qsize.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qptrlist.h>
#include <qpainter.h>



namespace KoverArtist
{
class RenderEffect;


/** This is the base class for renderers that render the project on
 *  a @ref QPaintDevice.
 */
class RendererBase
{
public:
   RendererBase(Project* p);
   virtual ~RendererBase();

   /** Render the project on the paint device pd. */
   virtual void render(QPaintDevice* pd, bool autoFit=false) = 0;

protected:
   /** Schedule a text for later rendering. The text is queued and will be
    * rendered by @ref renderQueuedTexts()
    */
   virtual void renderText(const QString& txt, QRect rect, int align,
                           const QFont& font, const QColor& color, int angle=0);

   /** Render all queued texts using the painter p. Clears the text queue.
    *  Must be called once at the end of the rendering.
    */
   virtual void renderQueuedTexts(QPainter& p);

   /** Clear the text queue. */
   virtual void deleteQueuedTexts();

   /** Prepare pixmaps for later rendering. */
   virtual void preparePixmaps();

   /** Create a w x h sized pixmap from image img. */
   virtual QPixmap createPixmap(int w, int h, const KoverArtist::Image& img) const;

   /** Start rendering an effect. */
   virtual void beginEffect(const Effect& e);

   /** End rendering an effect. */
   virtual void endEffect();

   /** Start rendering a surface. */
   virtual void beginSurface(QPaintDevice* dev, const QWMatrix& m, int width, int height);

   /** Set the background image for this surface from pixmap pix. */
   virtual void setSurfaceBackground(const QPixmap& pix);

   /** End rendering the current surface. */
   virtual void endSurface();

   /** Finish all rendering. Must be called once after all rendering is done. */
   virtual void finish();

   /** Scales model X coordinate to view coordinate. */
   int scaleX(int modelCoord) const;

   /** Scales model Y coordinate to view coordinate. */
   int scaleY(int modelCoord) const;

   /** Scales model size to view size. */
   QSize scale(const QSize&) const;

   /** Scales model point to view point. */
   QPoint scale(const QPoint&) const;

   /** Scales model rectangle to view rectangle. */
   QRect scale(const QRect&) const;

private:
   RendererBase(const RendererBase&);
   RendererBase& operator=(const RendererBase&);

protected:
   Project* mProject;
   double scaleFactorX, scaleFactorY;
   int mFrontWidth, mBackWidth, mWidth, mHeight;
   QPixmap mPixFront, mPixBack, mPixEff;
   QPainter mPaint, mPaintEff;
   RenderImage mImgBG;
   QRect mBBEff;

private:
   QPtrList<RenderText> mTextList;
   RenderEffect *mREff;
};

}

#endif /*rendererbase_h*/
