/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// mainwindow.h

#ifndef mainwindow_h
#define mainwindow_h

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <kapplication.h>
#include <kmainwindow.h>
#include <kurl.h>

#include "project.h"
#include "renderer.h"


class KRecentFilesAction;
class KPrinter;
class KURL;

class QTimer;
class QTextEdit;
class QTabBar;
class QTab;
class QWidgetStack;
class QLineEdit;
class QPushButton;
class QTextEdit;
class KColorButton;
class QComboBox;
class QLabel;
class QBoxLayout;
class KEditListBox;
class KURLRequester;
class QCheckBox;
class QSlider;

class ImageConfig;
class EffectConfig;
class TextStyleConfig;
class CddbQuery;


/**
* This class serves as the main window for KoverArtist.  It handles the
* menus, toolbars, and status bars.
*
* @short Main window class
* @author Stefan Taferner <taferner@kde.org>
* @version 0.1
*/
class MainWindow: public KMainWindow
{
   Q_OBJECT
public:
   /**
   * Default Constructor
   */
   MainWindow();

   /**
   * Default Destructor
   */
   virtual ~MainWindow();

   /** Import the disc d from url. */
   bool discImport(KoverArtist::Disc& d, const KURL& url);

   /** Export the disc d to url. */
   bool discExport(const KoverArtist::Disc& d, const KURL& url);

public slots:
   /** Mark project as modified. */
   void modified();

   /** Load the project from an url. */
   virtual bool load(const KURL& url);

   /** Save the project to an url. */
   virtual bool save(const KURL& url);

protected slots:
   void fileNew();
   void fileOpen();
   void fileRevert();
   bool fileSave();
   bool fileSaveAs();
   void filePrint();

   void windowNew();

   void discImport();
   void discImportCddb();
   void discExport();
   void discNew();
   void discDelete();
   void discFront();
   void discBack();
   void discClear();

   void searchCoverImages();

   void optionsPreferences();

   void changeStatusbar(const QString& text);
   void changeCaption(const QString& text=QString::null);

   void switchToTab(int);

   void slotTitleChanged();
   void slotSubTitleChanged();

   void slotDiscsInc();
   void slotDiscsDec();
   void slotDiscsEdited();

   void slotDiscTitleChanged();
   void slotDiscContentsChanged();

   void setFrontImageWrap(bool);
   void setFrontTitleHide(bool);

   void setBackgroundColor(const QColor&);
   void setOutlineColor(const QColor&);

   void slotCaseActivated(int);
   void redisplay();

   void updatePreview(int msec=-1);
   void updatePreviewFront(int msec=-1);
   void updatePreviewBack(int msec=-1);
   void updatePreviewNow();

   void updatePreviewTimeout();

   void cddbQueryAudioCdDone();

protected:
   virtual void dragEnterEvent(QDragEnterEvent *event);
   virtual void dropEvent(QDropEvent *event);

protected:
   /**
   * This function is called when it is time for the app to save its
   * properties for session management purposes.
   */
   void saveProperties(KConfig *);

   /**
   * This function is called when this app is restored.  The KConfig
   * object points to the session management config file that was saved
   * with @ref saveProperties
   */
   void readProperties(KConfig *);

   /** Number of discs in the project has changed. */
   virtual void numDiscsChanged();

   /** The window shall be closed. */
   virtual bool queryClose();

   /** An image url was dropped. Let the user select what to do with the image. */
   virtual bool imageDropped(const KURL& url, const QPoint& globalMousePos);

   /** Test if the given url can be decoded by looking at the url only. */
   virtual bool canDecode(const KURL& url) const;

   /** Test if the file is a image by looking at the filename. */
   virtual bool isImageFile(const QString& fname) const;

private:
   void setupAccel();
   void setupActions();
   void setupProjectTab();
   void setupOptionsTab();
   void setupDiscTab();

   void updateFromProject();
   void updateDiscFromProject();
   void scheduleUpdatePreview(int tmout=-1);

   bool confirmDiscNotEmpty(int discIndex);

   QPushButton* createButton(const char* icon, QWidget* parent,
                             const char* slot, QBoxLayout* box=0) const;
   QLabel* createCaptionLabel(const QString& aText, QWidget* aParent,
                              const char* aName=0) const;

private:
   MainWindow(const MainWindow&);
   MainWindow& operator=(const MainWindow&);

private:
   KPrinter *mPrinter;
   KRecentFilesAction *mRecentFiles;
   KoverArtist::Project mProject;
   KoverArtist::Renderer *mRender;
   CddbQuery *mCddbQuery;
   QFont mCaptionLblFont;
   int mIdActiveDisc, mNumTabsNoDisc, mCddbDiscId;
   QString mLocalFileName;
   QPixmap mPixPreview;
   bool mUpdPreviewFront, mUpdPreviewBack, mRendering;
   QLabel *mPreview;
   QTimer *mUpdateTimer, *mClearStatusbarTimer;
   QTabBar *mTabBar;
   QWidgetStack *mTabStack;
   QTab *mTabProject, *mTabOptions;
   QWidget *mBaseProject, *mBaseOptions, *mBaseDisc;

   QTextEdit *mEdtProjTitle, *mEdtProjSubTitle, *mEdtDiscTitle;
   QLineEdit *mEdtNumDiscs;
   QPushButton *mBtnDiscsInc, *mBtnDiscsDec;
   QComboBox *mCboCase;
   KEditListBox *mElbDiscContents;
   QCheckBox *mCbxFrontImageWrap, *mCbxFrontTitleHide;
   EffectConfig *mEfcTitle, *mEfcContents, *mEfcSide;
   TextStyleConfig *mTscTitle, *mTscSubTitle, *mTscContents, *mTscSide;
   ImageConfig *mImcFront, *mImcBack;
   KColorButton *mBtnBackground, *mBtnOutline;

   enum { ACT_DISC_NEW, ACT_DISC_DELETE, ACT_DISC_IMPORT, ACT_DISC_IMPORT_CDDB,
          ACT_DISC_FRONT, ACT_DISC_BACK, ACT_DISC_CLEAR, ACT_DISC_EXPORT,
          ACT_PROJ_SAVE, ACT_PROJ_REVERT,
          _NUM_ACTIONS };
   KAction *mAction[_NUM_ACTIONS];
};

#endif // _KOVERARTIST_H_

