/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* image.h
 */
#ifndef image_h
#define image_h

#include <qstring.h>
#include <qpixmap.h>
#include <qcolor.h>
#include <kurl.h>

class QDomElement;
class QDomDocument;
class KURL;


namespace KoverArtist
{

/** The position of an image.
 */
enum ImagePosition
{
   Centered=0, Tiled, CenteredTiled, CenteredMaxpect, Scaled, ScaleCrop
};



/** An image with layout information.
 */
class Image
{
public:
   Image();
   Image(const Image& o);
   Image& operator=(const Image& o);

   /** Returns the image url. */
   const KURL& url() const { return mUrl; }

   /** Load the image from an url. Returns true on success. */
   bool load(const KURL& url);

   /** Set the image url without loading the image. */
   void setUrl(const KURL& u) { mUrl = u; }

   /** Returns the image. */
   const QPixmap& pixmap() const { return mPixmap; }
   QPixmap& pixmap() { return mPixmap; }

   /** Test if the image is valid. */
   bool isNull() const { return mPixmap.isNull(); }

   /** Get/set the image position. */
   ImagePosition position() const { return mPosition; }
   void setPosition(ImagePosition ip) { mPosition = ip; }

   /** Get/set indicator if front image is drawn on sides too. */
   bool onSides() const { return mOnSides; }
   void setOnSides(bool f) { mOnSides = f; }

   /** Get/set image transparency: 0=fully transparent, 255=opaque. */
   int transparency() const { return mTransparency; }
   void setTransparency(int t) { mTransparency = t; }

   /** Fill the DOM element de with the object's contents. */
   void toDom(QDomDocument& doc, QDomElement& de, const QString& projectDir) const;

   /** Set the object from the DOM element de. */
   bool fromDom(const QDomElement& de, const QString& projectDir);

   /** Clear the object. */
   void clear();

   /** Returns image-position as string. */
   static const char* imagePositionAsString(ImagePosition ip);

   /** Translates string str to image-position. */
   static ImagePosition imagePositionFromString(const QString& str);

protected:
   /** Remove black lines from left/right side of the image by cutting the parts off. */
   QPixmap imageToPixmapBordersRemoved(QImage& img);

protected:
   KURL mUrl;
   ImagePosition mPosition;
   QPixmap mPixmap;
   bool mOnSides;
   int mTransparency;
};


} //namespace

#endif /*image_h*/
