/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <QtCore/QSettings>
#include <QtGui/QAction>
#include <QtGui/QApplication>
#include <QtGui/QComboBox>
#include <QtGui/QFileDialog>
#include <QtGui/QGridLayout>
#include <QtGui/QLabel>
#include <QtGui/QLineEdit>
#include <QtGui/QListWidget>
#include <QtGui/QMessageBox>
#include <QtGui/QPushButton>
#include <QtGui/QSpinBox>

#include "auth.h"
#include "connection.h"
#include "itemview.h"

#include "connectiondialog.h"

ConnectionDialog::ConnectionDialog( QWidget *parent )
  : QDialog( parent )
{
  QAction* quitAction = new QAction( tr( "&Quit" ), this );
  quitAction->setShortcut( Qt::CTRL + Qt::Key_Q );
  connect( quitAction, SIGNAL( triggered( bool ) ), qApp, SLOT( quit() ) );
  addAction( quitAction );

  QGridLayout *layout = new QGridLayout( this );

  QLabel *label = new QLabel( this );
  QPixmap pixmap( ":kolab_logo" );
  label->setPixmap( pixmap );
  label->setFixedSize( pixmap.size() );
  layout->addWidget( label, 0, 0, 1, 6, Qt::AlignHCenter | Qt::AlignTop );
  layout->setRowMinimumHeight( 0, pixmap.size().height() + layout->spacing() * 4 );

  mConnection = new QComboBox( this );
  layout->addWidget( mConnection, 1, 1, 1, 4 );

  label = new QLabel( tr( "&User:" ), this );
  mUser = new QLineEdit( this );
  label->setBuddy( mUser );

  layout->addWidget( label, 2, 0 );
  layout->addWidget( mUser, 2, 1, 1, 4 );

  label = new QLabel( tr( "&Password:" ), this );
  mPassword = new QLineEdit( this );
  mPassword->setEchoMode( QLineEdit::Password );
  label->setBuddy( mPassword );

  layout->addWidget( label, 3, 0 );
  layout->addWidget( mPassword, 3, 1, 1, 4 );

  QPushButton *button = new QPushButton( tr( "&Quit" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( reject() ) );
  layout->addWidget( button, 4, 1 );

  button = new QPushButton( tr( "&Configure ..." ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( configure() ) );
  layout->addWidget( button, 4, 2 );

  layout->addItem( new QSpacerItem( 70, 20 ), 5, 3 );

  mOkButton = new QPushButton( tr( "&Ok" ), this );
  mOkButton->setDefault( true );
  connect( mOkButton, SIGNAL( clicked() ), this, SLOT( accept() ) );
  layout->addWidget( mOkButton, 4, 4 );

  mConnectionInfoManager.load();

  updateButtonState();
  updateConnectionBox();
}

void ConnectionDialog::accept()
{
  if ( mUser->text().isEmpty() || mPassword->text().isEmpty() ) {
    QMessageBox::critical( 0, QString(), tr( "You have to enter the login data!" ) );
    return;
  }

  ConnectionInfo info = mConnectionInfoManager.connectionInfo( mConnection->currentText() );
  if ( !info.isValid() ) {
    QMessageBox::critical( 0, QString(), tr( "Your selected connection is invalid!" ) );
    return;
  }

  QLdap ldap( this );
  ldap.setHost( info.host() );
  ldap.setPort( info.port() );
  ldap.setBaseDn( info.baseDn() );
  ldap.setUser( info.nobodyUser() );
  ldap.setPassword( info.nobodyPassword() );

  const QString filter = QString( "(&(objectClass=kolabInetOrgPerson)(|(uid=%1)(mail=%2)))" )
                                .arg( mUser->text() ).arg( mUser->text() );
  QLdapResponse response = ldap.search( info.baseDn(), QLdap::Sub, filter, QStringList( "dn" ) );
  if ( !response.isValid() ) {
    QMessageBox::critical( this, QString(), tr( "Unable to resolve login to dn: %1" ).arg( ldap.errorString() ) );
    return;
  } else if ( response.entries().isEmpty() ) {
    QMessageBox::critical( this, QString(), tr( "User '%1' is unknown to the system." ).arg( mUser->text() ) );
    return;
  }

  Connection::self()->setName( info.name() );
  Connection::self()->setHost( info.host() );
  Connection::self()->setPort( info.port() );
  Connection::self()->setBaseDn( info.baseDn() );
  Connection::self()->setUser( response.entries().first().dn() );
  Connection::self()->setPassword( mPassword->text() );

  QString errorMsg;

  if ( !Auth::self()->authenticate( response.entries().first().dn(), errorMsg ) ) {
    QMessageBox::critical( 0, QString(), errorMsg );
    return;
  }

  done( Accepted );
}

void ConnectionDialog::configure()
{
  ConfigureDialog dlg( &mConnectionInfoManager, this );

  if ( dlg.exec() ) {
    updateButtonState();
    updateConnectionBox();

    mConnectionInfoManager.save();
  }
}

void ConnectionDialog::updateButtonState()
{
  if ( !mConnectionInfoManager.connectionInfos().isEmpty() ) {
    mOkButton->setEnabled( true );
    mConnection->setEnabled( true );
  } else {
    mOkButton->setEnabled( false );
    mConnection->setEnabled( false );
  }
}

void ConnectionDialog::updateConnectionBox()
{
  const QString currentConnection = mConnection->currentText();
  mConnection->clear();

  const ConnectionInfo::List connections = mConnectionInfoManager.connectionInfos();
  for ( int i = 0; i < connections.count(); ++i )
    mConnection->addItem( connections[ i ].name() );

  int index = mConnection->findText( currentConnection );
  mConnection->setCurrentIndex( index == -1 ? 0 : index );
}

ConnectionInfoDialog::ConnectionInfoDialog( QWidget *parent )
  : QDialog( parent )
{
  QGridLayout *layout = new QGridLayout( this );

  QLabel *label = new QLabel( tr( "Name:" ), this );
  mName = new QLineEdit( this );
  label->setBuddy( mName );

  layout->addWidget( label, 0, 0 );
  layout->addWidget( mName, 0, 1, 1, 2 );

  label = new QLabel( tr( "Host:" ), this );
  mHost = new QLineEdit( this );
  label->setBuddy( mHost );

  layout->addWidget( label, 1, 0 );
  layout->addWidget( mHost, 1, 1, 1, 2 );

  label = new QLabel( tr( "Port:" ), this );
  mPort = new QSpinBox( this );
  mPort->setRange( 1, 65536 );
  mPort->setValue( 389 );
  label->setBuddy( mPort );

  layout->addWidget( label, 2, 0 );
  layout->addWidget( mPort, 2, 1, 1, 2 );

  label = new QLabel( tr( "Base Dn:" ), this );
  mBaseDn = new QLineEdit( this );
  label->setBuddy( mBaseDn );

  layout->addWidget( label, 3, 0 );
  layout->addWidget( mBaseDn, 3, 1, 1, 2 );

  label = new QLabel( tr( "Nobody User:" ), this );
  mUser = new QLineEdit( this );
  label->setBuddy( mUser );

  layout->addWidget( label, 4, 0 );
  layout->addWidget( mUser, 4, 1, 1, 2 );

  label = new QLabel( tr( "Nobody Password:" ), this );
  mPassword = new QLineEdit( this );
  mPassword->setEchoMode( QLineEdit::Password );
  label->setBuddy( mPassword );

  layout->addWidget( label, 5, 0 );
  layout->addWidget( mPassword, 5, 1, 1, 2 );

  QPushButton *button = new QPushButton( tr( "&Ok" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( accept() ) );
  layout->addWidget( button, 6, 1 );

  button = new QPushButton( tr( "&Cancel" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( reject() ) );
  layout->addWidget( button, 6, 2 );
}

void ConnectionInfoDialog::setConnectionInfo( const ConnectionInfo &connectionInfo )
{
  mName->setText( connectionInfo.name() );
  mHost->setText( connectionInfo.host() );
  mPort->setValue( connectionInfo.port() );
  mBaseDn->setText( connectionInfo.baseDn() );
  mUser->setText( connectionInfo.nobodyUser() );
  mPassword->setText( connectionInfo.nobodyPassword() );
}

ConnectionInfo ConnectionInfoDialog::connectionInfo() const
{
  ConnectionInfo connectionInfo( mName->text() );

  connectionInfo.setHost( mHost->text() );
  connectionInfo.setPort( mPort->value() );
  connectionInfo.setBaseDn( mBaseDn->text() );
  connectionInfo.setNobodyUser( mUser->text() );
  connectionInfo.setNobodyPassword( mPassword->text() );

  return connectionInfo;
}

void ConnectionInfoDialog::accept()
{
  if ( mName->text().isEmpty() ||
       mHost->text().isEmpty() ||
       mBaseDn->text().isEmpty() ||
       mUser->text().isEmpty() ||
       mPassword->text().isEmpty() )
  {
    QMessageBox::critical( this, tr( "Invalid Input" ), tr( "You have to fill out all fields." ) );
    return;
  }

  QDialog::accept();
}

ConfigureDialog::ConfigureDialog( ConnectionInfoManager *manager, QWidget *parent )
  : QDialog( parent ), mManager( manager )
{
  QGridLayout *layout = new QGridLayout( this );

  mWidget = new QListWidget( this );
  layout->addWidget( mWidget, 0, 0, 3, 2 );

  QPushButton *button = new QPushButton( tr( "&Add" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( add() ) );
  layout->addWidget( button, 0, 2 );

  button = new QPushButton( tr( "&Edit" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( edit() ) );
  layout->addWidget( button, 1, 2 );

  button = new QPushButton( tr( "&Delete" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( remove() ) );
  layout->addWidget( button, 2, 2 );

  button = new QPushButton( tr( "&Ok" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( accept() ) );
  layout->addWidget( button, 3, 1 );

  button = new QPushButton( tr( "&Cancel" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( reject() ) );
  layout->addWidget( button, 3, 2 );

  const ConnectionInfo::List connections = manager->connectionInfos();
  for ( int i = 0; i < connections.count(); ++i )
    mWidget->addItem( connections[ i ].name() );

  mWidget->setCurrentRow( 0 );
}

void ConfigureDialog::add()
{
  ConnectionInfoDialog dlg( this );
  if ( !dlg.exec() )
    return;

  const ConnectionInfo info = dlg.connectionInfo();
  if ( !info.isValid() )
    return;

  if ( mManager->contains( info.name() ) )
    return;

  mManager->appendConnectionInfo( info );
  mWidget->addItem( info.name() );
}

void ConfigureDialog::edit()
{
  QListWidgetItem *item = mWidget->currentItem();
  if ( !item )
    return;

  ConnectionInfo info = mManager->connectionInfo( item->text() );
  if ( !info.isValid() )
    return;

  ConnectionInfoDialog dlg( this );
  dlg.setConnectionInfo( info );

  if ( !dlg.exec() )
    return;

  const ConnectionInfo newInfo = dlg.connectionInfo();
  mManager->removeConnectionInfo( info.name() );

  if ( mManager->contains( newInfo.name() ) )
    mManager->appendConnectionInfo( info );
  else
    mManager->appendConnectionInfo( newInfo );
}

void ConfigureDialog::remove()
{
  QListWidgetItem *item = mWidget->currentItem();
  if ( !item )
    return;

  mManager->removeConnectionInfo( item->text() );

  delete mWidget->takeItem( mWidget->currentRow() );
}
