/***************************************************************************
                          initialview.cpp  -  description
                             -------------------
    begin                : Sun Jan 12 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "initialview.h"

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qdatetime.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qtoolbutton.h>

#include <kapp.h>
#include <kcombobox.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <klocale.h>

#include "account.h"
#include "kmessdebug.h"


// The constructor
InitialView::InitialView(QWidget *parent, const char *name )
 : InitialViewInterface(parent, name)
{
  // Set the username box with the last used email address
  config_ = kapp->config();
  config_->setGroup( "InitialView" );
  defaultHandle_ = config_->readEntry( "defaultHandle", "me@hotmail.com" );

  // Enable auto completion
  handleCombobox_->setCompletionMode( KGlobalSettings::CompletionPopup );

  // Select an image to load into the main screen.. mmmm, eggs!
  // Applied both in InitialView and AutologinView.
  QString imageName;
  QDate currentDate = QDate::currentDate();
  if( currentDate.month() == 12 && currentDate.day() > 15 )
  {
    imageName = "kmesslogo2";
  }
  else
  {
    imageName = "kmesslogo";
  }

  // Load the chosen icon
  KIconLoader *loader = KGlobal::iconLoader();
  QPixmap icon = loader->loadIcon( imageName, KIcon::User );
  kmessLogo_->setPixmap( icon );
  kmessLogo_->setMinimumSize( icon.width(), icon.height() );

  connectButton_->setIconSet( QIconSet( loader->loadIcon( "connect_creating", KIcon::NoGroup, KIcon::SizeMedium ) ) );

  // Add the items to the initial status combo box
  QStringList statuses;
  statuses << i18n("Online")
           << i18n("Away")
           << i18n("Be Right Back")
           << i18n("Busy")
           << i18n("Out to Lunch")
           << i18n("On the Phone")
           << i18n("Invisible");
  initialStatus_->insertStringList( statuses );
}



// The destructor
InitialView::~InitialView()
{
#ifdef KMESSDEBUG_INITIALVIEW
  kdDebug() << "DESTROYED InitialView" << endl;
#endif
}



// Add an account to the list of displayed accounts from which the user can choose
void InitialView::addAccount(Account *account)
{
  const QString &handle = account->getHandle();

  // Add account to internal list
  accounts_.insert( handle, account );

  // Add an entry for the account to the dropdown list
  handleCombobox_->insertItem( handle ); // + " (" + account->getFriendlyName() + ")" );
  handleCombobox_->completionObject()->addItem( handle );

  // Select the account in the combobox if it's the default account.
  if( handle == defaultHandle_ )
  {
    handleCombobox_->setCurrentItem( defaultHandle_ );
    updateView();
  }
}



// Modify an account in the list of displayed accounts from which the user can choose
void InitialView::changedAccount( QString oldName, QString newName )
{
  if( accounts_[ oldName ] == 0 )
  {
    return;
  }

  Account *account = accounts_[ oldName ];
  accounts_.remove( oldName );
  accounts_.insert( newName, account );

  // Change completion box entry
  handleCombobox_->completionObject()->removeItem( oldName );
  handleCombobox_->completionObject()->addItem( newName );

  // Remove account from qcombobox
  bool foundItem = false;
  for( int i = 0; i < handleCombobox_->count(); i++ )
  {
    if( handleCombobox_->text(i).lower() == oldName.lower() )
    {
      handleCombobox_->removeItem(i);
      foundItem = true;
      break;
    }
  }
  handleCombobox_->insertItem( newName );

  if( ! foundItem )
  {
    kdWarning() << "InitialView::changedAccount() - Account not found in pulldown list!" << endl;
  }
}



// The "connect" button was pressed.
void InitialView::connectToMsn()
{
  QString handle = getSelectedHandle();

  // Avoid connecting with empty fields
  if( handle.isEmpty() || passwordEdit_->text().isEmpty() )
  {
    return;
  }

  // Save the entered text
  config_->setGroup( "InitialView" );
  config_->writeEntry( "defaultHandle", handle );

  // Update the initial status
  QString initialStatus = initialStatus_->currentText();
  if( initialStatus == i18n("Away") )
  { // Change status to AWAY
    initialStatus = "AWY";
  }
  else if( initialStatus == i18n("Be Right Back") )
  { // Change status to BE RIGHT BACK
    initialStatus = "BRB";
  }
  else if( initialStatus == i18n("Busy") )
  { // Change status to BUSY
    initialStatus = "BSY";
  }
  else if( initialStatus == i18n("Invisible") )
  { // Change status to INVISIBLE
    initialStatus = "HDN";
  }
  else if( initialStatus == i18n("Out to Lunch") )
  { // Change status to OUT TO LUNCH
    initialStatus = "LUN";
  }
  else if( initialStatus == i18n("On the Phone") )
  { // Change status to ON THE PHONE
    initialStatus = "PHN";
  }
  else
  { // Change status to ONLINE
    initialStatus = "NLN";
  }

  // Inform parent with the account information
  emit connectWithAccount( handle, passwordEdit_->text(), rememberCheckBox_->isChecked(), initialStatus );
}



// The account was deleted
void InitialView::deleteAccount(Account *account)
{
  // Remove account from hashmap.
  QString handle = account->getHandle();
  accounts_.remove(handle);

  // Remove account from completionbox (extended qlistbox)
  handleCombobox_->completionObject()->removeItem( handle );

  // Remove account from qcombobox
  bool foundItem = false;
  for( int i = 0; i < handleCombobox_->count(); i++ )
  {
    if( handleCombobox_->text(i).lower() == handle.lower() )
    {
      handleCombobox_->removeItem(i);
      foundItem = true;
      break;
    }
  }

  if( ! foundItem )
  {
    kdWarning() << "InitialView::deleteAccount: Account not found in pulldown list!" << endl;
  }
}



// Get the currently selected handle
QString InitialView::getSelectedHandle() const
{
  // Get handle, remove appended friendlyname if present
  QString handle = handleCombobox_->currentText();
  if( handle.contains(' ') )
  {
    handle = handle.section(' ', 0, 0);
  }
  return handle.lower();
}



// A profile was selected from the drop-down list, or written manually.
void InitialView::updateView()
{
  Account *account = accounts_[ getSelectedHandle() ];
  if( account != 0 )
  {
    // User typed the full account name without using autocompletion.
    passwordEdit_->setText( account->getPassword() );
    rememberCheckBox_->setChecked( ! account->isGuestAccount() );

    // Make sure the drop down list matches the user's initial status
    int item;
    QString status = account->getInitialStatus();
    if ( status == "AWY" )
    { // Change status to AWAY
      item = 1;
    }
    else if( status == "BRB" )
    { // Change status to BE RIGHT BACK
      item = 2;
    }
    else if( status == "BSY" )
    { // Change status to BUSY
      item = 3;
    }
    else if( status == "LUN" )
    { // Change status to OUT TO LUNCH
      item = 4;
    }
    else if( status == "PHN" )
    { // Change status to ON THE PHONE
      item = 5;
    }
    else if( status == "HDN" )
    { // Change status to INVISIBLE
      item = 6;
    }
    else
    { // Change status to ONLINE
      item = 0;
    }
    initialStatus_->setCurrentItem( item );
  }
  else
  {
    // User typed a new account name
    // Clear password again
    passwordEdit_->setText( QString::null );
    rememberCheckBox_->setChecked( false );
    initialStatus_->setCurrentItem( 0 );
  }

  // Put the attention back to the email field
  handleCombobox_->setFocus();
}


#include "initialview.moc"
