/**************************************************************************/
/*  Klavaro - a flexible touch typing tutor                               */
/*  Copyright (C) 2005 - 2008  Felipe Castro                              */
/*                                                                        */
/*  This program is free software, licensed under the terms of the GNU    */
/*  General Public License as published by the Free Software Foundation,  */
/*  which is also included in this package, in a file named "COPYING".    */
/**************************************************************************/

/*
 * Shared tutor window tasks
 */
#include <math.h>
#include <string.h>
#include <time.h>
#include <stdio.h>
#include <locale.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>

#include "support.h"
#include "interface.h"
#include "main.h"
#include "translation.h"
#include "keyboard.h"
#include "cursor.h"
#include "basic.h"
#include "adaptability.h"
#include "velocity.h"
#include "fluidness.h"
#include "tutor.h"

/*
 * Variables
 */
extern GtkWidget *window_main_;
extern GtkWidget *popup_other_;
extern gboolean callbacks_shield;

GtkWidget *window_tutor_ = NULL;

#define MAX_TOUCH_TICS 8000
struct
{
	TutorType type;
	TutorQuery query;
	GTimer *tmr;
	gdouble elapsed_time;
	gdouble touch_time[MAX_TOUCH_TICS + 1];
	guint ttidx;
	gint n_touchs;
	gint n_errors;
	gint retro_pos;
	gint correcting;
} tutor;

/*******************************************************************************
 * Interface functions
 */
TutorType
tutor_get_type ()
{
	return (tutor.type);
}

gchar *
tutor_get_type_name ()
{
	static gchar type_name[4][6] = { "basic", "adapt", "velo", "fluid" };

	return (type_name[tutor.type]);
}

TutorQuery
tutor_get_query ()
{
	return (tutor.query);
}

void
tutor_set_query (TutorQuery query)
{
	tutor.query = query;
}

void
tutor_init_timers ()
{
	tutor.tmr = g_timer_new ();
}

/**********************************************************************
 * Initialize the course 
 */
void
tutor_init (TutorType tt_type)
{
	gchar *tmp_title = NULL;
	gchar *tmp_name = NULL;
	GtkWidget *wg;

	if (window_tutor_ == NULL)
		window_tutor_ = create_window_tutor ();

	gtk_widget_show (window_tutor_);
	gtk_widget_hide (window_main_);
	gtk_widget_grab_focus (lookup_widget (window_tutor_, "entry_mesg"));

	tutor.type = tt_type;
	cursor_set_blink (FALSE);
	cursor_set_is_initializing (FALSE);


  /******************************
   * Set the layout for each exercise type
   */
	if (tutor.type == TT_BASIC || tutor.type == TT_FLUID)
	{
		gtk_widget_show (lookup_widget (window_tutor_, "frame_lesson"));
		gtk_widget_show (lookup_widget (window_tutor_, "vseparator_tutor"));
		if (tutor.type == TT_BASIC)
		{
			gtk_widget_show (lookup_widget (window_tutor_, "button_revert_lesson"));
			gtk_widget_show (lookup_widget (window_tutor_, "button_reset_lesson"));
			gtk_widget_show (lookup_widget (window_tutor_, "button_tutor_show_keyb"));
			gtk_label_set_text (GTK_LABEL (lookup_widget (window_tutor_, "label_lesson")), _("Lesson:"));
			gtk_spin_button_set_range (GTK_SPIN_BUTTON
						   (lookup_widget (window_tutor_, "spinbutton_lesson")), 1, 50);
		}
		else
		{
			gtk_widget_hide (lookup_widget (window_tutor_, "button_revert_lesson"));
			gtk_widget_hide (lookup_widget (window_tutor_, "button_reset_lesson"));
			gtk_widget_hide (lookup_widget (window_tutor_, "button_tutor_show_keyb"));
			gtk_label_set_text (GTK_LABEL
					    (lookup_widget (window_tutor_, "label_lesson")), _("Paragraphs:"));
			callbacks_shield = TRUE;
			gtk_spin_button_set_range (GTK_SPIN_BUTTON
						   (lookup_widget (window_tutor_, "spinbutton_lesson")), 1, 10);
			callbacks_shield = FALSE;
		}
	}
	else
	{
		gtk_widget_hide (lookup_widget (window_tutor_, "frame_lesson"));
		gtk_widget_hide (lookup_widget (window_tutor_, "vseparator_tutor"));
		gtk_widget_hide (lookup_widget (window_tutor_, "button_tutor_show_keyb"));
		basic_set_lesson (0);
	}

	wg = lookup_widget (window_tutor_, "button_tutor_other");
	if (tutor.type == TT_BASIC || tutor.type == TT_ADAPT)
		gtk_widget_hide (wg);
	else
		gtk_widget_show (wg);

  /******************************
   * Set decoration texts
   */
	switch (tutor.type)
	{
	case TT_BASIC:
		tmp_title = g_strdup (_("Klavaro - Basic Course"));
		tmp_name = g_strdup ("");
		break;

	case TT_ADAPT:
		tmp_title = g_strdup (_("Klavaro - Adaptability"));
		tmp_name =
			g_strdup (_
				  ("Adaptability exercises: automating the fingers"
				   " responses, typing over all the keyboard."));
		break;

	case TT_VELO:
		tmp_title = g_strdup (_("Klavaro - Velocity"));
		tmp_name = g_strdup (_("Velocity exercises: accelerate typing real words."));
		break;

	case TT_FLUID:
		tmp_title = g_strdup (_("Klavaro - Fluidness"));
		tmp_name = g_strdup (_("Fluidness exercises: accuracy typing good sense paragraphs."));
		break;
	}
	gtk_window_set_title (GTK_WINDOW (window_tutor_), tmp_title);
	wg = lookup_widget (window_tutor_, "label_heading");
	gtk_label_set_text (GTK_LABEL (wg), tmp_name);
	g_free (tmp_title);
	g_free (tmp_name);

  /******************************
   * Set specific variables
   */
	if (tutor.type == TT_BASIC)
	{
		basic_init ();
		return;
	}

	else if (tutor.type == TT_VELO)
		velo_init ();

	else if (tutor.type == TT_FLUID)
		fluid_init ();

	tutor.query = QUERY_INTRO;
	tutor_update ();
}


/**********************************************************************
 * Update what is shown in the tutor window. 
 */
void
tutor_update ()
{
	switch (tutor.query)
	{
	case QUERY_INTRO:
		tutor_update_intro ();
		break;

	case QUERY_START:
		tutor_update_start ();
		break;

	case QUERY_PROCESS_TOUCHS:
		tutor_message (_("STARTED! => "));
		gtk_widget_show (lookup_widget (window_tutor_, "button_tutor_restart"));
		gtk_widget_set_sensitive (lookup_widget (window_tutor_, "button_tutor_font"), TRUE);
		break;

	case QUERY_END:
		if (tutor.type == TT_BASIC)
			tutor_message (_("<<==  End of lesson." " Press [Enter] to start another. ==>> "));
		else
			tutor_message (_("<<==  End of exercise." " Press [Enter] to start another. ==>> "));
		gtk_widget_hide (lookup_widget (window_tutor_, "button_tutor_restart"));
		gtk_widget_set_sensitive (lookup_widget (window_tutor_, "button_tutor_font"), FALSE);
		break;
	}
}

void
tutor_update_intro ()
{
	gchar *tmp_name;
	gchar text[4096];
	GtkWidget *wg;
	GtkLabel *wg_label;
	GtkTextView *wg_text;
	GtkAdjustment *scroll;

	if (tutor.type == TT_BASIC)
	{
		callbacks_shield = TRUE;
		gtk_spin_button_set_value (GTK_SPIN_BUTTON
					   (lookup_widget (window_tutor_, "spinbutton_lesson")), basic_get_lesson ());
		callbacks_shield = FALSE;

		wg_label = GTK_LABEL (lookup_widget (window_tutor_, "label_heading"));
		gtk_label_set_text (wg_label, _("Learning the key positions."));
		tutor_message (_("Press any key to start the lesson. "));
	}
	else
		tutor_message (_("Press any key to start the exercise. "));

	gtk_widget_hide (lookup_widget (window_tutor_, "button_tutor_restart"));
	gtk_widget_set_sensitive (lookup_widget (window_tutor_, "button_tutor_font"), FALSE);

	tmp_name = g_strconcat ("_", tutor_get_type_name (), "_intro.txt", NULL);
	trans_read_text (text, 4096, tmp_name);
	g_free (tmp_name);
	wg_text = GTK_TEXT_VIEW (lookup_widget (window_tutor_, "text_tutor"));
	gtk_text_buffer_set_text (gtk_text_view_get_buffer (wg_text), text, -1);

	callbacks_shield = TRUE;
	gtk_entry_set_text (GTK_ENTRY (lookup_widget (window_tutor_, "entry_mesg")), "");
	callbacks_shield = FALSE;

	wg = lookup_widget (window_tutor_, "scrolledwindow_tutor_main");
	scroll = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (wg));
	gtk_adjustment_set_value (scroll, 0);
}

void
tutor_update_start ()
{
	gchar *tmp_name;
	gchar *text;
	gint level;
	GtkWidget *wg;
	GtkTextBuffer *buf;
	GtkTextIter start;
	GtkTextIter end;
	GtkAdjustment *scroll;

	/*
	 * Delete all the text on tutor window
	 */
	wg = lookup_widget (window_tutor_, "text_tutor");
	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));
	gtk_text_buffer_set_text (buf, "", -1);

	if (tutor.type == TT_BASIC)
	{
		callbacks_shield = TRUE;
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (lookup_widget
							    (window_tutor_, "spinbutton_lesson")), basic_get_lesson ());
		callbacks_shield = FALSE;

		if (main_preferences_exist ("tutor", "basic_lesson"))
		{
			level = main_preferences_get_int ("tutor", "basic_lesson");
			wg = lookup_widget (window_tutor_, "button_revert_lesson");
			if (level == basic_get_lesson ())
				gtk_widget_hide (wg);
			else
				gtk_widget_show (wg);

			wg = lookup_widget (window_tutor_, "button_reset_lesson");
			if (level == 1)
				gtk_widget_hide (wg);
			else
				gtk_widget_show (wg);
		}
		else
		{
			wg = lookup_widget (window_tutor_, "button_revert_lesson");
			gtk_widget_show (wg);
		}

		tmp_name = g_ucs4_to_utf8 (basic_get_char_set (), -1, NULL, NULL, NULL);
		text = g_strdup_printf (_("Keys: %s"), tmp_name);
		g_free (tmp_name);
		wg = lookup_widget (window_tutor_, "label_heading");
		gtk_label_set_text (GTK_LABEL (wg), text);
		g_free (text);
		basic_draw_lesson ();
	}

	switch (tutor.type)
	{
	case TT_BASIC:
		break;
	case TT_ADAPT:
		adapt_draw_random_pattern ();
		break;
	case TT_VELO:
		velo_draw_random_words ();
		break;
	case TT_FLUID:
		fluid_draw_random_paragraphs ();
	}

	/*
	 * Apply tutor background color and font to the text
	 */
	gtk_text_buffer_get_bounds (buf, &start, &end);
	gtk_text_iter_backward_char (&end);
	gtk_text_buffer_apply_tag_by_name (buf, "char_untouched", &start, &end);
	gtk_text_buffer_apply_tag_by_name (buf, "lesson_font", &start, &end);

	gtk_widget_hide (lookup_widget (window_tutor_, "button_tutor_restart"));
	gtk_widget_set_sensitive (lookup_widget (window_tutor_, "button_tutor_font"), TRUE);
	tutor_message (_("Start typing when you are ready. "));
	callbacks_shield = TRUE;
	gtk_entry_set_text (GTK_ENTRY (lookup_widget (window_tutor_, "entry_mesg")), "");
	callbacks_shield = FALSE;

	wg = lookup_widget (window_tutor_, "scrolledwindow_tutor_main");
	scroll = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (wg));
	gtk_adjustment_set_value (scroll, 0);
}


/**********************************************************************
 * Respond to each touch of the user, according to the tutor.query mode
 */
void
tutor_process_touch (gunichar user_chr)
{
	gboolean go_on;
	GtkTextView *wg_text;
	GtkTextBuffer *wg_buffer;
	GtkTextIter start;

	wg_text = GTK_TEXT_VIEW (lookup_widget (window_tutor_, "text_tutor"));
	wg_buffer = gtk_text_view_get_buffer (wg_text);

	switch (tutor.query)
	{
	case QUERY_PROCESS_TOUCHS:
		break;

	case QUERY_INTRO:
		tutor.query = QUERY_START;
		tutor_update ();
		tutor.n_touchs = 0;
		tutor.n_errors = 0;
		tutor.retro_pos = 0;
		tutor.correcting = 0;
		tutor.ttidx = 0;
		gtk_text_buffer_get_start_iter (wg_buffer, &start);
		gtk_text_buffer_place_cursor (wg_buffer, &start);
		if (cursor_get_is_initializing () == FALSE)
		{
			cursor_set_blink (FALSE);
			cursor_set_is_initializing (TRUE);
			g_timeout_add (TIME_CURSOR_ON + TIME_CURSOR_OFF + 30, &cursor_init, NULL);
		}
		if (tutor.type == TT_BASIC)
			hints_update_from_char (cursor_get_char ());
		return;

	case QUERY_START:
		tutor.query = QUERY_PROCESS_TOUCHS;
		tutor_update ();

		if (tutor.type == TT_BASIC)
			hints_update_from_char (cursor_get_char ());

		g_timer_start (tutor.tmr);
		if (tutor.type == TT_FLUID)
			tutor_eval_forward_backward (user_chr);
		else
			tutor_eval_forward (user_chr);
		return;

	case QUERY_END:
		if (user_chr == UPSYM)
		{
			basic_set_lesson_increased (FALSE);
			tutor.query = QUERY_INTRO;
			tutor_process_touch (L'\0');
		}
		else
		{
			tutor_beep ();
			tutor_update ();
		}
		return;
	}

	/*
	 * It is time to analise the correctness of typing.
	 */
	if (tutor.type == TT_FLUID)
		go_on = tutor_eval_forward_backward (user_chr);
	else
		go_on = tutor_eval_forward (user_chr);

	if (go_on == FALSE)
	{
		cursor_set_blink (FALSE);
		tutor.elapsed_time = g_timer_elapsed (tutor.tmr, NULL);

		tutor_calc_stats ();
		tutor.query = QUERY_END;
		tutor_update ();
		tutor_beep ();
	}
	else if (tutor.type == TT_BASIC)
		hints_update_from_char (cursor_get_char ());
}

/**********************************************************************
 * Advances the cursor one position and test for correctness,
 * in the shared tutor window.
 * Updates the variables:
 * cursor_pos, n_touchs and n_errors.
 */
gboolean
tutor_eval_forward (gunichar user_chr)
{
	gunichar real_chr;

	if (user_chr == L'\b')
	{
		tutor_beep ();
		return (TRUE);
	}

	tutor.n_touchs++;

	real_chr = cursor_get_char ();

	/*
	 * Compare the user char with the real char and set the color
	 */
	if (user_chr == real_chr)
		cursor_paint_char ("char_correct");
	else
	{
		tutor_beep ();
		cursor_paint_char ("char_wrong");
		tutor.n_errors++;
	}


	/*
	 * Go forwa and test end of text
	 */
	if (cursor_advance (1) != 1)
		return (FALSE);

	/*
	 * Test end of text
	 */
	if (cursor_get_char () == L'\n')
		if (cursor_advance (1) != 1)
			return (FALSE);

	return (TRUE);
}

/**********************************************************************
 * Like the previous, but allows to go back and forth.
 */
gboolean
tutor_eval_forward_backward (gunichar user_chr)
{
	gunichar real_chr;

	/*
	 * Work on backspaces
	 * L'\t' means a super <Ctrl> + <Backspace>
	 */
	if (user_chr == L'\b' || user_chr == L'\t')
	{
		tutor.touch_time[tutor.ttidx] = g_timer_elapsed (tutor.tmr, NULL);

		/*
		 * Test for end of errors to be corrected
		 */
		if (tutor.retro_pos == 0)
		{
			tutor_beep ();
			return (TRUE);
		}

		/*
		 * Go backwards and test for begin of the text
		 */
		if (cursor_advance (-1) != -1)
		{
			tutor_beep ();
			return (TRUE);
		}

		/*
		 * Test begin of line
		 */
		if (cursor_get_char () == L'\n')
			if (cursor_advance (-1) != -1)
			{
				tutor_beep ();
				return (TRUE);
			}

		/*
		 * Reinitialize the color
		 */
		cursor_paint_char ("char_untouched");

		/*
		 * Update state
		 */
		tutor.retro_pos--;
		tutor.correcting++;

		if (user_chr == L'\t')
			tutor_eval_forward_backward (L'\t');
		return (TRUE);
	}

	real_chr = cursor_get_char ();

	/*
	 * Compare the user char with the real char and set the color
	 */
	if (user_chr == real_chr && tutor.retro_pos == 0)
	{
		tutor.n_touchs++;
		if (tutor.ttidx < MAX_TOUCH_TICS)
		{
			tutor.touch_time[tutor.ttidx] =
				g_timer_elapsed (tutor.tmr, NULL) - tutor.touch_time[tutor.ttidx];
			tutor.ttidx++;
			tutor.touch_time[tutor.ttidx] = g_timer_elapsed (tutor.tmr, NULL);
		}
		if (tutor.correcting != 0)
		{
			cursor_paint_char ("char_retouched");
			tutor.n_errors++;
		}
		else
			cursor_paint_char ("char_correct");
	}
	else
	{
		tutor.touch_time[tutor.ttidx] = g_timer_elapsed (tutor.tmr, NULL);
		cursor_paint_char ("char_wrong");
		tutor_beep ();
		tutor.retro_pos++;
	}

	if (tutor.correcting > 0)
		tutor.correcting--;

	/*
	 * Go forward and test end of text
	 */
	if (cursor_advance (1) != 1)
	{
		if (tutor.retro_pos == 0)
			return (FALSE);
		tutor.retro_pos--;
	}

	/*
	 * Test end of paragraph
	 */

	if (cursor_get_char () == L'\n')
		if (cursor_advance (1) != 1 && tutor.retro_pos == 0)
			return (FALSE);

	return (TRUE);
}


/**********************************************************************
 * Calculate the final results
 */
void
tutor_calc_stats ()
{
	gint i;
	gint minutes;
	gint seconds;
	gdouble accuracy;
	gdouble touchs_per_second;
	gdouble velocity;
	gdouble fluidness;
	gdouble standard_deviation = 0;
	gdouble sum;
	gdouble average = 0;
	gdouble sample;
	gchar *tmp_locale;
	gchar *tmp_str = NULL;
	gchar *tmp_str2;
	gchar *tmp_name;
	FILE *fh;
	time_t tmp_time;
	struct tm *ltime;
	GtkWidget *wg;
	GtkTextBuffer *buf;

	/*
	 * Calculate statistics
	 */
	minutes = ((gulong) tutor.elapsed_time) / 60;
	seconds = ((gulong) tutor.elapsed_time) % 60;

	accuracy = 100 * (1.0 - (gfloat) tutor.n_errors / tutor.n_touchs);
	touchs_per_second = (tutor.n_touchs - tutor.n_errors) / tutor.elapsed_time;
	velocity = 10 * touchs_per_second;

	if (tutor.type == TT_FLUID)
	{
		/*
		 * "Magic" fluidness calculation
		 */
		sum = 0;
		for (i = 2; i < tutor.ttidx; i++)
		{
			if (tutor.touch_time[i] <= 0)
				tutor.touch_time[i] = 1.0e-8;
			sample = sqrt (1 / tutor.touch_time[i]);
			sum += sample;
		}
		if (i == 2)
			i++;
		average = sum / (i - 2);

		sum = 0;
		for (i = 2; i < tutor.ttidx; i++)
		{
			sample = sqrt (1 / tutor.touch_time[i]);
			sum += (sample - average) * (sample - average);
		}
		if (i < 4)
			i = 4;
		standard_deviation = sqrt (sum / (i - 3));

		if (average <= 0)
			average = 1.0e-9;
		fluidness = 100 * (1 - standard_deviation / average);
		if (fluidness < 0)
			fluidness = 0;
	}
	else
		fluidness = 0;

	/*
	 * Changing to "C" locale: remember to copy the previous value!
	 */
	tmp_locale = g_strdup (setlocale (LC_NUMERIC, NULL));
	if (tmp_locale != NULL)
		setlocale (LC_NUMERIC, "C");

	/*
	 * Logging
	 */
	tmp_name = g_strconcat (main_get_user_dir (), "stat_", tutor_get_type_name (), ".txt", NULL);
	assert_user_dir ();
	if (!(fh = (FILE *) g_fopen (tmp_name, "r")))
	{
		fh = (FILE *) g_fopen (tmp_name, "w");
		fprintf (fh, "Accuracy\tVelocity\tFluidness\tDate\tHour\tLesson\n");
	}
	else
	{
		fclose (fh);
		fh = (FILE *) g_fopen (tmp_name, "a");
	}
	if (fh)
	{
		tmp_time = time (NULL);
		ltime = localtime (&tmp_time);
		fprintf (fh,
			 "%.2f\t%.2f\t%.2f\t%i-%2.2i-%2.2i\t%2.2i:%2.2i\t",
			 accuracy, velocity, fluidness,
			 (ltime->tm_year) + 1900, (ltime->tm_mon) + 1,
			 (ltime->tm_mday), (ltime->tm_hour), (ltime->tm_min));
		switch (tutor.type)
		{
		case TT_BASIC:
			fprintf (fh, "%2.2i\n", basic_get_lesson ());
			break;
		case TT_ADAPT:
			fprintf (fh, "%s\n", _("Default"));
			break;
		case TT_VELO:
			fprintf (fh, "%s\n", velo_get_dict_name ());
			break;
		case TT_FLUID:
			fprintf (fh, "%s\n", fluid_get_paragraph_name ());
		}
		fclose (fh);
	}
	else
		g_message ("not able to log on this file:\n %s", tmp_name);
	g_free (tmp_name);

	if (tutor.type == TT_FLUID)
	{
		/*
		 * Logging the fluidness results of the last session
		 */
		tmp_name = g_strconcat (main_get_user_dir (), "deviation_fluid.txt", NULL);
		if ((fh = (FILE *) g_fopen (tmp_name, "w")))
		{
			g_message ("writing further fluidness results at:\n %s", tmp_name);
			fprintf (fh,
				 "(i)\tdt(i)\tsqrt(1/dt(i))\tAverage:\t%g\tStd. dev.:\t%g\n",
				 average, standard_deviation);
			for (i = 1; i < tutor.ttidx; i++)
				fprintf (fh, "%i\t%g\t%g\n", i, tutor.touch_time[i],
					 sqrt (1 / (tutor.touch_time[i] > 0 ? tutor.touch_time[i] : 1.0e-9)));
			fclose (fh);
		}
		else
			g_message ("not able to log on this file:\n %s", tmp_name);
		g_free (tmp_name);
	}

	/*
	 * Coming back to the right locale
	 */
	if (tmp_locale != NULL)
		setlocale (LC_NUMERIC, tmp_locale);
	g_free (tmp_locale);

	/*
	 * Print statistics
	 */
	if (tutor.type == TT_BASIC || tutor.type == TT_ADAPT)
	{
		tmp_str = g_strconcat ("\n", _("STATISTICS"), "\n",
				       _("Elapsed time:"), " %i ",
				       dngettext (PACKAGE, "minute and",
						  "minutes and", minutes),
				       " %i ", dngettext (PACKAGE, "second",
							  "seconds", seconds),
				       "\n", _("Accuracy:"), " %.1f %%\t\t",
				       _("Goal:"), " %i %%\n\n", _("Comments:"), "\n", NULL);
	}

	wg = lookup_widget (window_tutor_, "text_tutor");
	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));

	switch (tutor.type)
	{
	case TT_BASIC:
		tmp_str2 = g_strdup_printf (tmp_str, minutes, seconds, accuracy, 95);
		gtk_text_buffer_insert_at_cursor (buf, tmp_str2, strlen (tmp_str2));
		basic_comment (accuracy);
		break;
	case TT_ADAPT:
		tmp_str2 = g_strdup_printf (tmp_str, minutes, seconds, accuracy, 99);
		gtk_text_buffer_insert_at_cursor (buf, tmp_str2, strlen (tmp_str2));
		adapt_comment (accuracy);
		break;
	case TT_VELO:
		tmp_str = g_strconcat ("\n", _("STATISTICS"), "\n",
				       _("Elapsed time:"), " %i ",
				       dngettext (PACKAGE, "minute and",
						  "minutes and", minutes),
				       " %i ", dngettext (PACKAGE, "second",
							  "seconds", seconds),
				       "\n", _("Accuracy:"), " %.1f %%\t\t",
				       _("Goal:"), " %i %%", "\n",
				       _("Characters per second:"),
				       " %.2f\t\t", _("Goal:"), " %i ",
				       _("(CPS)"), "\n",
				       _("Words per minute:"), " %.1f\t\t",
				       _("Goal:"), " %i ", _("(WPM)"), "\n\n", _("Comments:"), "\n", NULL);
		tmp_str2 = g_strdup_printf (tmp_str, minutes, seconds, accuracy, 97, velocity / 10, 5, velocity, 50);
		gtk_text_buffer_insert_at_cursor (buf, tmp_str2, strlen (tmp_str2));
		velo_comment (accuracy, velocity);
		break;
	case TT_FLUID:
		tmp_str = g_strconcat ("\n", _("STATISTICS"), "\n",
				       _("Elapsed time:"), " %i ",
				       dngettext (PACKAGE, "minute and",
						  "minutes and", minutes),
				       " %i ", dngettext (PACKAGE, "second",
							  "seconds", seconds),
				       "\n", _("Accuracy:"), " %.1f %%\t\t",
				       _("Goal:"), " %i %%", "\n",
				       _("Characters per second:"),
				       " %.2f\t\t", _("Goal:"), " %i ",
				       _("(CPS)"), "\n",
				       _("Words per minute:"), " %.1f\t\t",
				       _("Goal:"), " %i ", _("(WPM)"), "\n",
				       _("Fluidness:"), " %.1f %%\t\t",
				       _("Goal:"), " %i %%\n\n", _("Comments:"), "\n", NULL);
		tmp_str2 =
			g_strdup_printf (tmp_str, minutes, seconds, accuracy, 98, velocity / 10, 5, velocity, 50,
					 fluidness, 90);
		gtk_text_buffer_insert_at_cursor (buf, tmp_str2, strlen (tmp_str2));
		fluid_comment (accuracy, velocity, fluidness);
	}
	g_free (tmp_str);
	g_free (tmp_str2);

	gtk_text_view_scroll_mark_onscreen (GTK_TEXT_VIEW (wg), gtk_text_buffer_get_insert (buf));
}

/**********************************************************************
 * Formats and draws one paragraph at the tutor window
 */
void
tutor_draw_paragraph (gchar * utf8_text)
{
	static gchar *tmp1;
	static gchar *tmp2 = NULL;
	gchar *ptr;
	GtkWidget *wg;
	GtkTextBuffer *buf;

	g_free (tmp1);
	g_free (tmp2);

	if (g_utf8_strrchr (utf8_text, -1, L'\n') == NULL)
	{
		g_message ("paragraph not terminated by carriage return: adding one.");
		tmp1 = g_strconcat (utf8_text, "\n", NULL);
	}
	else
		tmp1 = g_strdup (utf8_text);

	ptr = g_utf8_strrchr (tmp1, -1, L'\n');
	if (ptr)
		*ptr = '\0';
	else
		g_error ("draw_paragraph () ==> string error");

	wg = lookup_widget (window_tutor_, "text_tutor");
	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));

	tmp2 = g_strconcat (tmp1, keyb_get_utf8_paragraph_symbol (), "\n", NULL);

	gtk_text_buffer_insert_at_cursor (buf, tmp2, -1);
}

/**********************************************************************
 * Put 'mesg' in the message entry line of the shared tutor window
 */
void
tutor_message (gchar * mesg)
{
	GtkWidget *wg;

	if (mesg == NULL)
	{
		g_message ("tutor_message() --> can't show NULL message!");
		return;
	}

	wg = lookup_widget (window_tutor_, "label_tutor_mesg");
	gtk_label_set_text (GTK_LABEL (wg), mesg);
}

/**********************************************************************
 * Beeps (or not) at the user, in the tutor window
 */
void
tutor_beep ()
{
	GtkWidget *wg;

	wg = lookup_widget (window_tutor_, "checkbutton_beep");
	if ((GTK_TOGGLE_BUTTON (wg))->active)
		gdk_beep ();
}

/**********************************************************************
 * Load the list of files to include in the set of "other exercises"
 */
void
tutor_load_list_other (gchar * file_name_end, GtkListStore * list)
{
	gchar *tmp_str;
	gchar *dentry;
	GDir *dir;
	GtkTreeIter iter;

	assert_user_dir ();
	dir = g_dir_open (main_get_user_dir (), 0, NULL);
	while ((dentry = g_strdup (g_dir_read_name (dir))) != NULL)
	{
		if (strlen (dentry) < 5)
		{
			g_free (dentry);
			continue;
		}
		if (!(tmp_str = strrchr (dentry, '.')))
		{
			g_free (dentry);
			continue;
		}
		if (strcmp (file_name_end, tmp_str))
		{
			g_free (dentry);
			continue;
		}

		*(strrchr (dentry, '.')) = '\0';
		gtk_list_store_append (list, &iter);
		gtk_list_store_set (list, &iter, 0, dentry, -1);
		g_free (dentry);
	}
	g_dir_close (dir);

	gtk_list_store_append (list, &iter);
	gtk_list_store_set (list, &iter, 0, _("--> Default"), -1);
}
