#include <string.h>

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <glib.h>

#include "../../gui/kpmainwindow.h"

#include "../../kipina-i18n.h"
#include "../../kptraininglog.h"
#include "../../kpcalendartime.h"
#include "../../kpworkout.h"
#include "../../kppresetdata.h"
#include "../../kputil.h"

KPTrainingLog *tlog;
GString *exercises_file;
GString *sports_file;
GSList *exercises;
GHashTable *sports;
GHashTable *sub_sports;

static gboolean   read_sports (void);
static gboolean   read_exercises (void);
static gboolean   read_sport_type (xmlNodePtr node);
static gboolean   read_exercise (xmlNodePtr node);
static void       add_sport_id (guint id, gboolean sub, const gchar *sport);
G_CONST_RETURN
gchar            *get_sport_for_id (guint id, gboolean sub);

  
void
exercise_reader_init (const gchar *fe, const gchar *fs)
{
  exercises_file = g_string_new (fe);
  sports_file = g_string_new (fs);
  sports = g_hash_table_new_full (g_direct_hash, g_direct_equal, NULL, g_free);
  sub_sports = g_hash_table_new_full (g_direct_hash, g_direct_equal, NULL, g_free);

  tlog = kp_main_window_get_log ();
}


void
exercise_reader_deinit (void)
{
  g_string_free (exercises_file, TRUE);
  g_string_free (sports_file, TRUE);
}


void
exercise_reader_set_filenames (const gchar *fe, const gchar *fs)
{
  g_string_assign (exercises_file, fe);
  g_string_assign (sports_file, fs);
}

gboolean
exercise_reader_read_data (void)
{
  kp_debug ("Reading sports..");
  if (!read_sports ())
    return FALSE;

  kp_debug ("Reading exercises..");
  if (!read_exercises ())
    return FALSE;

  return TRUE;
}

static gboolean
read_exercises (void)
{
  xmlDocPtr doc;
  xmlNodePtr node;

  doc = xmlParseFile (exercises_file->str);
  if (!doc)
    return FALSE;

  node = xmlDocGetRootElement (doc);
  if (!node || !KP_TAG_MATCH (node, "exercise-list")) {
    xmlFreeDoc (doc);
    return FALSE;
  }

  for (node = node->children; node; node = node->next)
    if (KP_TAG_MATCH (node, "exercise")) {
      if (!read_exercise (node)) {
        xmlFreeDoc (doc);
        return FALSE;
      }
    }
  
  xmlFreeDoc (doc);

  return TRUE;
}

static gboolean
read_sports (void)
{
  xmlDocPtr doc;
  xmlNodePtr node;

  doc = xmlParseFile (sports_file->str);
  if (!doc)
    return FALSE;

  node = xmlDocGetRootElement (doc);
  if (!node || !KP_TAG_MATCH (node, "sport-type-list")) {
    xmlFreeDoc (doc);
    return FALSE;
  }

  for (node = node->children; node; node = node->next)
    if (KP_TAG_MATCH (node, "sport-type")) {
      if (!read_sport_type (node)) {
        xmlFreeDoc (doc);
        return FALSE;
      }
    }
  
  xmlFreeDoc (doc);

  return TRUE;
}

  
static gchar *
rgb_to_hex_str (xmlNodePtr node)
{
  xmlChar *tr, *tg, *tb;
  gchar *color;
  guint r, g, b;
  
  tr = xmlGetProp (node, BAD_CAST ("red"));
  tg = xmlGetProp (node, BAD_CAST ("green"));
  tb = xmlGetProp (node, BAD_CAST ("blue"));
      
  r = (guint) strtod ((gchar *) tr, NULL);
  g = (guint) strtod ((gchar *) tg, NULL);
  b = (guint) strtod ((gchar *) tb, NULL);

  color = g_strdup_printf ("#%.2x%.2x%.2x", r, g, b);
  
  return color;
}


static gboolean
read_sport_subtype (xmlNodePtr node, KPPresetDataItem *sport)
{
  KPPresetDataItem *sub_sport;
  gdouble val;
  xmlChar *tmp;
  guint id;

  kp_debug ("Reading sport subtype..");

  id = 0;
  sub_sport = kp_preset_data_item_new ();
  
  for (node = node->children; node; node = node->next) {
    tmp = xmlNodeGetContent (node);
    
    if (KP_TAG_MATCH (node, "name")) {
      sub_sport->name = g_strconcat (sport->name, " - ", tmp, NULL);
    }
    if (KP_TAG_MATCH (node, "id")) {
      val = kp_number ((gchar *)tmp);
      if (val >= 0)
        id = val;
    }

    g_free (tmp);
  }

  g_return_val_if_fail (id != 0, FALSE);
  
  sub_sport->abbreviation = g_strdup (sport->abbreviation);
  sub_sport->description = g_strdup (sport->description);
  sub_sport->data = g_strdup (sport->data);

  kp_preset_data_add_item (KP_PRESET_DATA_SPORT, sub_sport);
  add_sport_id (id, TRUE, sport->name);

  return TRUE;
}

static gboolean
read_sport_type (xmlNodePtr node)
{
  KPPresetDataItem *sport;
  xmlNodePtr child;
  xmlNodePtr parent;
  gchar *t;
  gdouble id = 0;

  sport = kp_preset_data_item_new (); 
  
  parent = node;
  for (node = node->children; node; node = node->next) {
    if (KP_TAG_MATCH (node, "id")) { 
      t = (gchar *) xmlNodeGetContent (node);
      id = kp_number (t);
      g_free (t);
    }

    if (KP_TAG_MATCH (node, "name")) {
      sport->name = (gchar *) xmlNodeGetContent (node);
      sport->abbreviation = (gchar *) xmlNodeGetContent (node);
      sport->description = g_strdup ("");
    }
   
    if (KP_TAG_MATCH (node, "icon"))
      g_warning ("Exercise icons are not supported in Kipin yet!");  

    if (KP_TAG_MATCH (node, "color"))
     sport->data = rgb_to_hex_str (node);
  }

  /* Subtypes must be parsed after other things have been parsed, because
   * the other things are needed when adding subtypes */
  node = parent->children;

  g_return_val_if_fail (node != NULL, FALSE);
  for (; node; node = node->next) {
    /* Find subtype list */
    if (KP_TAG_MATCH (node, "sport-subtype-list")) {
      /* Find all subtypes */
      for (child = node->children; child; child = child->next) {
        if (KP_TAG_MATCH (child, "sport-subtype"))
          read_sport_subtype (child, sport);
      }
    }
  }
  
  if (id <= 0) {
    g_free (sport);
    return FALSE;
  }

  kp_debug ("Found sport: %s", sport->name);
 
  add_sport_id (id, FALSE, sport->name);
  kp_preset_data_add_item (KP_PRESET_DATA_SPORT, sport);
  
  return TRUE;
}


static gboolean
read_exercise (xmlNodePtr node)
{
  KPCalendarTime *ctime;
  KPWorkout *wo;
  KPParam *param;
  gdouble d, m, y, h, M, s;
  gchar *tmp;
  GValue val;

  guint id = 0;
  guint sub_id = 0;

  val.g_type = 0;
  
  kp_debug ("Reading exercise..");
  
  wo = kp_workout_new ();
  
  for (node = node->children; node; node = node->next) {
    tmp = (gchar *) xmlNodeGetContent (node);
    
    if (KP_TAG_MATCH (node, "sport-type-id")) 
      id = kp_number (tmp);

    if (KP_TAG_MATCH (node, "sport-subtype-id"))
      sub_id = kp_number (tmp);

    if (KP_TAG_MATCH (node, "intensity")) {
      param = kp_param_new ("intensity");
      kp_param_set_string (param, tmp);
      kp_param_list_insert (wo->param_list, "detail", param);
    }

    if (KP_TAG_MATCH (node, "distance")) {
      param = kp_param_new ("distance");
      kp_param_set_double (param, kp_number (tmp));
      kp_param_list_insert (wo->param_list, "default", param);
    }
 
    if (KP_TAG_MATCH (node, "duration")) {
      param = kp_param_new ("duration");
      kp_param_set_time (param, 1000 * (guint) kp_number (tmp));
      kp_param_list_insert (wo->param_list, "default", param);
    }

    if (KP_TAG_MATCH (node, "comment")) 
      kp_workout_set_comment (wo, tmp);
      
    if (KP_TAG_MATCH (node, "avg-speed")) {
      param = kp_param_new ("avg-speed");
      kp_param_set_double (param, kp_number (tmp));
      kp_param_list_insert (wo->param_list, "detail", param);
    }

    if (KP_TAG_MATCH (node, "ascent")) {
      param = kp_param_new ("ascent");
      kp_param_set_double (param, kp_number (tmp));
      kp_param_list_insert (wo->param_list, "detail", param);
    }

    if (KP_TAG_MATCH (node, "avg-heartrate")) {
      param = kp_param_new ("avg-heartrate");
      kp_param_set_uint (param, (guint) kp_number (tmp));
      kp_param_list_insert (wo->param_list, "detail", param);
    }

    if (KP_TAG_MATCH (node, "date")) {
      g_return_val_if_fail (strlen (tmp) == 33, FALSE);
    
      tmp[4] = tmp[7] = tmp[10] = tmp[13] = tmp[16] = tmp[19] = '\0';

      y = kp_number (&tmp[0]);
      m = kp_number (&tmp[5]);
      d = kp_number (&tmp[8]);
      h = kp_number (&tmp[11]);
      M = kp_number (&tmp[14]);
      s = kp_number (&tmp[17]);
     
      if (y > 0 && m > 0 && d > 0 && h >= 0 && M >= 0 && s >= 0) {
        ctime = kp_calendar_time_new_dmyhms (d, m, y, h, M, s);
        KP_CALENDAR_ENTRY (wo)->datetime = ctime;
      }
    }
    
    g_free (tmp);
  }

  if (sub_id > 0)
    kp_workout_set_sport (wo, get_sport_for_id (sub_id, TRUE));
  else if (id > 0)
    kp_workout_set_sport (wo, get_sport_for_id (id, FALSE));
  
  kp_training_log_add (tlog, KP_CALENDAR_ENTRY (wo));

  return TRUE;
}


G_CONST_RETURN gchar *
get_sport_for_id (guint id, gboolean sub)
{
  const gchar *str;
  
  str = g_hash_table_lookup ((sub) ? sub_sports : sports,
                             GUINT_TO_POINTER (id));
  if (!str)
    str = _("Unknown sport");
 
  return str;
}

static void
add_sport_id (guint id, gboolean sub, const gchar *sport)
{
  g_return_if_fail (sport != NULL);
  g_return_if_fail (id > 0);

  g_hash_table_insert ((sub) ? sub_sports : sports, GUINT_TO_POINTER (id),
                       g_strdup (sport));
}



