#include <string.h>
#include <glib.h>
#include <glib-object.h>

#include "kpworkoutmodel.h"
#include "kpcalendarentry.h"
#include "kptraininglog.h"
#include "kppresetdata.h"
#include "kipina-i18n.h"
#include "kpworkout.h"
#include "kputil.h"

/* GObject stuff */
static void       kp_workout_class_init           (GObjectClass *klass,
                                                   gpointer data);
static void       kp_workout_model_init           (KPWorkoutModelIface *iface);
static void       kp_workout_instance_init        (GObject *object,
                                                   gpointer data);
static void       kp_workout_instance_finalize    (GObject *object);


/* KPCalendarEntry virtual method implementations */
static gchar     *kp_workout_get_human_name       (KPCalendarEntry *entry);
static gchar     *kp_workout_get_icon_name        (KPCalendarEntry *entry);
static xmlNodePtr kp_workout_to_xml               (KPCalendarEntry *entry);
static G_CONST_RETURN gchar *
                  kp_workout_to_calendar_string   (KPCalendarEntry *entry);
static gboolean   kp_workout_parse                (KPCalendarEntry *entry,
                                                   xmlNodePtr node);
/* KPWorkoutModel methods */
static guint      kp_workout_get_duration         (KPWorkoutModel *wo);
static gdouble    kp_workout_get_distance         (KPWorkoutModel *wo);
static guint      kp_workout_get_pace             (KPWorkoutModel *wo);
static gdouble    kp_workout_get_speed            (KPWorkoutModel *wo);

GType
kp_workout_get_type ()
{
  static GType kp_workout_type = 0;

  if (!kp_workout_type) {
    static const GTypeInfo kp_workout_info = {
      sizeof (KPWorkoutClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_workout_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPWorkout),
      0,
      (GInstanceInitFunc) kp_workout_instance_init,
      NULL
    };
    static const GInterfaceInfo workout_model_info = {
      (GInterfaceInitFunc) kp_workout_model_init,
       NULL,
       NULL
    };
    kp_workout_type = g_type_register_static (KP_TYPE_CALENDAR_ENTRY,
                                             "KPWorkout",
                                             &kp_workout_info,
                                              0);
    g_type_add_interface_static (kp_workout_type,
                                 KP_TYPE_WORKOUT_MODEL,
                                &workout_model_info);
  }
  return kp_workout_type;
}


static gchar *
kp_workout_get_icon_name (KPCalendarEntry *entry)
{
  return g_strdup ("runner.xpm");
}


/* Default implementation for get_sport_list() method */
static GSList *
_kp_workout_get_sport_list (KPWorkout *wo, guint *n)
{
  GSList *list = NULL;
  list = g_slist_prepend (list, g_strdup (kp_workout_get_sport (wo)));
  return wo->sports;
}



GSList *
kp_workout_get_sport_list (KPWorkout *wo, guint *n)
{
  return KP_WORKOUT_GET_CLASS (wo)->get_sport_list (wo, n);
}



static void
kp_workout_class_init (GObjectClass *klass, gpointer data)
{
  KPCalendarEntryClass *entry_class;
  KPWorkoutClass *workout_class;
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_workout_instance_finalize;

  entry_class = KP_CALENDAR_ENTRY_CLASS (klass);
  entry_class->get_human_name = kp_workout_get_human_name;
  entry_class->get_icon_name = kp_workout_get_icon_name;
  entry_class->to_string = kp_workout_to_calendar_string;
  entry_class->to_xml = kp_workout_to_xml;
  entry_class->parse = kp_workout_parse;

  workout_class = KP_WORKOUT_CLASS (klass);
  workout_class->get_sport_list = _kp_workout_get_sport_list;
}

static void
kp_workout_model_init (KPWorkoutModelIface *iface)
{
  iface->get_duration = kp_workout_get_duration;
  iface->get_distance = kp_workout_get_distance;
  iface->get_speed = kp_workout_get_speed;
  iface->get_pace = kp_workout_get_pace;
}


static void
kp_workout_instance_init (GObject *object, gpointer data)
{
  KPWorkout *wo;

  wo = KP_WORKOUT (object);
  
  wo->sports = NULL;
  wo->param_list = kp_param_list_new ();
}

static void
kp_workout_instance_finalize (GObject *object)
{
  GObjectClass *parent_class;
  
  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}

/**
 * kp_workout_new:
 * 
 * Create a new instance of #KPWorkout.
 * 
 * Returns: A new #KPWorkout.
 */
KPWorkout *
kp_workout_new ()
{
  return g_object_new (kp_workout_get_type (), NULL);
}

/**
 * kp_workout_is_valid:
 * @wo: A #KPWorkout
 *
 * Check the validity of the #KPWorkout.
 * 
 * Returns: TRUE if workout is valid and FALSE otherwise.
 */
gboolean
kp_workout_is_valid (KPWorkout *wo)
{
  if (wo == NULL)
    return FALSE;

  if (KP_CALENDAR_ENTRY (wo)->datetime == NULL)
    return FALSE;

  return TRUE;
}


/**
 * kp_workout_copy:
 * @wo: A #KPWorkout
 *
 * Copies all the data of @wo and returns the copied workout.
 *
 * Returns: A newly-allocated copy of @wo.
 */
KPWorkout *
kp_workout_copy (KPWorkout *wo)
{
  KPCalendarTime *ct_copy;
  KPWorkout *copy;
  KPDate date;
  KPTime t;
  
  g_return_val_if_fail (KP_IS_WORKOUT (wo), NULL);

  copy = kp_workout_new ();

  /* Get the date */
  kp_calendar_entry_get_date (KP_CALENDAR_ENTRY (wo), &date);
  kp_calendar_entry_get_time (KP_CALENDAR_ENTRY (wo), &t);
 
  /* Set the date on copy */
  ct_copy = KP_CALENDAR_ENTRY (copy)->datetime;
  kp_calendar_time_set_dmy (ct_copy, date.d, date.m, date.y);
  kp_calendar_time_set_hmst (ct_copy, t.h, t.m, t.s, 0);
  
  kp_debug ("Old: %p, New: %p\n", KP_CALENDAR_ENTRY (wo), KP_CALENDAR_ENTRY (copy));
 
  /*  Unref param list that was created in _init() */
  g_object_unref (copy->param_list);
  
  copy->param_list = kp_param_list_copy (wo->param_list);

  kp_workout_set_sport (copy, kp_workout_get_sport (wo));
  
  return copy;
}

/**
 * kp_workout_set_comment:
 * @wo: A #KPWorkout
 * @comment: Comment string
 *
 * Set the comment of the workout.
 */
void
kp_workout_set_comment (KPWorkout *wo, const gchar *comment)
{
  g_return_if_fail (KP_IS_WORKOUT (wo));
  g_return_if_fail (comment != NULL);

  kp_param_list_set_string (wo->param_list, "default", "comment", comment);
}

/**
 * kp_workout_get_comment:
 * @wo: A #KPWorkout
 *
 * Get the comment of the workout.
 *
 * Returns: A newly-allocated string that must be freed.
 */
gchar *
kp_workout_get_comment (KPWorkout *wo)
{
  g_return_val_if_fail (KP_IS_WORKOUT (wo), NULL);
  return kp_param_list_get_as_string (wo->param_list, "comment");
}

/**
 * kp_workout_get_distance:
 * @wo: a KPWorkout
 *
 * Returns distance as gdouble.
 **/
static gdouble
kp_workout_get_distance (KPWorkoutModel *model)
{
  g_return_val_if_fail (KP_IS_WORKOUT_MODEL (model), 0);
  return kp_param_list_get_double (KP_WORKOUT (model)->param_list, "distance");
}


/**
 * kp_workout_set_sport:
 * @wo: A #KPWorkot
 * @sport: The name of the sport, can be NULL.
 *
 * Set the sport of the workout.
 */
void
kp_workout_set_sport (KPWorkout *wo, const gchar *sport)
{
  g_return_if_fail (KP_IS_WORKOUT (wo));

  if (wo->sports) {
    if (wo->sports->data)
      g_free (wo->sports->data);
    wo->sports->data = g_strdup (sport);
  } else
    wo->sports = g_slist_prepend (wo->sports, g_strdup (sport));
  
  kp_debug ("Sport set: %s\n", sport);
}


/**
 * kp_workout_get_sport:
 * @wo: A #KPWorkout
 * 
 * Get the sport of the workout.
 * 
 * Returns: String that must NOT be freed.
 */
G_CONST_RETURN gchar *
kp_workout_get_sport (KPWorkout *wo)
{
  static GString *buf = NULL;
  KPPresetDataItem *item;
  const gchar *sport;
  gchar *fmt;
  GSList *node;
  guint i;
  
  g_return_val_if_fail (KP_IS_WORKOUT (wo), NULL);

  if (!buf)
    buf = g_string_new ("");
  
  if (!wo->sports)
    return NULL;

  /* If there is more than one sport (split workout) */
  if (wo->sports->next) {
    g_string_assign (buf, "");
    for (i=0, node = wo->sports; node; i++, node = node->next) {
      sport = (const gchar *) node->data;
      item = kp_preset_data_get_item (KP_PRESET_DATA_SPORT, sport);
      
      fmt = (i == 0) ? "%s" : " + %s";
      
      /* If there is an abbreviation, use it */
      if (item != NULL && item->abbreviation != NULL) {
        g_string_append_printf (buf, fmt, item->abbreviation);
      } else {
        g_string_append_printf (buf, fmt, sport);
      }
    }
  } else
    /* Normal workout */
    g_string_assign (buf, wo->sports->data);
  
  return buf->str;
}



/**
 * kp_workout_get_pace:
 * @wo: A #KPWorkout
 * * 
 * Get pace as milliseconds.
 *
 * Returns: Pace as milliseconds.
 */
guint32
kp_workout_get_pace (KPWorkoutModel *wo)
{
  guint duration;
  gint32 distance;
  
  duration = kp_workout_model_get_duration (KP_WORKOUT_MODEL (wo));
  distance = kp_workout_model_get_distance (KP_WORKOUT_MODEL (wo)) * 1000;

  if (duration > 0 && distance > 0) { 
    duration = (guint32)((gdouble) duration / ((gdouble) distance / 1000));
    /* Remove .xxx */
    duration /= 1000;
    return duration * 1000;
  }
  
  return 0;
}


gdouble
kp_workout_get_speed (KPWorkoutModel *wo)
{
  gdouble duration;
  gdouble distance;

  duration = (gdouble) kp_workout_model_get_duration (KP_WORKOUT_MODEL (wo));
  distance = kp_workout_model_get_distance (KP_WORKOUT_MODEL (wo));

  if (duration > 0.0 && distance > 0.0) {
    duration /= (60 * 60 * 1000);
    return (gdouble) distance / (gdouble) duration;
  }
  return 0.0; 
}


/**
 * kp_workout_get_duration:
 * @wo: A #KPWorkout
 * 
 * Retrieve duration of the kp_workout in milliseconds.
 *
 * Returns: Number of milliseconds
 */
static guint
kp_workout_get_duration (KPWorkoutModel *wo)
{
  KPParam *param;
  g_return_val_if_fail (KP_IS_WORKOUT (wo), 0); 

  param = kp_param_list_get_param (KP_WORKOUT (wo)->param_list, "duration");
  
  return (param) ? kp_param_get_time (param) : 0;
}


/**
 * kp_workout_get_formatted_date:
 * @wo: a KPWorkout.
 *
 * Returns date in format CCYY-MM-DDThh:mm:ss. Result-string
 * is malloc'd, so it must be freed.
 **/
gchar *
kp_workout_get_formatted_date (KPWorkout *wo)
{
  GDate *date;
  gchar *format;

  g_return_val_if_fail (wo != NULL, NULL);
  g_return_val_if_fail (KP_CALENDAR_ENTRY (wo)->datetime != NULL, NULL);

  date = kp_calendar_time_get_date (KP_CALENDAR_ENTRY (wo)->datetime);
  g_return_val_if_fail (date != NULL, NULL);

  format = g_strdup_printf ("%.4d-%.2d-%.2dT%.2d:%.2d:%.2d",
                            g_date_get_year (date),
                            g_date_get_month (date),
                            g_date_get_day (date),
                            KP_CALENDAR_ENTRY (wo)->datetime->h,
                            KP_CALENDAR_ENTRY (wo)->datetime->m,
                            KP_CALENDAR_ENTRY (wo)->datetime->s);
  
  return format;
}


/**
 * Get kp_workout as a string that can be passed directly to CalendarView or 
 * something like that.
 *
 * Returns a const string that must not be freed by the caller.
 */
static G_CONST_RETURN gchar *
kp_workout_to_calendar_string (KPCalendarEntry *entry)
{
  KPPresetDataItem *sport;
  KPParam *param;
  gdouble distance;
  static GString *string = NULL;
  gchar *duration;
  gchar *markup_escape;
  gchar *comment;
  gchar *color;
  KPWorkout *wo;
  const gchar *sport_str;
  gchar *intensity;
  gchar *prefix;
  gchar *tmp;
  gchar *str;
  KPParamListCategory *cat;
  GList *cat_node, *param_node;
  
  g_return_val_if_fail (KP_IS_WORKOUT (entry), NULL);
  wo = KP_WORKOUT (entry);

  if (string == NULL)
    string = g_string_new (NULL);
  else
    g_string_assign (string, "");
 
  distance = kp_workout_model_get_distance (KP_WORKOUT_MODEL (wo));
  param = kp_param_list_get_param (KP_WORKOUT (wo)->param_list, "duration");
  duration = (param) ? kp_param_get_as_string (param) : 0;

  intensity = kp_param_list_get_as_string (KP_WORKOUT (wo)->param_list, "intensity");
  
  sport_str = kp_workout_get_sport (wo);
  g_return_val_if_fail (sport_str != NULL, NULL);
    
  if ((sport = kp_preset_data_get_item (KP_PRESET_DATA_SPORT, sport_str))) {
    color = sport->data;
    if (sport->abbreviation)
      sport_str = sport->abbreviation;
  } else
    color = NULL;
  
  /* Set the distance and the duration */
  if (distance > 0.0 && duration)
    g_string_printf (string, "<b>%s</b> %.1fkm\n%s\n", sport_str, distance,
                     duration);
  else if (duration)
    g_string_printf (string, "<b>%s</b>: %s\n", sport_str, duration);
  else 
    g_string_printf (string, "<b>%s</b>: %.1fkm\n", sport_str, distance);

  if (duration)
    g_free (duration);

  if (intensity) {
    g_string_append_printf (string, "intensity: %s\n", intensity);
    g_free (intensity);
  }
  
  /* Additional Params */
  cat_node = KP_WORKOUT (entry)->param_list->list;  
  for (; cat_node; cat_node = cat_node->next) {
    cat = (KPParamListCategory *) cat_node->data;

    /* "default" params are shown already */
    if (strcmp (cat->name, "default") != 0) {
      for (param_node = cat->list; param_node; param_node = param_node->next) {
        /* Append a param to a string */
        tmp = kp_param_get_as_string (KP_PARAM (param_node->data));
        g_string_append_printf (string, 
                               "%s: %s\n", 
                                kp_param_get_name (KP_PARAM (param_node->data)),
                                tmp);
        g_free (tmp);
      }
    }
  }
  
  
  /* Set the comment */

  comment = kp_workout_get_comment (KP_WORKOUT (entry));
  if (comment) {
    markup_escape = g_markup_escape_text (comment, -1);
    string = g_string_append (string, markup_escape);
    g_free (markup_escape);
    g_free (comment);
  }
 
  if (string->str[string->len-1] == '\n')
    g_string_erase (string, string->len-1, 1);
    
  /* Set the time */
  str = kp_calendar_time_to_time_string (entry->datetime);
  tmp = g_strdup_printf ("<b>%s</b>: ", str);
  g_string_prepend (string, tmp);
  g_free (tmp);
  g_free (str);
 
  /* Set the color */
  if (color) {
    prefix = g_strdup_printf ("<span color=\"%s\">", color);
    g_string_append (string, "</span>");
    g_string_prepend (string, prefix);
    g_free (prefix);
  }
  
  return string->str;
}


gboolean
kp_workout_is_default_param (const gchar *name)
{
  guint i;
 
  g_return_val_if_fail (name != NULL, FALSE);
  
  static gchar *params[] = {
    "distance",
    "duration",
    "comment",
    "intensity"
  };

  for (i=0; i < G_N_ELEMENTS (params); i++)
    if (strcmp (params[i], name) == 0)
      return TRUE;
  
  return FALSE;
}

 
static void
kp_workout_param_import (KPWorkout *wo, xmlNodePtr node)
{
  KPParam *param;
  gchar *name;
  gchar *strval;
  gchar *type;
  gdouble val;

  name = (gchar *) xmlGetProp (node, BAD_CAST ("name"));
  type = (gchar *) xmlGetProp (node, BAD_CAST ("type"));
  strval = (gchar *) xmlNodeGetContent (node); 
  param = kp_param_new (name);

  if (!name || !type || !strval)
    return;    
   
  if (strcmp (type, "float") == 0) {
    val = kp_number (strval);
    kp_param_set_double (param, (val > 0.0) ? val : 0.0);
  }
  else if (strcmp (type, "uint") == 0) {
    val = kp_number (strval);  
    kp_param_set_uint (param, (val > 0.0) ? (guint) val : 0);
  }
  else if (strcmp (type, "bool") == 0) 
    kp_param_set_boolean (param, (strcmp (strval, "true") == 0));
  else 
    kp_param_set_automatic_as_string (param, strval);

  kp_param_list_insert (wo->param_list,
                       (kp_workout_is_default_param (name) ? "default" : "detail"),
                        param);
  g_free (name);
  g_free (strval);
  g_free (type);
}


/**
 * kp_workout_parse:
 * @entry: A #KPCalendarEntry
 * @node: A #xmlNodePtr.
 *
 * Parses a kp_workout and returns it. If kp_workout's type
 * is interval, interval-params are stored to params-hashtable.
 * Interval-things' keys use "__interval__"-prefix to prevent
 * name-collisions which should be impossible this way.
 *
 * Returns: TRUE if successful or FALSE otherwise.
 */
static gboolean
kp_workout_parse (KPCalendarEntry *entry, xmlNodePtr node)
{
  xmlNodePtr child;
  xmlChar *date;
  xmlChar *prop = xmlGetProp (node, BAD_CAST ("type"));

  kp_workout_set_sport (KP_WORKOUT (entry), (const gchar *) prop);
  g_free (prop);
  
  child = node->children;

  while (child) {
    
    /** <datetime> **/
    if (KP_TAG_MATCH (child, "datetime")) {
      date = xmlNodeGetContent (child);
      if (!kp_calendar_time_set_datetime (KP_CALENDAR_ENTRY (entry)->datetime,
                                          (const gchar *) date))
        return FALSE;
      g_free (date);
    }

    /** <param> **/
    else if (KP_TAG_MATCH (child, "param")) {
      if (KP_IS_WORKOUT (entry))
        kp_workout_param_import (KP_WORKOUT (entry), child);
    }
    else if (KP_TAG_MATCH (child, "text")) {
      /* doesn't need to care about these */    
    } else {
      g_warning ("<workout> has an unknown child: %s", child->name);
    }
    child = child->next;
  }

  if (!kp_workout_is_valid (KP_WORKOUT (entry)))
    g_warning ("Parse error: this isn't valid workout!\n");
  else
    return TRUE;

  g_object_unref (entry);

  return FALSE;
}

static gchar *
kp_workout_get_human_name (KPCalendarEntry *entry)
{
  return g_strdup (_("Workout"));
}

static xmlNodePtr
kp_workout_to_xml (KPCalendarEntry *entry)
{
  xmlNodePtr date_child;
  xmlNodePtr node;
  xmlAttrPtr attr = NULL;
  xmlChar *date;

  g_return_val_if_fail (KP_IS_WORKOUT (entry), NULL);

  node = xmlNewNode (NULL, BAD_CAST ("workout"));

  if (kp_workout_get_sport (KP_WORKOUT (entry)) != NULL) {
    attr = xmlNewProp (node, BAD_CAST ("type"), 
                             BAD_CAST (kp_workout_get_sport (KP_WORKOUT (entry))));
    g_return_val_if_fail (attr != NULL, NULL);
  }

  date = BAD_CAST (kp_workout_get_formatted_date (KP_WORKOUT (entry)));
  date_child = xmlNewChild (node, NULL, BAD_CAST ("datetime"), date);
  g_return_val_if_fail (attr != NULL, NULL);

  kp_param_list_export_as_xml (KP_WORKOUT (entry)->param_list, node);
  
  return node;
}

