#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <glib.h>
#include <glib-object.h>

#include "kpcalendarentry.h"
#include "kpworkoutmodel.h"
#include "kputil.h"
#include "kplogstatitem.h"
#include "kplogstat.h"

/* Well, this whole log stat thingy is very inefficient, but it is easy to use
 * so it's good for me :-) */

static void       kp_log_stat_class_init           (GObjectClass *klass,
                                                    gpointer data);
static void       kp_log_stat_instance_init        (GObject *object,
                                                    gpointer data);
static void       kp_log_stat_instance_finalize    (GObject *object);


GType
kp_log_stat_get_type ()
{
  static GType kp_log_stat_type = 0;

  if (!kp_log_stat_type) {
    static const GTypeInfo kp_log_stat_info = {
      sizeof (KPLogStatClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_log_stat_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPLogStat),
      0,
      (GInstanceInitFunc) kp_log_stat_instance_init,
      NULL
    };
    kp_log_stat_type = g_type_register_static (KP_TYPE_CALENDAR_ENTRY,
                                             "KPLogStat",
                                             &kp_log_stat_info,
                                              0);
  }
  return kp_log_stat_type;
}


static void
kp_log_stat_class_init (GObjectClass *klass, gpointer data)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_log_stat_instance_finalize;
}


static gboolean
str_equal (gconstpointer a, gconstpointer b)
{
  return strcmp (a, b) == 0;
}

static void
kp_log_stat_instance_init (GObject *object, gpointer data)
{
  KPLogStat *stat;

  stat = KP_LOG_STAT (object);
  stat->buf = g_string_new_len (NULL, 64);
  stat->data = g_hash_table_new_full (g_str_hash, str_equal, g_free, 
                                     (GDestroyNotify) kp_log_stat_item_free);
}


  
static void
kp_log_stat_instance_finalize (GObject *object)
{
  GObjectClass *parent_class;
  
  g_hash_table_destroy (KP_LOG_STAT (object)->data);
  g_string_free (KP_LOG_STAT (object)->buf, TRUE);
  
  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}

/**
 * kp_log_stat_new:
 * 
 * Create a new unset instance of #KPLogStat.
 * 
 * Returns: A new #KPLogStat.
 */
KPLogStat *
kp_log_stat_new (KPTrainingLog *log)
{
  KPLogStat *stat;

  stat = g_object_new (kp_log_stat_get_type (), NULL);
  stat->log = log;
  /*kp_log_stat_load (stat, log);*/
  
  return stat;
}


/*
 * Month must be between 1 and 12
 *
 */
KPLogStatItem *
kp_log_stat_get_month_stats (KPLogStat *stat, guint year, guint month, 
                             const gchar *sport, const gchar *intensity) 
{
  KPLogStatItem *item;
  GList *node;
  TYear *ty = NULL;
  
  g_return_val_if_fail (KP_IS_LOG_STAT (stat), NULL);
  g_return_val_if_fail (KP_IS_TRAINING_LOG (stat->log), NULL);

  if ((ty = kp_training_log_get_year (stat->log, year)) == NULL)
    return NULL;

  item = kp_log_stat_item_new (KP_PERIOD_MONTH, sport, intensity);
  
  for (node = ty->workouts[month-1]; node; node = node->next) {
    item->n_entries++;
    if (KP_IS_WORKOUT (node->data))
      kp_log_stat_item_add_workout (item, KP_WORKOUT (node->data));
  }
  return item;
}



/**
 * kp_log_stat_get_year_stats:
 * @stat: A #KPLogStat
 * @year: Year to get stats for
 * @sport: sport name or NULL
 * @intensity: intensity string or NULL
 *
 * Get stats for @year. If sport or intensity is specified,
 * only workouts which match with these, are counted. 
 *
 * !! 
 * KPLogStatItem field n_entries will contain all the entries in that year,
 * INCLUDING those workouts with non-match sport and/or intensity.
 * !!
 *
 * Returns: A #KPLogStatItem or NULL if something fails.
 */
KPLogStatItem *
kp_log_stat_get_year_stats (KPLogStat *stat, guint year, const gchar *sport,
                            const gchar *intensity) 
{
  KPLogStatItem *item;
  GList *node;
  TYear *ty = NULL;
  guint m;

  /* Remove this after */
  guint tmp;
  guint ok=0, fail=0;
  
  g_return_val_if_fail (KP_IS_LOG_STAT (stat), NULL);
  g_return_val_if_fail (KP_IS_TRAINING_LOG (stat->log), NULL);

  if ((ty = kp_training_log_get_year (stat->log, year)) == NULL) {
    g_warning ("No year %u in the log!\n", year);
    return NULL;
  }

  item = kp_log_stat_item_new (KP_PERIOD_YEAR, sport, intensity);
  
  for (m=0; m < 12; m++) {
    for (node = ty->workouts[m]; node; node = node->next) {
      item->n_entries++;
      if (KP_IS_WORKOUT (node->data)) 
        kp_log_stat_item_add_workout (item, KP_WORKOUT (node->data));
    }
  }
  return item;
}


KPLogStatItem *
kp_log_stat_get_stats (KPLogStat *stat)
{
  KPLogStatItem *item, *yitem;
  GList *l;

  item = kp_log_stat_item_new (KP_PERIOD_ALL_TIME, NULL, NULL);
  
  for (l = stat->log->year_list; l; l = l->next) {
    yitem = kp_log_stat_get_year_stats (stat, ((TYear *) l->data)->year, 
                                        NULL, NULL);

    kp_log_stat_item_add_item (item, yitem);

    kp_log_stat_item_free (yitem);
  }
  g_print ("OK N entries: %u, N workouts: %u\n", item->n_entries, item->n_workouts);

  return item;
}
