#include <stdio.h>

#include <glib.h>
#include <glib-object.h>

#include "kpcalendarentry.h"
#include "kpcalendartime.h"
#include "kputil.h"

enum {
  CHANGED_SIGNAL,
  LAST_SIGNAL
};

static guint      kp_calendar_entry_signals[LAST_SIGNAL] = { 0 };

static void       kp_calendar_entry_class_init        (GObjectClass *klass,
                                                       gpointer data);
static void       kp_calendar_entry_instance_init     (GObject *object,
                                                       gpointer data);
static void       kp_calendar_entry_instance_finalize (GObject *object);

static void       entry_date_changed                  (KPCalendarTime *ct,
                                                       KPdmyhmst *date,
                                                       KPCalendarEntry *entry);
GType
kp_calendar_entry_get_type (void)
{
  static GType kp_calendar_entry_type = 0;

  if (!kp_calendar_entry_type) {
    static const GTypeInfo kp_calendar_entry_info = {
      sizeof (KPCalendarEntryClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_calendar_entry_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPCalendarEntry),
      0,
      (GInstanceInitFunc) kp_calendar_entry_instance_init,
      NULL
    };

    kp_calendar_entry_type = g_type_register_static (G_TYPE_OBJECT,
                                                    "KPCalendarEntry",
                                                    &kp_calendar_entry_info, 0);
  }
  return kp_calendar_entry_type;
}

static void
kp_calendar_entry_class_init (GObjectClass *klass, gpointer data)
{
  GObjectClass *object_class;
  KPCalendarEntryClass *entry_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_calendar_entry_instance_finalize;
  
  entry_class = KP_CALENDAR_ENTRY_CLASS (klass);
  entry_class->get_human_name = NULL;
  entry_class->to_string = NULL;
  entry_class->to_xml = NULL;
  entry_class->parse = NULL;

  kp_calendar_entry_signals[CHANGED_SIGNAL]
    = g_signal_new ("changed",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPCalendarEntryClass, changed),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__STRING,
                    G_TYPE_NONE,
                    1,
                    G_TYPE_STRING);
}

static void
kp_calendar_entry_instance_init (GObject *object, gpointer data)
{
  KPCalendarEntry *entry;
  static guint id_number = 1;
  
  entry = KP_CALENDAR_ENTRY (object);
  entry->datetime = kp_calendar_time_new ();
  entry->id = id_number;
  
  g_signal_connect (G_OBJECT (KP_CALENDAR_ENTRY (object)->datetime), "changed",
                    G_CALLBACK (entry_date_changed), object);
  id_number++;
}

static void
entry_date_changed (KPCalendarTime *ct, KPdmyhmst *date, KPCalendarEntry *entry)
{
#if 0
  kp_debug ("Datetime changed, old was: %d.%d.%d %d:%d\n",
      date->d, date->m, date->y, date->h, date->min);
#endif
}

static void
kp_calendar_entry_instance_finalize (GObject *object)
{
  /*g_object_unref (KP_CALENDAR_ENTRY (object)->datetime);
  kp_debug ("Calendar entry finalized.");*/
}

KPCalendarEntry *
kp_calendar_entry_new ()
{
  return g_object_new (kp_calendar_entry_get_type (), NULL);
}


G_CONST_RETURN gchar *
kp_calendar_entry_to_string (KPCalendarEntry *entry)
{
  return KP_CALENDAR_ENTRY_GET_CLASS (entry)->to_string (entry);
}

xmlNodePtr 
kp_calendar_entry_to_xml (KPCalendarEntry *entry)
{
  return KP_CALENDAR_ENTRY_GET_CLASS (entry)->to_xml (entry);
}


gboolean
kp_calendar_entry_parse (KPCalendarEntry *entry, xmlNodePtr node)
{
  return KP_CALENDAR_ENTRY_GET_CLASS (entry)->parse (entry, node);
}

gint
kp_calendar_entry_cmp (const KPCalendarEntry *entry, const KPCalendarEntry *entry2)
{
  GDate *date1;
  GDate *date2;

  date1 = kp_calendar_time_get_date (KP_CALENDAR_ENTRY (entry)->datetime);
  date2 = kp_calendar_time_get_date (KP_CALENDAR_ENTRY (entry2)->datetime);

  return g_date_compare (date1, date2);
}

gchar *
kp_calendar_entry_get_human_name (KPCalendarEntry *entry)
{
  return KP_CALENDAR_ENTRY_GET_CLASS (entry)->get_human_name (entry);
}

/**
 * kp_calendar_entry_get_icon_name:
 * @entry: A #KPCalendarEntry
 *
 * Get name of the icon for some KPCalendarEntry derived object.
 * 
 * Returns: A newly-allocated string that is a path to the icon file.
 */
gchar *
kp_calendar_entry_get_icon_name (KPCalendarEntry *entry)
{
  return KP_CALENDAR_ENTRY_GET_CLASS (entry)->get_icon_name (entry);
}


gboolean
kp_calendar_entry_get_date (KPCalendarEntry *entry, KPDate *date)
{
  KPDate d;
  
  g_return_val_if_fail (KP_IS_CALENDAR_ENTRY (entry), FALSE);
  g_return_val_if_fail (date != NULL, FALSE);
  
  if (!kp_calendar_time_to_kpdate (entry->datetime, &d))
    return FALSE;

  g_return_val_if_fail (g_date_valid_dmy (d.d, d.m, d.y), FALSE);
    
  date->d = d.d;
  date->m = d.m;
  date->y = d.y;
  
  return TRUE;
}


guint
kp_calendar_entry_get_id (KPCalendarEntry *entry)
{
  return entry->id;
}


gboolean
kp_calendar_entry_get_time (KPCalendarEntry *entry, KPTime *time)
{
  g_return_val_if_fail (KP_IS_CALENDAR_ENTRY (entry), FALSE);
  g_return_val_if_fail (time != NULL, FALSE);
  g_return_val_if_fail (entry->datetime->m < 60, FALSE);
  g_return_val_if_fail (entry->datetime->s < 60, FALSE);
  
  time->h = entry->datetime->h;
  time->m = entry->datetime->m;
  time->s = entry->datetime->s;

  return TRUE;
}
