#include <string.h>
#include <time.h>

#include <gtk/gtk.h>

#include "kpdateentry.h"

#include "../kipina-i18n.h"
#include "../kputil.h"

#define MENU_TEXT_SELECT    "Select Time"
#define MENU_TEXT_EXACT     "Exact time of day (hh:mm:ss)"


static gchar *time_menu_texts[] = {
  N_(MENU_TEXT_SELECT),
  N_("Morning"),
  N_("Noon"),
  N_("Afternoon"),
  N_("Evening"),
  N_(MENU_TEXT_EXACT) /* This item must be the last one in the list! */
};

enum {
  DATE_SELECTED_SIGNAL,
  TIME_SELECTED_SIGNAL,
  LAST_SIGNAL
};

static guint    kp_date_entry_signals[LAST_SIGNAL] = { 0 };
static void     kp_date_entry_class_init     (KPDateEntryClass *klass);
static void     kp_date_entry_finalize       (GObject *object);
static void     kp_date_entry_init           (KPDateEntry *dialog);
static void     update_display_widgets       (KPDateEntry *entry);
static void     button_toggled               (GtkToggleButton *button,
                                              KPDateEntry *entry);
static void     time_menu_changed            (GtkComboBox *box, 
                                              KPDateEntry *dentry);
static GObjectClass *parent_class;

typedef struct KPDateEntryPrivateData_
{
  GtkWidget    *entry_date;
  GtkWidget    *entry_time;
  GtkWidget    *button_date;
  GtkWidget    *combo_time;

  KPTime        time;
  KPDate        date;
} KPDateEntryPrivateData;

#define KP_DATE_ENTRY_PRIVATE_DATA(widget) \
  (((KPDateEntryPrivateData*) \
  (KP_DATE_ENTRY (widget)->private_data)))


GType
kp_date_entry_get_type (void)
{
  static GType kp_date_entry_type = 0;

  if (kp_date_entry_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPDateEntryClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_date_entry_class_init,
      NULL,
      NULL,
      sizeof (KPDateEntry),
      0,
      (GInstanceInitFunc) kp_date_entry_init,
      NULL,
    };

    kp_date_entry_type = g_type_register_static (GTK_TYPE_TABLE,
                                                "KPDateEntry",
                                                &our_info, 0);
  }
  return kp_date_entry_type;
}


static void
kp_date_entry_class_init (KPDateEntryClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_date_entry_finalize;

  kp_date_entry_signals[DATE_SELECTED_SIGNAL]
    = g_signal_new ("date-selected",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPDateEntryClass, date_selected),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__POINTER,
                    G_TYPE_NONE,
                    1,
                    G_TYPE_POINTER);
  kp_date_entry_signals[TIME_SELECTED_SIGNAL]
    = g_signal_new ("time-selected",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPDateEntryClass, time_selected),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__POINTER,
                    G_TYPE_NONE,
                    1,
                    G_TYPE_POINTER);
}



static void
kp_date_entry_init (KPDateEntry *dentry)
{
  KPDateEntryPrivateData *p_data;
  guint i;
  
  dentry->private_data = g_new0 (KPDateEntryPrivateData, 1);
  p_data = KP_DATE_ENTRY_PRIVATE_DATA (dentry);
  
  p_data->entry_date = gtk_entry_new ();
  p_data->entry_time = gtk_entry_new ();
  gtk_editable_set_editable (GTK_EDITABLE (p_data->entry_date), FALSE);
  
  p_data->button_date = gtk_toggle_button_new_with_label (_("Select Date..."));
  p_data->combo_time = gtk_combo_box_new_text ();

  for (i=0; i < G_N_ELEMENTS (time_menu_texts); i++)
    gtk_combo_box_append_text (GTK_COMBO_BOX (p_data->combo_time),
                               time_menu_texts[i]);

  gtk_combo_box_set_active (GTK_COMBO_BOX (p_data->combo_time), 0);
  
  g_signal_connect (G_OBJECT (p_data->combo_time), "changed",
                    G_CALLBACK (time_menu_changed), dentry);

  gtk_table_set_col_spacings (GTK_TABLE (dentry), 6);
  gtk_table_set_row_spacings (GTK_TABLE (dentry), 6);
 
  gtk_table_attach (GTK_TABLE (dentry), p_data->button_date, 0, 1, 0, 1, 
                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (dentry), p_data->entry_date, 1, 2, 0, 1,
                    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (dentry), p_data->combo_time, 0, 1, 1, 2,
                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (dentry), p_data->entry_time, 1, 2, 1, 2,
                    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  
  gtk_widget_show (p_data->entry_date);
  gtk_widget_show (p_data->button_date);
  gtk_widget_show (p_data->combo_time);
  
  p_data->date.d = 0;
  p_data->date.m = 0;
  p_data->date.y = 0;
  p_data->time.h = 0;
  p_data->time.m = 0;
  p_data->time.s = 0;
  p_data->time.initialized = FALSE;
  
  g_signal_connect (G_OBJECT (p_data->button_date), "toggled",
                    G_CALLBACK (button_toggled), dentry);
}


static void
kp_date_entry_finalize (GObject *object)
{
  KPDateEntry *dialog;

  g_return_if_fail (KP_IS_DATE_ENTRY (object));

  dialog = KP_DATE_ENTRY (object);

  g_return_if_fail (dialog->private_data != NULL);

  g_free (dialog->private_data);
  
  G_OBJECT_CLASS (parent_class)->finalize (object);
}

/**
 * kp_date_entry_new:
 * @date: A #KPDate
 * @log: A #KPTrainingLog
 * @wo: A #KPWorkout
 *
 * Create an instance of #KPDateEntry.
 *
 * Returns: A #KPDateEntry
 */
GtkWidget *
kp_date_entry_new (void)
{
  return g_object_new (KP_TYPE_DATE_ENTRY, NULL);
}


/**
 * kp_date_entry_get_date:
 * @entry: A #KPDateEntry
 * @date: A location to store date or NULL
 *
 * Get the date that user has selected.
 * 
 * Returns: FALSE if date is not selected and TRUE otherwise
 */
gboolean
kp_date_entry_get_date (KPDateEntry *entry, KPDate *date)
{
  KPDateEntryPrivateData *p_data;
  p_data = KP_DATE_ENTRY_PRIVATE_DATA (entry);
  
  if (p_data->date.d == 0 || p_data->date.m == 0 || p_data->date.y == 0)
    return FALSE;
  else
    g_return_val_if_fail (g_date_valid_dmy (p_data->date.d, p_data->date.m,
                                            p_data->date.y), FALSE);
 
  if (date) {
    date->d = p_data->date.d;
    date->m = p_data->date.m;
    date->y = p_data->date.y;
  }
  
  return TRUE;
}

  
static gchar *
get_combo_box_active_string (GtkComboBox *box, KPDateEntry *dentry)
{
  KPDateEntryPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *str;
  
  p_data = KP_DATE_ENTRY_PRIVATE_DATA (dentry);
 
  g_return_val_if_fail (GTK_IS_COMBO_BOX (box), NULL);
  
  model = gtk_combo_box_get_model (box);
  gtk_combo_box_get_active_iter (box, &iter);
  gtk_tree_model_get (model, &iter, 0, &str, -1);
 
  return str;
}


#define IS_NUMBER(x) ((((guint) (x) - '0') < 10))

gboolean
get_number (const gchar *ptr, guint *num)
{
  if (IS_NUMBER (*ptr) && IS_NUMBER (*(ptr + 1))) {
    if (num)
      *num = ((guint) *ptr - '0') * 10 + ((guint) *(ptr + 1) - '0');
  } else
    return FALSE;
    
  return TRUE;
}
     
/**
 * kp_date_entry_get_time:
 * @time: A location to store time or NULL
 *
 * Get the time entered by the user.
 *
 * Returns: TRUE if the time is OK and FALSE otherwise.
 */
gboolean
kp_date_entry_get_time (KPDateEntry *entry, KPTime *time)
{
  KPDateEntryPrivateData *p_data;
  const gchar *str;
  gchar *active;
  
  p_data = KP_DATE_ENTRY_PRIVATE_DATA (entry);
 
  active = get_combo_box_active_string (GTK_COMBO_BOX (p_data->combo_time), 
                                        entry); 
  g_return_val_if_fail (active != NULL, FALSE);
  
  if (strcmp (active, _(MENU_TEXT_SELECT)) == 0) {
    g_free (active);
    return FALSE;
  } else if (strcmp (active, _(MENU_TEXT_EXACT)) != 0) {

    if (strcmp (active, _("Morning")) == 0)
      kp_time_set_tod (time, KP_TOD_MORNING);
    else if (strcmp (active, _("Noon")) == 0)
      kp_time_set_tod (time, KP_TOD_NOON);
    else if (strcmp (active, _("Afternoon")) == 0)
      kp_time_set_tod (time, KP_TOD_AFTERNOON);
    else if (strcmp (active, _("Evening")) == 0)
      kp_time_set_tod (time, KP_TOD_EVENING);
    else
      g_return_val_if_reached (FALSE);
   
    g_print ("Time: %u.%u.%u\n", time->h, time->m, time->s);
    g_free (active);

    kp_time_set_hms (&p_data->time, time->h, time->m, time->s);
    
    return TRUE;
  }
  g_free (active);
    
  str = gtk_entry_get_text (GTK_ENTRY (p_data->entry_time));
  
  g_return_val_if_fail (str != NULL, FALSE);
  g_return_val_if_fail (strlen (str) == 8, FALSE);

  if (str[2] != ':' || str[5] != ':')
    return FALSE;
  if (time) {
    if (get_number (&str[0], &time->h) == FALSE
     || get_number (&str[3], &time->m) == FALSE
     || get_number (&str[6], &time->s) == FALSE)
        return FALSE;
  } else {
    if (get_number (&str[0], NULL) == FALSE
     || get_number (&str[3], NULL) == FALSE
     || get_number (&str[6], NULL) == FALSE)
        return FALSE;
  }
  kp_time_set_hms (&p_data->time, time->h, time->m, time->s);

  return TRUE;
}



void
kp_date_entry_set_time (KPDateEntry *entry, KPTime *time)
{
  KPDateEntryPrivateData *p_data;
  
  p_data = KP_DATE_ENTRY_PRIVATE_DATA (entry);
  p_data->time.h = time->h;
  p_data->time.m = time->m;
  p_data->time.s = time->s;
  p_data->time.initialized = TRUE;
  
  update_display_widgets (entry);
} 



static void
update_display_widgets (KPDateEntry *entry)
{
  KPDateEntryPrivateData *p_data;
  struct tm tm;
  gchar buf[64];
  char *fmt = "%x";
  gchar *tmp;
  guint i;
  
  p_data = KP_DATE_ENTRY_PRIVATE_DATA (entry);
 
  /* Date */
  tm.tm_year = p_data->date.y - 1900;
  tm.tm_mon = p_data->date.m - 1;
  tm.tm_mday = p_data->date.d;
  tm.tm_hour = tm.tm_min = tm.tm_sec = 0;

  strftime (buf, sizeof (buf), fmt, &tm);

  gtk_entry_set_text (GTK_ENTRY (p_data->entry_date), buf);

  /* Time */
  if (p_data->time.initialized == FALSE)
    return;
  
  tmp = kp_time_tod_to_string (&p_data->time);
  if (tmp == NULL)
    return;

  for (i=0; i < G_N_ELEMENTS (time_menu_texts); i++) {
    if (strcmp (time_menu_texts[i], tmp) == 0) {
      gtk_combo_box_set_active (GTK_COMBO_BOX (p_data->combo_time), i);
      return;
    }
    if (strcmp (time_menu_texts[i], MENU_TEXT_EXACT) == 0) {
      gtk_combo_box_set_active (GTK_COMBO_BOX (p_data->combo_time), i);
      return;
    }
  }
}

void
kp_date_entry_set_date (KPDateEntry *entry, KPDate *date)
{
  KPDateEntryPrivateData *p_data;
  
  p_data = KP_DATE_ENTRY_PRIVATE_DATA (entry);
  
  g_return_if_fail (KP_IS_DATE_ENTRY (entry));
  g_return_if_fail (g_date_valid_dmy (date->d, date->m, date->y));

  p_data->date.d = date->d;
  p_data->date.m = date->m;
  p_data->date.y = date->y;

  update_display_widgets (entry);
}


static void
day_selected (GtkCalendar *calendar, KPDateEntry *entry)
{
  KPDateEntryPrivateData *p_data;
  KPDate date;
  
  p_data = KP_DATE_ENTRY_PRIVATE_DATA (entry);
  gtk_calendar_get_date (calendar, &date.y, &date.m, &date.d);
  date.m++;
  
  if (g_date_valid_dmy (date.d, date.m, date.y)) {
  
    p_data->date.d = date.d;
    p_data->date.m = date.m;
    p_data->date.y = date.y;

    g_signal_emit (G_OBJECT (entry), 
                   kp_date_entry_signals[DATE_SELECTED_SIGNAL],
                   0, &date);
  } else
    g_return_if_reached ();
    
  update_display_widgets (entry);
}



static void
button_toggled (GtkToggleButton *button, KPDateEntry *entry)
{
  KPDateEntryPrivateData *p_data;
  static gboolean visible = FALSE;
  GtkWidget *parent;
  GtkWidget *cal;
  GdkScreen *screen;
  gint x, y, wx, wy;
  static GtkWidget *dialog;
  
  p_data = KP_DATE_ENTRY_PRIVATE_DATA (entry);
 
  if (visible) {
    gtk_widget_destroy (dialog);
    visible = FALSE; 
    return;
  }
  
  parent = GTK_WIDGET (gtk_widget_get_toplevel (GTK_WIDGET (entry)));
  dialog = gtk_dialog_new_with_buttons ("Select a date!",
                                        GTK_WINDOW (parent),
                                        GTK_DIALOG_DESTROY_WITH_PARENT,
                                        NULL); 
  
  screen = gtk_widget_get_screen (GTK_WIDGET (entry));
  gtk_window_set_screen (GTK_WINDOW (dialog), screen);
  
  gdk_window_get_origin (GTK_WIDGET (button)->window, &x, &y);
  gdk_window_get_root_origin (GTK_WIDGET (button)->window, &wx, &wy); 
  gtk_window_move (GTK_WINDOW (dialog), x + wx, y + wy);
    
  gtk_window_stick (GTK_WINDOW (dialog));
  cal = gtk_calendar_new ();
  g_signal_connect (G_OBJECT (cal), "day-selected",
                    G_CALLBACK (day_selected), entry);
  gtk_calendar_set_display_options (GTK_CALENDAR (cal),
                                    GTK_CALENDAR_SHOW_HEADING |
                                    GTK_CALENDAR_SHOW_DAY_NAMES |
                                    GTK_CALENDAR_SHOW_WEEK_NUMBERS |
                                    GTK_CALENDAR_WEEK_START_MONDAY);
  
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), cal);

  gtk_window_set_decorated (GTK_WINDOW (dialog), FALSE);
  
  gtk_widget_show_all (dialog);
  visible = TRUE;
}

static void
time_menu_changed (GtkComboBox *box, KPDateEntry *dentry)
{
  KPDateEntryPrivateData *p_data;
  KPTime t;
  gchar *str;
  
  p_data = KP_DATE_ENTRY_PRIVATE_DATA (dentry);
 
  g_return_if_fail (GTK_IS_COMBO_BOX (box));
  
  str = get_combo_box_active_string (box, dentry);
  if (str == NULL)
    return;
  
  if (strcmp (str, _(MENU_TEXT_EXACT)) == 0) {
    gtk_widget_show (GTK_WIDGET (p_data->entry_time));
    return;
  } else {
    if (kp_date_entry_get_time (dentry, &t)) {
      g_print ("Time: %u:%u:%u\n", t.h, t.m, t.s);
      g_signal_emit (G_OBJECT (dentry), 
                     kp_date_entry_signals[TIME_SELECTED_SIGNAL], 0, &t);
    
    }
    gtk_widget_hide (GTK_WIDGET (p_data->entry_time));
  }
}




