/* $Id: hooks.c,v 1.18 2006/07/31 20:02:27 ekalin Exp $ */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <kcconfig.h>
#endif

#include <string.h>
#include <libintl.h>
#include <locale.h>
#include <gtk/gtk.h>
#include <glade/glade.h>

#include "kildclient.h"
#include "perlscript.h"


/*************************
 * File global variables *
 *************************/


/***********************
 * Function prototypes *
 ***********************/
static void     save_hook_list(FILE *fp, GSList *hookptr);



gboolean
connect_hook(World       *world,
             const gchar *hookname,
             int          pos,
             gchar       *action,
             gchar       *name,
             gboolean     enabled)
{
  GSList **hook_list;
  Hook    *hook;
  gint    *hook_pos;

  hook_list = get_hook_list_for_writing(world, hookname, &hook_pos);
  if (!hook_list) {
    ansitextview_append_stringf(world->gui,
                                _("hook: hook '%s' not found\n"),
                                hookname);
    return FALSE;
  }

  if (pos == -1) {
    hook = g_new0(Hook, 1);
    if (world->plugins_being_defined) {
      hook->owner_plugin = world->plugins_being_defined->data;
      *hook_list = g_slist_append(*hook_list, hook);
    } else {
      *hook_list = g_slist_insert(*hook_list, hook, *hook_pos);
    }
  } else {
    hook = (Hook *) g_slist_nth_data(*hook_list, pos);
    if (!hook) {
      ansitextview_append_string_nl(world->gui, _("No such hook"));
      return FALSE;
    }
  }

  if (action) {
    g_free(hook->action);
    hook->action = action;
  }
  if (name) {
    g_free(hook->name);
    hook->name = name;
  }
  hook->enabled = enabled;

  if (pos == -1) {
    if (!world->plugins_being_defined) {
      we_hook_insert_hook(world, hookname, hook, (*hook_pos)++);
    } else {
      we_hook_insert_hook(world, hookname, hook, g_slist_length(*hook_list));
    }
  } else {
    we_hook_update_hook(world, hookname, hook);
  }

  return TRUE;
}


gboolean
execute_hook(World *world, const gchar *hookname, const char *hookdata)
{
  PerlInterpreter *old_interpreter;
  GSList          *hookptr = NULL;
  Hook            *hook;
  SV              *Phookdata;

  hookptr = get_hook_list_for_reading(world, hookname, NULL);

  if (hookptr) {
    old_interpreter = PERL_GET_CONTEXT;
    PERL_SET_CONTEXT(world->perl_interpreter);

    gboolean dont_flush = world->dont_flush;
    world->dont_flush = TRUE;
    world_for_perl = world;

    if (hookdata) {
      Phookdata = get_sv("hookdata", TRUE);
      SvUTF8_on(Phookdata);
      sv_setpv(Phookdata, hookdata);
    }

    while (hookptr) {
      hook = (Hook *) hookptr->data;
      if (hook->enabled) {
        parse_commands(world, hook->action, strlen(hook->action));
      }
      hookptr = hookptr->next;
    }

    world_for_perl = currentWorld;
    world->dont_flush = dont_flush;
    PERL_SET_CONTEXT(old_interpreter);
  }
  return TRUE;
}


void
list_hooks(World *world, const gchar *hookname, Plugin *plugin)
{
  GSList *hooklist;
  Hook   *hook;
  guint   total_width;
  guint   field_width;
  int     i;

  hooklist = get_hook_list_for_reading(world, hookname, NULL);
  if (!hooklist) {
    ansitextview_append_string_nl(world->gui, _("Hook not found"));
    return;
  }

  /* The rows argument is not used. */
  ansitextview_get_size(world->gui, &field_width, &total_width);

  /* If the screen is really narrow, we can do nothing. */
  if (total_width < 14) {
    total_width = 14;
  }
  field_width = total_width - 9;

  ansitextview_append_stringf(world->gui, _("Hooks for '%s':\n"), hookname);
  ansitextview_append_string_nl(world->gui, _("Num Ena Action"));
  ansitextview_append_string(world->gui, "--- --- ");
  for (i = 0; i < field_width; ++i) {
    ansitextview_append_string(world->gui, "-");
  }
  ansitextview_append_string(world->gui, "\n");

  i = 0;
  while (hooklist) {
    hook = (Hook *) hooklist->data;
    if (hook->owner_plugin == plugin) {
      ansitextview_append_stringf(world->gui,
                                  "%3d %-3.3s %-*.*s\n",
                                  i,
                                  hook->enabled ? _("y") : _("n"),
                                  field_width, field_width, hook->action);
    }
    hooklist = hooklist->next;
    ++i;
  }
}


void
list_hook(World *world, const gchar *hookname, int pos)
{
  GSList *hooklist;
  Hook   *hook;

  hooklist = get_hook_list_for_reading(world, hookname, NULL);

  if (!hooklist) {
    ansitextview_append_string_nl(world->gui, _("Hook not found"));
    return;
  }
  hook = (Hook *) g_slist_nth_data(hooklist, pos);
  if (!hook) {
    ansitextview_append_string_nl(world->gui, _("No such hook"));
    return;
  }

  ansitextview_append_stringf(world->gui,
                              _("Hook number %d for event %s\n"
                                "  Name:    %s\n"
                                "  Action:  %s\n"
                                "  Enabled: %s\n"),
                              pos,
                              hookname,
                              hook->name ? hook->name : _("Not set"),
                              hook->action,
                              hook->enabled ? _("yes") : _("no"));
}


gboolean
delete_hook(World *world, const gchar *hookname, int pos)
{
  GSList **hooklist;
  GSList  *hook;
  gint    *hook_pos;

  hooklist = get_hook_list_for_writing(world, hookname, &hook_pos);
  if (!hooklist) {
    return FALSE;
  }

  hook = g_slist_nth(*hooklist, pos);
  if (!hook) {
    return FALSE;
  }

  we_hook_delete_hook(world, hookname, (Hook *) hook->data);

  *hooklist = g_slist_remove_link(*hooklist,
                                  hook);
  if (!((Hook *) hook->data)->owner_plugin) {
    --(*hook_pos);
  }
  free_hook((Hook *) hook->data, NULL);
  g_slist_free(hook);

  return TRUE;
}


gboolean
move_hook(World *world, const gchar *hookname, gint old_pos, gint new_pos)
{
  GSList  **hooklist;
  gint     *hook_pos;
  GSList   *hookitem;
  gpointer  hook;

  hooklist = get_hook_list_for_writing(world, hookname, &hook_pos);
  if (!hooklist) {
    return FALSE;
  }

  if (new_pos < 0 || new_pos >= *hook_pos) {
    new_pos = *hook_pos - 1;
  }

  hookitem = g_slist_nth(*hooklist, old_pos);
  if (!hookitem) {
    return FALSE;
  }
  hook = hookitem->data;

  *hooklist = g_slist_delete_link(*hooklist, hookitem);
  *hooklist = g_slist_insert(*hooklist, hook, new_pos);

  we_hook_delete_hook(world, hookname, hook);
  we_hook_insert_hook(world, hookname, hook, new_pos);

  return TRUE;
}


void
free_hook(Hook *hook, gpointer data)
{
  g_free(hook->name);
  g_free(hook->action);

  g_free(hook);
}


void
save_hooks(FILE *fp, World *world)
{
  fprintf(fp, "  <hooksv2>\n");

  if (world->hooks.OnConnect) {
    fprintf(fp, "    <hooklistv2 for=\"OnConnect\">\n");
    save_hook_list(fp, world->hooks.OnConnect);
  }

  if (world->hooks.OnDisconnect) {
    fprintf(fp, "    <hooklistv2 for=\"OnDisconnect\">\n");
    save_hook_list(fp, world->hooks.OnDisconnect);
  }

  if (world->hooks.OnReceivedText) {
    fprintf(fp, "    <hooklistv2 for=\"OnReceivedText\">\n");
    save_hook_list(fp, world->hooks.OnReceivedText);
  }

  if (world->hooks.OnSentCommand) {
    fprintf(fp, "    <hooklistv2 for=\"OnSentCommand\">\n");
    save_hook_list(fp, world->hooks.OnSentCommand);
  }

  if (world->hooks.OnGetFocus) {
    fprintf(fp, "    <hooklistv2 for=\"OnGetFocus\">\n");
    save_hook_list(fp, world->hooks.OnGetFocus);
  }

  if (world->hooks.OnLoseFocus) {
    fprintf(fp, "    <hooklistv2 for=\"OnLoseFocus\">\n");
    save_hook_list(fp, world->hooks.OnLoseFocus);
  }

  if (world->hooks.OnCloseConnected) {
    fprintf(fp, "    <hooklistv2 for=\"OnCloseConnected\">\n");
    save_hook_list(fp, world->hooks.OnCloseConnected);
  }


  fprintf(fp, "  </hooksv2>\n");
}


GSList **
get_hook_list_for_writing(World        *world,
                          const gchar  *hookname,
                          gint        **poscounter)
{
  if (strcmp(hookname, "OnConnect") == 0) {
    *poscounter = &world->hooks.OnConnect_pos;
    return &world->hooks.OnConnect;
  }
  if (strcmp(hookname, "OnDisconnect") == 0) {
    *poscounter = &world->hooks.OnDisconnect_pos;
    return &world->hooks.OnDisconnect;
  }
  if (strcmp(hookname, "OnReceivedText") == 0) {
    *poscounter = &world->hooks.OnReceivedText_pos;
    return &world->hooks.OnReceivedText;
  }
  if (strcmp(hookname, "OnSentCommand") == 0) {
    *poscounter = &world->hooks.OnSentCommand_pos;
    return &world->hooks.OnSentCommand;
  }
  if (strcmp(hookname, "OnGetFocus") == 0) {
    *poscounter = &world->hooks.OnGetFocus_pos;
    return &world->hooks.OnGetFocus;
  }
  if (strcmp(hookname, "OnLoseFocus") == 0) {
    *poscounter = &world->hooks.OnGetFocus_pos;
    return &world->hooks.OnLoseFocus;
  }
  if (strcmp(hookname, "OnCloseConnected") == 0) {
    *poscounter = &world->hooks.OnCloseConnected_pos;
    return &world->hooks.OnCloseConnected;
  }

  return NULL;
}


GSList *
get_hook_list_for_reading(World       *world,
                          const gchar *hookname,
                          gint        *pos)
{
  gint    *dummy;
  GSList **hooklist = get_hook_list_for_writing(world, hookname, &dummy);

  if (hooklist) {
    if (pos) {
      *pos = *dummy;
    }
    return *hooklist;
  }

  return NULL;
}


GtkTreeModel *
get_hook_model(World *world, const gchar *hookname)
{
  if (strcmp(hookname, "OnConnect") == 0) {
    return world->hooks.OnConnect_model;
  }
  if (strcmp(hookname, "OnDisconnect") == 0) {
    return world->hooks.OnDisconnect_model;
  }
  if (strcmp(hookname, "OnReceivedText") == 0) {
    return world->hooks.OnReceivedText_model;
  }
  if (strcmp(hookname, "OnSentCommand") == 0) {
    return world->hooks.OnSentCommand_model;
  }
  if (strcmp(hookname, "OnGetFocus") == 0) {
    return world->hooks.OnGetFocus_model;
  }
  if (strcmp(hookname, "OnLoseFocus") == 0) {
    return world->hooks.OnLoseFocus_model;
  }
  if (strcmp(hookname, "OnCloseConnected") == 0) {
    return world->hooks.OnCloseConnected_model;
  }

  return NULL;
}


static
void
save_hook_list(FILE *fp, GSList *hookptr)
{
  Hook *hook;

  while (hookptr) {
    hook = (Hook *) hookptr->data;
    if (!hook->owner_plugin) {
      save_hook(fp, hook);
    }
    hookptr = hookptr->next;
  }
  fprintf(fp, "    </hooklistv2>\n");
}


void
save_hook(FILE *fp, Hook *hook)
{
  fprintf(fp, "      <hookv2 ");
  if (hook->name) {
    fprintf_escaped(fp, "name=\"%s\" ", hook->name);
  }
  fprintf_escaped(fp, "enabled=\"%d\">%s</hookv2>\n",
                  hook->enabled,
                  hook->action);
}
