/*
    This file is part of the KDE project.
    Copyright (c) 2006-2007 Friedrich W. H. Kossebau <kossebau@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301  USA
*/


// kde
#include <kdebug.h>
#include <kabc/addressee.h>
#include <dcopclient.h>
#include <kconfig.h>
#include <ktrader.h>
#include <kparts/componentfactory.h>
#include <ksycoca.h>
// lib
#include "propertyadapter.h"
#include "services_p.h"


namespace Khalkhi {

static const char DCOPName[] = "Services";
static const char DCOPConfigObjectName[] = "KhalkhiConfig";
static const char DCOPChangeSignalName[] = "changed()";
static const char DCOPServiceConfigChangeSignalName[] = "changed(int,QString,QString)";

static const char ConfigFileName[] =      "khalkhirc";
static const char GeneralGroupId[] =      "General";
static const char PropertiesGroupId[] =   "Properties";
static const char PropertiesSortingId[] = "Sorting";
static const char PropertiesHiddenId[] =  "Hidden";

static const char PropertyGroupIdTemplate[] = "Property:%1";
static const char ActionsSortingId[] =        "ActionsSorting";
static const char ActionsHiddenId[] =         "HiddenActions";
static const char ActionsMainId[] =           "MainActions";
static const char DataActionsSortingId[] =    "DataActionsSorting";
static const char DataActionsHiddenId[] =     "HiddenDataActions";
static const char DataActionsMainId[] =       "MainDataActions";
static const char StatesSortingId[] =         "StatesSorting";
static const char StatesHiddenId[] =          "HiddenStates";

static const char ABIVersionId[] =            "X-KDE-ABI-Version";
static const char OnlyShowInContextId[] =     "X-KDE-OnlyShowInContext";
static const char NotShowInContextId[] =      "X-KDE-NotShowInContext";
static const char CategoriesId[] =            "X-KDE-Categories";

static const char ServiceABIVersion[] = "1";

ServicesPrivate::ServicesPrivate( Services *S )
: DCOPObject( DCOPName ), Base( S )
{
    connectDCOPSignal( 0, DCOPConfigObjectName, DCOPChangeSignalName,
                       "onConfigChange()", false );
    connectDCOPSignal( 0, DCOPConfigObjectName, DCOPServiceConfigChangeSignalName,
                       "onServiceConfigChange(int,QString,QString)", false );

    connect( KSycoca::self(), SIGNAL(databaseChanged()), SLOT(onKSyCoCaChange()) );
}

void ServicesPrivate::onConfigChange()
{
    Manager.removeActionService( QString::null, true );
    Manager.removeDataActionService( QString::null, true );

    for( PropertyManagerList::Iterator ManagerIt = PropertyManagers.begin();
         ManagerIt != PropertyManagers.end(); ++ManagerIt )
    {
        (*ManagerIt)->removeActionService( QString::null, true );
        (*ManagerIt)->removeDataActionService( QString::null, true );
        (*ManagerIt)->removeStatusService( QString::null, true );
    }
    PropertyManagers.clear();//TODO: clear client lists, too?

    load();

    emit Base->changed();
}

void ServicesPrivate::onServiceConfigChange( int ServiceType, QString PropertyId, QString ServiceId )
{
    if( PropertyId.isNull() )
    {
        Manager.reloadConfig( ServiceType, ServiceId );
    }
    else
    {
        PropertyManager *Manager = PropertyManagers[PropertyId];

        if( Manager )
            Manager->reloadConfig( ServiceType, ServiceId );
    }
}

void ServicesPrivate::loadActionServices( const QStringList &SortingIds,
                                                          const QStringList &HiddenIds )
{
    KTrader::OfferList ServiceOffers = KTrader::self()->query( "khalkhi/actionservice" );
    for( KTrader::OfferListIterator it = ServiceOffers.begin(); it != ServiceOffers.end(); ++it )
    {
        const KService::Ptr Plugin = *it;

        const QString ServiceId = Plugin->property( QString::fromLatin1("X-KDE-ActionService") ).toString();
        kdDebug() << "ActionService "<<ServiceId<<" found: "<<Plugin->library() << endl;

        const QString Version = Plugin->property( QString::fromLatin1(ABIVersionId) ).toString();
        if( Version != QString::fromLatin1(ServiceABIVersion) )
        {
            kdDebug() << "ActionService version incompatible: "
                <<Version<< "!="<< QString::fromLatin1(ServiceABIVersion)<< endl;
            continue;
        }

        if( HiddenIds.contains(ServiceId) )
        {
            kdDebug() << "ActionService hidden: "<<ServiceId << endl;
            continue;
        }

        QStringList Arguments( ServiceId );
        Arguments.append( Plugin->property( QString::fromLatin1(OnlyShowInContextId) ).toStringList().join(";") );
        Arguments.append( Plugin->property( QString::fromLatin1(NotShowInContextId) ).toStringList().join(";") );
        ActionService *Service =
            KParts::ComponentFactory::createInstanceFromService<ActionService>( Plugin, 0, 0, Arguments );

        if( Service )
        {
            kdDebug() << "Plugin loaded with Arguments: "<< Arguments<< endl;
//             bool AsGlobal = MainActionIds[PropertyId].contains( ServiceId );
            Manager.addActionService( Service/*, AsGlobal*/ );
        }
        else
            kdDebug() << "Plugin not loadable: "<<ServiceId << endl;
    }
    Manager.setActionServiceSorting( SortingIds );
}

void ServicesPrivate::loadDataActionServices( const QStringList &SortingIds,
                                                              const QStringList &HiddenIds )
{
    KTrader::OfferList ServiceOffers = KTrader::self()->query( "khalkhi/dataactionservice" );
    for( KTrader::OfferListIterator it = ServiceOffers.begin(); it != ServiceOffers.end(); ++it )
    {
        const KService::Ptr Plugin = *it;

        const QString ServiceId = Plugin->property( QString::fromLatin1("X-KDE-DataActionService") ).toString();
        kdDebug() << "DataActionService "<<ServiceId<<" found: "<<Plugin->library() << endl;

        const QString Version = Plugin->property( QString::fromLatin1(ABIVersionId) ).toString();
        if( Version != QString::fromLatin1(ServiceABIVersion) )
        {
            kdDebug() << "DataActionService version incompatible: "
                <<Version<< "!="<< QString::fromLatin1(ServiceABIVersion)<< endl;
            continue;
        }

        if( HiddenIds.contains(ServiceId) )
        {
            kdDebug() << "DataActionService hidden: "<<ServiceId << endl;
            continue;
        }

        QStringList Arguments( ServiceId );
        Arguments.append( Plugin->property( QString::fromLatin1(OnlyShowInContextId) ).toStringList().join(";") );
        Arguments.append( Plugin->property( QString::fromLatin1(NotShowInContextId) ).toStringList().join(";") );
        DataActionService *Service =
            KParts::ComponentFactory::createInstanceFromService<DataActionService>( Plugin, 0, 0, Arguments );

        if( Service )
        {
            kdDebug() << "Plugin loaded: "<<Service->id() << endl;
//             bool AsGlobal = MainActionIds[PropertyId].contains( ServiceId );
            Manager.addDataActionService( Service/*, AsGlobal*/ );
        }
        else
            kdDebug() << "Plugin not loadable: "<<ServiceId << endl;
    }
    Manager.setDataActionServiceSorting( SortingIds );
}

void ServicesPrivate::loadStatusServices( const QStringList &SortingIds,
                                                          const QStringList &HiddenIds )
{
    KTrader::OfferList ServiceOffers = KTrader::self()->query( "khalkhi/statusservice" );
    for( KTrader::OfferListIterator it = ServiceOffers.begin(); it != ServiceOffers.end(); ++it )
    {
        const KService::Ptr Plugin = *it;

        const QString ServiceId = Plugin->property( QString::fromLatin1("X-KDE-StatusService") ).toString();
        kdDebug() << "StatusService "<<ServiceId<<" found: "<<Plugin->library() << endl;

        const QString Version = Plugin->property( QString::fromLatin1(ABIVersionId) ).toString();
        if( Version != QString::fromLatin1(ServiceABIVersion) )
        {
            kdDebug() << "StatusService version incompatible: "
                <<Version<< "!="<< QString::fromLatin1(ServiceABIVersion)<< endl;
            continue;
        }

        if( HiddenIds.contains(ServiceId) )
        {
            kdDebug() << "StatusService hidden: "<<ServiceId << endl;
            continue;
        }

        QStringList Arguments( ServiceId );
        Arguments.append( Plugin->property( QString::fromLatin1(OnlyShowInContextId) ).toStringList().join(";") );
        Arguments.append( Plugin->property( QString::fromLatin1(NotShowInContextId) ).toStringList().join(";") );
        StatusService *Service =
            KParts::ComponentFactory::createInstanceFromService<StatusService>( Plugin, 0, 0, Arguments );

        if( Service )
        {
            kdDebug() << "Plugin loaded: "<<Service->id() << endl;
//             bool AsGlobal = MainActionIds[PropertyId].contains( ServiceId );
            Manager.addStatusService( Service/*, AsGlobal*/ );
        }
        else
            kdDebug() << "Plugin not loadable: "<<ServiceId << endl;
    }
    Manager.setStatusServiceSorting( SortingIds );
}

// TODO: versioning support, by writing the v no from the header into a call, which gets tested
void ServicesPrivate::load()
{
    KConfig Config( ConfigFileName );

    Config.setGroup( PropertiesGroupId );
    const QStringList SortedPropertyIds = Config.readListEntry( PropertiesSortingId );
    const QStringList HiddenPropertyIds = Config.readListEntry( PropertiesHiddenId );
    loadProperties( SortedPropertyIds, HiddenPropertyIds );

    Config.setGroup( GeneralGroupId );
    const QStringList ActionSortingIds = Config.readListEntry( ActionsSortingId );
    const QStringList HiddenActionIds = Config.readListEntry( ActionsHiddenId );
    loadActionServices( ActionSortingIds, HiddenActionIds );
    const QStringList DataActionSortingIds = Config.readListEntry( DataActionsSortingId );
    const QStringList HiddenDataActionIds = Config.readListEntry( DataActionsHiddenId );
    loadDataActionServices( DataActionSortingIds, HiddenDataActionIds );
    const QStringList StatusSortingIds = Config.readListEntry( StatesSortingId );
    const QStringList HiddenStatusIds = Config.readListEntry( StatesHiddenId );
    loadStatusServices( StatusSortingIds, HiddenStatusIds );

    QMap<QString,QStringList> ActionSortingIdsMap;
    QMap<QString,QStringList> HiddenActionIdsMap;
    QMap<QString,QStringList> MainActionIdsMap;

    QMap<QString,QStringList> DataActionSortingIdsMap;
    QMap<QString,QStringList> HiddenDataActionIdsMap;
    QMap<QString,QStringList> MainDataActionIdsMap;

    QMap<QString,QStringList> StatesSortingIdsMap;
    QMap<QString,QStringList> HiddenStatesIdsMap;

    for( PropertyManagerList::ConstIterator ManagerIt = PropertyManagers.begin();
         ManagerIt != PropertyManagers.end(); ++ManagerIt )
    {
        const QString &PropertyId = (*ManagerIt)->id();

        Config.setGroup( QString::fromLatin1(PropertyGroupIdTemplate).arg(PropertyId) );

        ActionSortingIdsMap[PropertyId] = Config.readListEntry( ActionsSortingId );
        HiddenActionIdsMap[PropertyId] =  Config.readListEntry( ActionsHiddenId );
        MainActionIdsMap[PropertyId] =    Config.readListEntry( ActionsMainId );

        DataActionSortingIdsMap[PropertyId] = Config.readListEntry( DataActionsSortingId );
        HiddenDataActionIdsMap[PropertyId] =  Config.readListEntry( DataActionsHiddenId );
        MainDataActionIdsMap[PropertyId] =    Config.readListEntry( DataActionsMainId );

        StatesSortingIdsMap[PropertyId] = Config.readListEntry( StatesSortingId );
        HiddenStatesIdsMap[PropertyId] =  Config.readListEntry( StatesHiddenId );
    }
    loadActionServices( ActionSortingIdsMap, HiddenActionIdsMap, MainActionIdsMap );
    loadDataActionServices( DataActionSortingIdsMap, HiddenDataActionIdsMap, MainDataActionIdsMap );
    loadStatusServices( StatesSortingIdsMap, HiddenStatesIdsMap );
}


void ServicesPrivate::loadProperties( const QStringList &SortedPropertyIds, const QStringList &HiddenPropertyIds )
{
    KTrader::OfferList PropertyOffers = KTrader::self()->query( "khalkhi/property" );

    for( KTrader::OfferListIterator it = PropertyOffers.begin(); it != PropertyOffers.end(); ++it )
    {
        const KService::Ptr Plugin = *it;
        kdDebug() << "Property found: "<<(**it).library() << endl;

        const QString PropertyId = Plugin->property( QString::fromLatin1("X-KDE-KhalkhiProperty") ).toString();

        const QString Version = Plugin->property( QString::fromLatin1(ABIVersionId) ).toString();
        if( Version != QString::fromLatin1(ServiceABIVersion) )
        {
            kdDebug() << "Property version incompatible: "
                <<Version<< "!="<< QString::fromLatin1(ServiceABIVersion)<< endl;
            continue;
        }

        if( HiddenPropertyIds.contains(PropertyId) )
            kdDebug() << "Property hidden: "<<PropertyId << endl;
        else
        {
            PropertyAdapter *Adapter =
                KParts::ComponentFactory::createInstanceFromService<PropertyAdapter>( Plugin, 0, 0, PropertyId );
            if( Adapter )
            {
                kdDebug() << "Property loaded: "<<Adapter->id() << endl;
                PropertyManagers.append( new PropertyManager(Adapter) );
            }
            else
                kdDebug() << "Property not loadable: "<<PropertyId << endl;
        }
    }
    PropertyManagers.setOrder( SortedPropertyIds );
}

void ServicesPrivate::loadActionServices( const QMap<QString,QStringList> &ActionSortingIds,
                                                   const QMap<QString,QStringList> &HiddenActionIds,
                                                   const QMap<QString,QStringList> &MainActionIds )
{
    KTrader::OfferList ServiceOffers = KTrader::self()->query( "khalkhi/propertyactionservice" );
    for( KTrader::OfferListIterator it = ServiceOffers.begin(); it != ServiceOffers.end(); ++it )
    {
        const KService::Ptr Plugin = *it;

        const QString PropertyId = Plugin->property( QString::fromLatin1("X-KDE-KhalkhiProperty") ).toString();
        const QString ServiceId = Plugin->property( QString::fromLatin1("X-KDE-ActionService") ).toString();
        kdDebug() << "ActionService "<<ServiceId<<" found: "<<Plugin->library() << " for " << PropertyId << endl;

        const QString Version = Plugin->property( QString::fromLatin1(ABIVersionId) ).toString();
        if( Version != QString::fromLatin1(ServiceABIVersion) )
        {
            kdDebug() << "ActionService version incompatible: "
                <<Version<< "!="<< QString::fromLatin1(ServiceABIVersion)<< endl;
            continue;
        }

        PropertyManagerList::ConstIterator ManagerIt = PropertyManagers.find( PropertyId );
        if( ManagerIt == PropertyManagers.end() )
        {
            kdDebug() << "Property not loaded: "<<PropertyId << endl;
            continue;
        }
        if( HiddenActionIds[PropertyId].contains(ServiceId) )
        {
            kdDebug() << "ActionService hidden: "<<ServiceId << endl;
            continue;
        }
        QStringList Arguments( ServiceId );
        Arguments.append( Plugin->property( QString::fromLatin1(OnlyShowInContextId) ).toStringList().join(";") );
        Arguments.append( Plugin->property( QString::fromLatin1(NotShowInContextId) ).toStringList().join(";") );
        PropertyActionService *Service =
            KParts::ComponentFactory::createInstanceFromService<PropertyActionService>( Plugin, 0, 0, Arguments );

        if( Service )
        {
            // hacky
            const bool AsMain = 
                ActionSortingIds[PropertyId].contains( ServiceId ) ? // already configured?
                    MainActionIds[PropertyId].contains( ServiceId ) :
                    !Plugin->property( QString::fromLatin1(CategoriesId) ).toStringList().isEmpty(); // abuse categories
            (*ManagerIt)->addActionService( Service, AsMain );
            kdDebug() << (AsMain?"Plugin loaded as Main: ":"Plugin loaded: ") <<Service->id() << endl;
        }
        else
            kdDebug() << "Plugin not loadable: "<<ServiceId << endl;
    }
    for( PropertyManagerList::Iterator ManagerIt = PropertyManagers.begin();
         ManagerIt != PropertyManagers.end(); ++ManagerIt )
        (*ManagerIt)->setActionServiceSorting( ActionSortingIds[(*ManagerIt)->id()] );
}

void ServicesPrivate::loadDataActionServices( const QMap<QString,QStringList> &DataActionSortingIds,
                                                       const QMap<QString,QStringList> &HiddenDataActionIds,
                                                       const QMap<QString,QStringList> &MainDataActionIds )
{
    KTrader::OfferList ServiceOffers = KTrader::self()->query( "khalkhi/propertydataactionservice" );
    for( KTrader::OfferListIterator it = ServiceOffers.begin(); it != ServiceOffers.end(); ++it )
    {
        const KService::Ptr Plugin = *it;

        const QString PropertyId = Plugin->property( QString::fromLatin1("X-KDE-KhalkhiProperty") ).toString();
        const QString ServiceId = Plugin->property( QString::fromLatin1("X-KDE-DataActionService") ).toString();
        kdDebug() << "DataActionService "<<ServiceId<<" found: "<<Plugin->library() << " for " << PropertyId << endl;

        const QString Version = Plugin->property( QString::fromLatin1(ABIVersionId) ).toString();
        if( Version != QString::fromLatin1(ServiceABIVersion) )
        {
            kdDebug() << "DataActionService version incompatible: "
                <<Version<< "!="<< QString::fromLatin1(ServiceABIVersion)<< endl;
            continue;
        }

        PropertyManagerList::ConstIterator ManagerIt = PropertyManagers.find( PropertyId );
        if( ManagerIt == PropertyManagers.end() )
        {
            kdDebug() << "Property not loaded: "<<PropertyId << endl;
            continue;
        }
        if( HiddenDataActionIds[PropertyId].contains(ServiceId) )
        {
            kdDebug() << "DataActionService hidden: "<<ServiceId << endl;
            continue;
        }

        QStringList Arguments( ServiceId );
        Arguments.append( Plugin->property( QString::fromLatin1(OnlyShowInContextId) ).toStringList().join(";") );
        Arguments.append( Plugin->property( QString::fromLatin1(NotShowInContextId) ).toStringList().join(";") );
        PropertyDataActionService *Service =
            KParts::ComponentFactory::createInstanceFromService<PropertyDataActionService>( Plugin, 0, 0, Arguments );
        if( Service )
        {
            // hacky
            const bool AsMain = 
                DataActionSortingIds[PropertyId].contains( ServiceId ) ? // already configured?
                    MainDataActionIds[PropertyId].contains( ServiceId ) :
                    !Plugin->property( QString::fromLatin1(CategoriesId) ).toStringList().isEmpty(); // abuse categories
            (*ManagerIt)->addDataActionService( Service, AsMain );
            kdDebug() << (AsMain?"Plugin loaded as Main: ":"Plugin loaded: ")<<Service->id() << endl;
        }
        else
            kdDebug() << "Plugin not loadable: "<<ServiceId << endl;
    }
    for( PropertyManagerList::Iterator ManagerIt = PropertyManagers.begin();
         ManagerIt != PropertyManagers.end(); ++ManagerIt )
        (*ManagerIt)->setDataActionServiceSorting( DataActionSortingIds[(*ManagerIt)->id()] );
}

void ServicesPrivate::loadStatusServices( const QMap<QString,QStringList> &StatesSortingIds,
                                                   const QMap<QString,QStringList> &HiddenStatesIds )
{
    KTrader::OfferList StatusServiceOffers = KTrader::self()->query( "khalkhi/propertystatusservice" );
    for( KTrader::OfferListIterator it = StatusServiceOffers.begin(); it != StatusServiceOffers.end(); ++it )
    {
        const KService::Ptr Plugin = *it;

        const QString PropertyId = Plugin->property( QString::fromLatin1("X-KDE-KhalkhiProperty") ).toString();
        const QString ServiceId = Plugin->property( QString::fromLatin1("X-KDE-StatusService") ).toString();
        kdDebug() << "StatusService "<<ServiceId<<" found: "<<Plugin->library() << " for " << PropertyId << endl;

        const QString Version = Plugin->property( QString::fromLatin1(ABIVersionId) ).toString();
        if( Version != QString::fromLatin1(ServiceABIVersion) )
        {
            kdDebug() << "StatusService version incompatible: "
                <<Version<< "!="<< QString::fromLatin1(ServiceABIVersion)<< endl;
            continue;
        }

        PropertyManagerList::ConstIterator ManagerIt = PropertyManagers.find( PropertyId );
        if( ManagerIt == PropertyManagers.end() )
        {
            kdDebug() << "Property not loaded: "<<PropertyId << endl;
            continue;
        }
        if( HiddenStatesIds[PropertyId].contains(ServiceId) )
        {
            kdDebug() << "StatusService hidden: "<<ServiceId << endl;
            continue;
        }

        QStringList Arguments( ServiceId );
        Arguments.append( Plugin->property( QString::fromLatin1(OnlyShowInContextId) ).toStringList().join(";") );
        Arguments.append( Plugin->property( QString::fromLatin1(NotShowInContextId) ).toStringList().join(";") );
        PropertyStatusService *Service =
            KParts::ComponentFactory::createInstanceFromService<PropertyStatusService>( Plugin, 0, 0, Arguments );
        if( Service )
        {
            kdDebug() << "StatusService loaded: "<<Service->id() << endl;
            (*ManagerIt)->addStatusService( Service );
        }
        else
            kdDebug() << "StatusService not loadable: "<<ServiceId << endl;
    }
    for( PropertyManagerList::Iterator ManagerIt = PropertyManagers.begin();
         ManagerIt != PropertyManagers.end(); ++ManagerIt )
        (*ManagerIt)->setStatusServiceSorting( StatesSortingIds[(*ManagerIt)->id()] );
}

void ServicesPrivate::addManager( PropertyManager* Manager )
{
    PropertyManagers.append( Manager );

    for( PropertyStatusServiceClientList::Iterator ClientIt = StatusServiceClients.begin();
         ClientIt != StatusServiceClients.end(); ++ClientIt )
        Manager->registerClient( *ClientIt );

    for( AllPropertiesGlobalActionServiceClientList::Iterator ClientIt = GlobalActionServiceClients.begin();
         ClientIt != GlobalActionServiceClients.end(); ++ClientIt )
        Manager->registerClient( *ClientIt );

    informStatusServiceClients();
    informAllPropertiesGlobalActionServiceClients();
}


void ServicesPrivate::removeManager( const QString &PropertyId, bool Delete )
{
    PropertyManagers.remove( PropertyId, Delete );

    informStatusServiceClients();
    informAllPropertiesGlobalActionServiceClients();
}


void ServicesPrivate::informStatusServiceClients()
{
    // inform clients
    for( PropertyStatusServiceClientList::Iterator ClientIt = StatusServiceClients.begin();
         ClientIt != StatusServiceClients.end(); ++ClientIt )
        (*ClientIt)->onPropertyManagerChange();
}

void ServicesPrivate::informAllPropertiesGlobalActionServiceClients()
{
    // inform clients
    for( AllPropertiesGlobalActionServiceClientList::Iterator ClientIt = GlobalActionServiceClients.begin();
         ClientIt != GlobalActionServiceClients.end(); ++ClientIt )
        (*ClientIt)->onPropertyManagerChange();
}

void ServicesPrivate::onKSyCoCaChange()
{
#if 0
    if( KSycoca::self()->isChanged( "services" ) )
    {
    // ksycoca has new KService objects for us, make sure to update
    // our 'copies' to be in sync with it. Not important for OK, but
    // important for Apply (how to differentiate those 2?).
    // See BR 35071.
    QValueList<TypesListItem *>::Iterator it = m_itemsModified.begin();
    for( ; it != m_itemsModified.end(); ++it ) {
        QString name = (*it)->name();
        if ( removedList.find( name ) == removedList.end() ) // if not deleted meanwhile
            (*it)->refresh();
    }
    m_itemsModified.clear();
    }
#endif
}


ServicesPrivate::~ServicesPrivate()
{
}

}

#include "services_p.moc"
