//
//  kjots
//
//  Copyright (C) 1997 Christoph Neerfeld <Christoph.Neerfeld@home.ivm.de>
//  Copyright (C) 2002, 2003 Aaron J. Seigo <aseigo@kde.org>
//  Copyright (C) 2003 Stanislav Kljuhhin <crz@hot.ee>
//  Copyright (C) 2005-2006 Jaison Lee <lee.jaison@gmail.com>
//  Copyright (C) 2007 Stephen Kelly <steveire@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//

//Own Header
#include "kjotsedit.h"

#include <QMimeData>
#include <QTextCursor>
#include <QStackedWidget>
#include <QUrl>
#include <QMenu>
#include <QContextMenuEvent>

#include <kaction.h>
#include <kactioncollection.h>
#include <kfontdialog.h>

#include "kjotsentry.h"
#include "bookshelf.h"

KJotsEdit::KJotsEdit ( QWidget *parent ) : KTextEdit(parent)
{
    setAcceptRichText(true);
    setWordWrapMode(QTextOption::WordWrap);
}

KJotsEdit::~KJotsEdit()
{
}

void KJotsEdit::contextMenuEvent( QContextMenuEvent *event )
{
    QMenu *popup = createStandardContextMenu();
    connect( popup, SIGNAL( triggered ( QAction* ) ),
             this, SLOT( menuActivated( QAction* ) ) );

    popup->addSeparator();
    popup->addAction(actionCollection->action("change_font"));
    popup->addAction(actionCollection->action("copyIntoTitle"));

    popup->exec( event->globalPos() );

    delete popup;
}

void KJotsEdit::DelayedInitialization ( KActionCollection *collection, Bookshelf *shelf )
{
    bookshelf = shelf;
    actionCollection = collection;

    connect(actionCollection->action("italics"), SIGNAL(triggered()), SLOT(onItalics()));
    connect(actionCollection->action("underline"), SIGNAL(triggered()), SLOT(onUnderline()));
    connect(actionCollection->action("bold"), SIGNAL(triggered()), SLOT(onBold()));
    connect(actionCollection->action("strike_out"), SIGNAL(triggered()), SLOT(onStrikeOut()));
    connect(actionCollection->action("linkify"), SIGNAL(triggered()), SLOT(onLinkify()));

    connect(actionCollection->action("auto_bullet"), SIGNAL(triggered()), SLOT(onAutoBullet()));
    connect(actionCollection->action("horizontal_rule"), SIGNAL(triggered()), SLOT(onRule()));
    connect(actionCollection->action("change_font"), SIGNAL(triggered()), SLOT(onFonts()));

    connect(bookshelf, SIGNAL(itemSelectionChanged()), SLOT(onBookshelfSelection()));
    connect(this, SIGNAL(selectionChanged()), SLOT(onSelectionChanged()));
    connect(this, SIGNAL(textChanged()), SLOT(onTextChanged()));

    connect(this, SIGNAL(currentCharFormatChanged(const QTextCharFormat &)),
            this, SLOT(updateCurrentCharFormat(const QTextCharFormat &)));

}

void KJotsEdit::disableEditing ( void )
{
    if ( currentPage ) {
        currentPage->setCursor(textCursor());
        setDocument(0);
        currentPage = 0;
    }

    setReadOnly(true);
    setEnabled(false);
}

void KJotsEdit::onBookshelfSelection ( void )
{
    QList<QTreeWidgetItem*> selection = bookshelf->selectedItems();
    int selectionSize = selection.size();

    if (selectionSize !=  1) {
        disableEditing();
    } else {
        KJotsPage *newPage = dynamic_cast<KJotsPage*>(selection[0]);
        if ( !newPage ) {
            disableEditing();
        } else {
            if ( currentPage != newPage ) {
                if ( currentPage ) {
                    currentPage->setCursor(textCursor());
                }
                currentPage = newPage;

                setEnabled(true);
                setReadOnly(false);
                setDocument(currentPage->body());
                if ( !currentPage->getCursor().isNull() ) {
                    setTextCursor(currentPage->getCursor());
                }

                QStackedWidget *stack = static_cast<QStackedWidget*>(parent());
                stack->setCurrentWidget(this);
                setFocus();

                if ( textCursor().atStart() )
                {
                    // Reflect formatting when switching pages and the first word is formatted
                    // Work-around for qrextedit bug. The format does not seem to exist
                    // before the first character. Submitted to qt-bugs
                    moveCursor(QTextCursor::Right);
                    moveCursor(QTextCursor::Left);
                }

            }
        }
    }
}

void KJotsEdit::onItalics ( void )
{
    QTextCharFormat fmt;
    fmt.setFontItalic(actionCollection->action("italics")->isChecked());
    mergeFormatOnWordOrSelection(fmt);
}

void KJotsEdit::onUnderline ( void )
{
    QTextCharFormat fmt;
    fmt.setFontUnderline(actionCollection->action("underline")->isChecked());
    mergeFormatOnWordOrSelection(fmt);
}

void KJotsEdit::onBold ( void )
{
    QTextCharFormat fmt;
    fmt.setFontWeight(actionCollection->action("bold")->isChecked() ? QFont::Bold : QFont::Normal);
    mergeFormatOnWordOrSelection(fmt);
}

void KJotsEdit::onStrikeOut ( void )
{
    QTextCharFormat fmt;
    fmt.setFontStrikeOut(actionCollection->action("strike_out")->isChecked() ? true : false);
    mergeFormatOnWordOrSelection(fmt);
}

void KJotsEdit::updateCurrentCharFormat(const QTextCharFormat &format)
{
    fontChanged(format.font());
}

void KJotsEdit::fontChanged(const QFont &f)
{
    actionCollection->action("italics")->setChecked(f.italic());
    actionCollection->action("bold")->setChecked(f.bold());
    actionCollection->action("underline")->setChecked(f.underline());
    actionCollection->action("strike_out")->setChecked(f.strikeOut());
}

void KJotsEdit::mergeFormatOnWordOrSelection(const QTextCharFormat &format)
{
    QTextCursor cursor = this->textCursor();
    if (!cursor.hasSelection())
        cursor.select(QTextCursor::WordUnderCursor);
    cursor.mergeCharFormat(format);
    this->mergeCurrentCharFormat(format);
}

void KJotsEdit::onAutoBullet ( void )
{
    KTextEdit::AutoFormatting currentFormatting = autoFormatting();

    if ( currentFormatting == KTextEdit::AutoBulletList ) {
        setAutoFormatting(KTextEdit::AutoNone);
        actionCollection->action("auto_bullet")->setChecked( false );
    } else {
        setAutoFormatting(KTextEdit::AutoBulletList);
        actionCollection->action("auto_bullet")->setChecked( true );
    }
}

void KJotsEdit::onRule ( void )
{
    QTextCursor c = textCursor();
    QTextBlockFormat bf = c.blockFormat();
    QTextCharFormat cf = c.charFormat();
    c.insertHtml("<hr>");
    c.insertBlock(bf, cf);
    setTextCursor(c);
}

void KJotsEdit::onFonts ( void )
{
    QFont newFont;
    int result = KFontDialog::getFont( newFont );
    if ( result == KFontDialog::Accepted ) {
        setCurrentFont(newFont);
    }
}

void KJotsEdit::onLinkify ( void )
{
    QTextCursor selectionCursor = textCursor();
    Q_ASSERT(selectionCursor.hasSelection());

    QString linkified = QString("<a href='%1'>%2</a> ")
        .arg(selectionCursor.selectedText()).arg(selectionCursor.selectedText());

    insertHtml(linkified);
}

void KJotsEdit::onSelectionChanged ( void )
{
    QTextCursor selectionCursor = textCursor();
    if ( selectionCursor.hasSelection() ) {
        QUrl selectedUrl ( selectionCursor.selectedText() );
        if ( !selectedUrl.isEmpty() && selectedUrl.isValid() ) {
            actionCollection->action("linkify")->setEnabled(true);
            return;
        }
    }

    actionCollection->action("linkify")->setEnabled(false);
}

void KJotsEdit::onTextChanged ( void )
{
    if ( currentPage ) {
        currentPage->parentBook()->setDirty(true);
    }
}

bool KJotsEdit::canInsertFromMimeData ( const QMimeData *source ) const
{
    if ( source->formats().contains("kjots/internal_link") ) {
        return true;
    } else if ( source->hasUrls() ) {
        return true;
    } else {
        return KTextEdit::canInsertFromMimeData(source);
    }
}

void KJotsEdit::insertFromMimeData ( const QMimeData *source )
{
    if ( source->formats().contains("kjots/internal_link") ) {
        insertHtml(source->data("kjots/internal_link"));
    } else if ( source->hasUrls() ) {
        foreach ( QUrl url, source->urls() ) {
            if ( url.isValid() ) {
                QString html = QString ( "<a href='%1'>%2</a> " )
                    .arg(QString::fromUtf8(url.toEncoded()))
                    .arg(url.toString(QUrl::RemovePassword));
                insertHtml(html);
            }
        }
    } else {
        KTextEdit::insertFromMimeData(source);
    }
}

#include "kjotsedit.moc"
/* ex: set tabstop=4 softtabstop=4 shiftwidth=4 expandtab: */
/* kate: tab-indents off; replace-tabs on; tab-width 4; remove-trailing-space on; encoding utf-8;*/
