/***************************************************************************
 *   Copyright (C) 2006 by Rohan McGovern                                  *
 *   rohan.pm@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <kdebug.h>
#include <klocale.h>
#include <stdexcept>
#include <qvbox.h>
#include <qlabel.h>

#include "dbusobject.h"
#include "dbusinterface.h"
#include "dbusservice.h"
#include "dbusutil.h"
#include "dbus/qdbusproxy.h"
#include "dbus/qdbusmessage.h"
#include "dbus/qdbuserror.h"

class DBusObject::Private {
public:
    /**
     * Introspect self.
     * @returns Data for this object returned by Introspectable.Introspect.
     */
    QDomElement introspect()
    throw( QDBusSendError, QDBusXmlError );

    QString discoverPath( QDomElement const & elem );

    QDBusProxy * proxy;
    DBusObject * p;
};

DBusObject::DBusObject(
  QListViewItem * parent,
  QDomElement const & elem,
  QDBusProxy * proxy
) throw( QDBusSendError, QDBusXmlError )
 : DBusItem( parent, "/", "Object" ),
   d( new DBusObject::Private() )
{
    d->proxy = proxy;
    d->p = this;

    QDomElement myElem = elem;

    ensureElementIsNamed( myElem, "node" );

    m_object = d->discoverPath( myElem );
    m_service = discoverService();

    // If no name is given, then this is the root object '/'
    if ( myElem.hasAttribute( "name" ) && myElem.attribute( "name" ) != "" ) {
        setText( 0, myElem.attribute( "name" ) );
    }

    /* The spec says this: if an object is described at all, then it is
     * described fully.  If it is not described at all, then it may
     * be necessary to Introspect that object.  Of course, we know the
     * "/" object is always fully introspected, because that's always the
     * first thing we introspect.
     */
    if ( m_object != "/" && !myElem.hasChildNodes() )
        myElem = d->introspect();

    // At this point we are guaranteed to be introspected down
    // to the next level.  Child objects may need further inspection, but they
    // will handle that in their own constructors.

    // It's possible we're an empty object.  If so, we're done.
    if ( !myElem.hasChildNodes() )
        return;

    // Now iterate through self, enumerating interfaces, properties, and
    // other objects.
    myElem = myElem.firstChild().toElement();

    try {
        do {
            if ( myElem.tagName() == "node" )
                new DBusObject( this, myElem, d->proxy );
            else if ( myElem.tagName() == "interface" )
                new DBusInterface( this, myElem );
            else
                kdWarning() << "Unhandled node in introspect data: "
                            << myElem.tagName() << endl;
            myElem = myElem.nextSibling().toElement();
        } while ( !myElem.isNull() );
    }
    catch ( std::runtime_error const & e ) {
        m_ok = false;
        m_error = e.what();
    }
}



QDomElement DBusObject::Private::introspect()
throw ( QDBusSendError, QDBusXmlError ) {

    proxy->setService( p->m_service );
    proxy->setPath( p->m_object );
    proxy->setInterface( "org.freedesktop.DBus.Introspectable" );

    QDBusMessage reply = proxy->sendWithReply(
      "Introspect",
      QValueList<QVariant>()
    );
    if ( reply.type() == QDBusMessage::InvalidMessage )
        throw QDBusSendError(
            QString("%1: %2")
            .arg( proxy->lastError().name() )
            .arg( proxy->lastError().message() )
        );

    QDomDocument doc;
    if ( !doc.setContent( reply[0].toString() ) )
        throw QDBusXmlError(
            i18n(
              "Do not translate 'Introspect'.",
              "XML parse error in reply to Introspect"
            )
        );

    QDomElement docElem = doc.documentElement();
    DBusItem::ensureElementIsNamed( docElem, "node" );

    return docElem;
}

QString DBusObject::Private::discoverPath( QDomElement const & elem ) {

    // Only root object may omit name
    if ( !elem.hasAttribute( "name" )
       || elem.attribute( "name" ) == ""
       || elem.attribute( "name" ) == "/" )
        return "/";

    // Append our relative path parent's path
    QListViewItem * parent = p->parent();
    if ( !parent || !dynamic_cast< DBusObject * >(parent) )
        throw std::logic_error(
            "DBusObject is not root object, but is not child of "
            "another DBusObject!"
        );

    DBusObject * obj = dynamic_cast< DBusObject * >(parent);

    if ( obj->object() == "/" )
        return "/" + elem.attribute( "name" );
    else
        return obj->object() + "/" + elem.attribute( "name" );

}

QWidget * DBusObject::widget( QWidget * parent ) const {
    QScrollView * scrollview = new QScrollView( parent );
    QVBox * vbox = new QVBox( scrollview->viewport() );
    scrollview->addChild( vbox );
    scrollview->setMidLineWidth( 0 );
    scrollview->setLineWidth( 0 );
    scrollview->setResizePolicy( QScrollView::AutoOneFit );

    new QLabel(
      i18n( "Object: %1" ).arg( m_object ),
      vbox
    );

    new QLabel(
      i18n( "Service: %1" ).arg( m_service ),
      vbox
    );

    addErrorInfo( vbox );

    return scrollview;
}

DBusObject::~DBusObject() {
    delete d;
}
