/***************************************************************************
 *   Copyright (C) 2006 by Rohan McGovern                                  *
 *   rohan.pm@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "dbusmethod.h"
#include "dbusmethodargument.h"
#include "dbusutil.h"
#include "dbustreewidget.h"

#include <kdebug.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <klineedit.h>

#include <qlabel.h>
#include <qscrollview.h>
#include <qtextedit.h>
#include <qvbox.h>
#include <qstring.h>

class DBusMethod::Private {
public:
    // Helper: adds 'execute' button to widget
    void placeButton( QWidget * widget );
    QPtrList< KLineEdit > inputBoxes;
    QPtrList< QTextEdit > outputBoxes;
    DBusMethod * parent;
};

DBusMethod::DBusMethod(
  QListViewItem * parent,
  QDomElement const & elem
) throw( QDBusXmlError )
  : DBusMethodOrSignal( parent, elem, "Method" ),
    d( new DBusMethod::Private() )
{
    d->parent = this;
}

QPtrList< DBusMethodArgument > DBusMethod::inArgs() const {

    QPtrList< DBusMethodArgument > inArgs = m_args;
    while ( 0 != inArgs.current() ) {
        if ( inArgs.current()->direction() != "in" )
            inArgs.take();
        inArgs.next();
    }

    return inArgs;
}

QPtrList< DBusMethodArgument > DBusMethod::outArgs() const {

    QPtrList< DBusMethodArgument > outArgs = m_args;
    while ( 0 != outArgs.current() ) {
        if ( outArgs.current()->direction() != "out" )
            outArgs.take();
        outArgs.next();
    }

    return outArgs;
}


QWidget * DBusMethod::widget( QWidget * parent ) const {
    d->inputBoxes.clear();
    d->outputBoxes.clear();

    QScrollView * scrollview = new QScrollView( parent );
    QVBox * vbox = new QVBox( scrollview->viewport() );
    scrollview->addChild( vbox );
    scrollview->setMidLineWidth( 0 );
    scrollview->setLineWidth( 0 );
    scrollview->setResizePolicy( QScrollView::AutoOneFit );

    new QLabel(
      i18n( "Method: %1" ).arg( m_signature ),
      vbox
    );

    new QLabel(
      i18n( "Interface: %1" ).arg( m_interface ),
      vbox
    );

    new QLabel(
      i18n( "Object: %1" ).arg( m_object ),
      vbox
    );

    new QLabel(
      i18n( "Service: %1" ).arg( m_service ),
      vbox
    );

    QPtrList< DBusMethodArgument > inArgs = this->inArgs();
    QPtrList< DBusMethodArgument > outArgs = this->outArgs();

    /// FIXME use a better way of making space in the layout
    new QLabel( vbox );

    if ( inArgs.count() > 0 ) {
        new QLabel( i18n( "<b>Input:</b>" ), vbox );
    }

    for ( unsigned int i = 0; i < inArgs.count(); i++ ) {
        QHBox * hbox = new QHBox( vbox );
        new QLabel( hbox );
        (new QLabel(
          inArgs.at(i)->niceType() + " " + inArgs.at(i)->name(),
          hbox 
        ))->setMargin( 5 );
        new QLabel( "   ", hbox );

        KLineEdit * myEdit = new KLineEdit( hbox );
        d->inputBoxes.append( myEdit );
        QObject::connect(
          myEdit,
          SIGNAL(returnPressed()),
          dynamic_cast< DBusTreeWidget * >(
            listView()->parentWidget()->parentWidget()
          )->dispatcher(),
          SLOT(executeMethod())
        );
        new QLabel( "   ", hbox );

        // If we're at the end, and there are no output args, place button
        if ( outArgs.count() == 0 && inArgs.count() == i+1 ) {
            d->placeButton( hbox );
            new QLabel( "   ", hbox );
        } // if ( outArgs.count() == 0 && inArgs.count() == i+1 )

        
    }




    if ( outArgs.count() > 0 ) {
        new QLabel( i18n( "<b>Output:</b>" ), vbox );
    }




    for ( unsigned int i = 0; i < outArgs.count(); i++ ) {
        QHBox * hbox = new QHBox( vbox );
        new QLabel( hbox );
        (new QLabel(
          outArgs.at(i)->niceType() + " " + outArgs.at(i)->name(),
          hbox 
        ))->setMargin( 5 );

        new QLabel( "   ", hbox );

        QTextEdit * myEdit = new QTextEdit( hbox );
        myEdit->setReadOnly( true );
        myEdit->setTextFormat( Qt::PlainText );
        myEdit->setFixedHeight( 24 );
        d->outputBoxes.append( myEdit );

        new QLabel( "   ", hbox );

        // If we're at the end and haven't placed button, then do so now
        if ( outArgs.count() == i+1 ) {
            d->placeButton( hbox );
            new QLabel( "   ", hbox );
        } // if ( outArgs.count() == i+1 )


    }


    // If there are no input or output args, we didn't add button above...
    if ( inArgs.count() == 0 && outArgs.count() == 0 ) {
        QHBox * hbox = new QHBox( vbox );
        d->placeButton( hbox );
        new QLabel( hbox );
        new QLabel( hbox );
        new QLabel( hbox );
        new QLabel( hbox );
        new QLabel( hbox );
        new QLabel( hbox );
        new QLabel( hbox );
    } // if ( inArgs.count() == 0 && outArgs.count() == 0 )


    addAnnotationInfo( vbox );
    addErrorInfo( vbox );

    vbox->adjustSize();
    scrollview->adjustSize();

    return scrollview;
}



void DBusMethod::Private::placeButton( QWidget * widget ) {

    KPushButton * myButton = new KPushButton(
        i18n("Execute Method"),
        widget
    );

    if ( !dynamic_cast< DBusTreeWidget * >(
      parent->listView()->parentWidget()->parentWidget()
    ) )
        throw std::logic_error(
          "DBusMethod is in a list which is not child of DBusTreeWidget!"
        );

    if ( !QObject::connect(
      myButton,
      SIGNAL(clicked()),
      dynamic_cast< DBusTreeWidget * >(
        parent->listView()->parentWidget()->parentWidget()
      )->dispatcher(),
      SLOT(executeMethod())
    ) )
        throw std::runtime_error("Couldn't connect button to dispatcher!");
}



QPtrList< KLineEdit > DBusMethod::inputBoxes() const {
    return d->inputBoxes;
}

QPtrList< QTextEdit > DBusMethod::outputBoxes() const {
    return d->outputBoxes;
}

DBusMethod::~DBusMethod() {
    delete d;
}
