/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

#include "KayaAPI.h"
#include "Heap.h"
#include "Array.h"
#include "VM.h"
#include "VMState.h"
#include <cstdlib>
#include <stdarg.h>

KValue newKayaValue()
{
    return MKVAL(NULL,KVT_NULL);
}

KValue KayaInt(kint x)
{
    KValue v = newKayaValue();
    KayaSetInt(v,x);
    return v;
}

KValue KayaPtr(void* x)
{
    KValue v = newKayaValue();
    KayaSetInt(v,(kint)x);
    return v;
}

KValue KayaChar(wchar_t c)
{
    KValue v = newKayaValue();
    KayaSetChar(v,c);
    return v;
}

KValue KayaFloat(double f)
{
    KValue v = newKayaValue();
    KayaSetFloat(v,f);
    return v;
}

KValue KayaString(wchar_t* str)
{
    KValue v = newKayaValue();
    KayaSetString(v,str);
    return v;
}

KValue KayaCharString(char* str)
{
    KValue v = newKayaValue();
    KayaSetCharString(v,str);
    return v;
}

kint KayaGetInt(KValue v)
{
    return v->getInt();
}

wchar_t KayaGetChar(KValue v)
{
    return (wchar_t)(v->getInt());
}

double KayaGetFloat(KValue v)
{
    return v->getReal();
}

wchar_t* KayaGetString(KValue v)
{
    return v->getString()->getVal();
}

void KayaSetInt(KValue v,kint i)
{
    v->setInt(i);
}

void KayaSetChar(KValue v,wchar_t c)
{
    v->setInt((int)c);
}

void KayaSetFloat(KValue v,double f)
{
    v->setReal(f);
}

void KayaSetString(KValue v,wchar_t* s)
{
    v->setString(NEWSTRING(s));
}

void KayaSetCharString(KValue v,char* s)
{
    v->setString(NEWSTRING(strtowc(s)));
}


KArray newKayaArray(kint size)
{
    return new Array(size);
}

KArray KayaGetArray(KValue v)
{
    return v->getArray();
}

void KayaSetArray(KValue v, KArray a)
{
    v->setArray(a);
}


void KayaArrayPush(KArray a,KValue v)
{
    a->push_back(v);
}

void KayaArraySet(KArray a, kint index, KValue v)
{
    Value* vloc = a->lookup(index);
    vloc->setPtr(v);
}

KValue KayaArrayLookup(KArray a, kint index)
{
    return a->lookup(index);
}

kint KayaArraySize(KArray a)
{
    return a->size();
}

KValue KayaCall(KValue fn, kint args, ...)
{
    VMState* vm = initstack();
//    Value* arg;
    Value** arglist = new Value*[args];
    va_list argp;
    va_start(argp,args);
    kint i;
    // Gah! Need to push backwards!
    for(i=0;i<args;++i) {
	arglist[args-i-1] = va_arg(argp,Value*);
    }
    va_end(argp);
    for(i=0;i<args;++i) {
	PUSH(arglist[i]);
    }
    delete arglist;
    TRY(cbexcept);
    CALLFUN(fn);
    if (vm->emptyStack()) {
	return NULL;
    }
    else
	return vm->doPop();

    LABEL(cbexcept); // There was an exception in the callback. This is fatal.
    cerr << "Exception in callback: ";
    Exception* except = vm->doPop()->getExcept();
    except->show();
    exit(1);
}

KValue KayaUnion(kint tag, kint args)
{
  if (args == 0) {
    return MKVAL((void*)tag,KVT_CUNION);
  } else {
    Union* u = new Union(NULL,args,false);
    return MKVAL((void*)u,KVT_UNION+tag);
  }
}

KValue KayaArrayVal(KArray ar)
{
    KValue v = newKayaValue();
    KayaSetArray(v,ar);
    return v;
}

kint KayaUnionGetTag(KValue v)
{
  kint vt = v->getType();
  if (vt >= KVT_UNION) {
    return vt-KVT_UNION;
  } else if (vt == KVT_CUNION) {
    return v->getInt();
  } else {
    return -1; 
// shouldn't happen if API used correctly, but API users should error
// check for it.
  }
}

KValue KayaUnionGetArg(KValue v,kint i)
{
    Union* u = v->getUnion();
    return u->args[i];
}

void KayaUnionSetArg(KValue v,kint i,KValue val)
{
    Union* u = v->getUnion();
    u->args[i] = val;
}

void* KayaAlloc(kint size)
{
    // Just use libgc...
    return GC_MALLOC_ATOMIC(size);
}

kint orArray(KArray a, kint(*func)(KValue))
{
    kint val = 0;
    kint size = KayaArraySize(a);
    for(kint i=0;i<size;++i) {
	KValue v = KayaArrayLookup(a,i);
	val = val | func(v);
    }
    return val;
}
