{-
    Kaya - My favourite toy language.
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

module TAC where

-- This module implements Three Address Code (sort of). 
-- Exports compileAll, which compiles well typed expressions to
-- bytecode (using fcompile).
-- Also some helpers for generating raw C.

import Language
import Debug.Trace
import List
import KeyGen
import Dataflow

data CompileResult = RawCode String  -- Raw C, eg for #includes
		   | ByteCode (Name,Type,Code) -- Function name and bytecode 
		   | GlobCode Int -- Global variable
		   | ExternDef (Name, Type)
                   | ExcCode Name Bool
   deriving Show

type Var = (Name, Bool) -- Bool now unused, used to mean whether it went on the stack or heap...
type Tmp = Int

data TAC = DECLARE Var
	 | DECLAREARG Var
	 | DECLAREQUICK Var
         | HEAPVAL Var -- Convert a stack allocated var to a heap allocated var
	 | TMPINT Tmp
	 | TMPREAL Tmp
	 | TMPVAL Var
	 | ARRAY Var
	 | SET Var Int Var -- Int is number of indices to get off stack
	 | TOINDEX
	 | SETRV -- Set the return value. Like settop, but doesn't discard both
         | SETRVREAL Tmp
	 | SETTOP
	 | ADDTOP
	 | SUBTOP
	 | MULTOP
	 | DIVTOP
	 | MKARRAY Int
	 | USETMP Var -- Flag that Var was used, do nothing
	 | DEREF
	 | TMPSET Tmp Int
	 | RTMPSET Tmp Double
	 | CALL Var
	 | CALLNAME Mangled
         | FASTCALL Mangled [Var]
	 | CALLTOP
	 | TAILCALL Var
	 | TAILCALLNAME Mangled
	 | TAILCALLTOP
	 | CLOSURE Mangled Int Int
	 | CLOSURELOC Var Int 
	 | CLOSURETOP Int 
	 | FOREIGNCALL String String Type [Type] -- name, library, type, args
	 | MKCON Int Int -- Create a constructor, tag and arity
	 | MKCONZERO Int -- Create a constructor, tag with arity zero
	 | MKCONRV Int Int -- Create a constructor, tag and arity, set as retrn value
	 | MKCONZERORV Int -- Create a constructor, tag with arity zero, set as return value
	 | MKEXCEPT -- Create an exception from top two stack items
	 | MKNEWEXCEPT Name Int -- Create an exception with name and arity
	 | EQEXCEPT -- Check two exceptions for equality
	 | NEEXCEPT -- Check two exceptions for inequality
	 | EQSTRING -- Check two strings for equality
	 | NESTRING -- Check two stringss for inequality
	 | EQSTRINGW String -- Check string for equality with stack top
	 | NESTRINGW String -- Check string for inequality with stack top
         | JEQSTRING Name -- jump if two string equal
         | JNESTRING Name -- jump if two string not equal
         | JEQSTRINGW Name String -- jump if string equal to stack top
         | JNESTRINGW Name String -- jump if string not equal to stack top
         | JEXNE Name Name -- jump if top exception not equal to stack top
	 | GETFUN Var Var
	 | GETVAL Tmp
	 | GETRVAL Tmp
	 | PUSHGETVAL PushItem Tmp
	 | PUSHGETRVAL PushItem Tmp
	 | INTGETRVAL Tmp
	 | GETINDEX
         | PUSHGETINDEX PushItem
         | PUSHTOINDEX PushItem
	 | PROJARG Int Int -- Project first arg, check tag is second arg.
	 | SETPROJARG PushItem Int Int PushItem -- Project second arg, check tag is third arg, from first to fourth arg
	 | EXPROJARG Var Int -- Project into first arg from second arg of exception
	 | INFIX Tmp Op Tmp Tmp
         | ADDINPLACE PushItem Int
         | SUBINPLACE PushItem Int
	 | REALINFIX Tmp Op Tmp Tmp
	 | REALINFIXBOOL Op Tmp Tmp
         | INFIXJFALSE Op Tmp Tmp Name
         | INFIXJTRUE Op Tmp Tmp Name
         | UNARYJFALSE UnOp Tmp Name
         | UNARYJTRUE UnOp Tmp Name
	 | INTPOWER Tmp Tmp Tmp
	 | REALPOWER Tmp Tmp Tmp
	 | UNARY Tmp UnOp Tmp
	 | REALUNARY Tmp UnOp Tmp
	 | APPEND
         | APPENDCHAR Int
         | APPENDTMP Tmp
         | APPENDSTR String
         | APPENDTOP
         | APPENDTOPINT
	 | PRINTINT
	 | PRINTSTR
	 | PRINTEXC
	 | NEWLINE
	 | LABEL Name
	 | JUMP Name
	 | JFALSE Name
	 | JTRUE Name
	 | JTFALSE Tmp Name
	 | JTTRUE Tmp Name
	 | TRY Name -- label to jump to on failure
	 | TRIED -- Pop last item from try stack
	 | THROW -- Throw an exception
	 | RESTORE -- Restore stack state to before the exception
	 | CASE [[TAC]] -- Get tag from top stack item.
	 | CONSTCASE PrimType [(Const, [TAC])] [TAC]
	 | PROJ Var Int -- Project an argument from top stack item into Var
	 | PUSHGLOBAL String Int
	 | CREATEGLOBAL String Int
	 | PUSH PushItem
	 | PUSH2 PushItem PushItem
	 | PUSH3 PushItem PushItem PushItem
	 | PUSH4 PushItem PushItem PushItem PushItem
         | STACKINT Var Int -- Allocate int on the stack (in Var) and push it
         | STACKTMP Var Tmp -- Allocate int on the stack (in Var) and push it
         | STACKREAL Var PushItem -- Allocate int on the stack (in Var) and push it
         | STACKSTR Var PushItem -- Allocate int on the stack (in Var) and push it
         | TMPSETTOP Int
	 | PUSHSETTOP PushItem
         | SETLOOKUP Int Var
	 | RETURN
	 | SETVAL Var Int
	 | SETINT Var Tmp
	 | PUSHSETINT PushItem Tmp
	 | SETVAR Var Var
	 | GETLENGTH
	 | POP Var
	 | POPARG Var
	 | POPANDCOPYARG Var
         | REMEMBER Var
         | NOTEVAR Var
         | COPYARG Var
	 | POPINDEX Var
	 | DISCARD
	 | STR2INT
	 | INT2STR
	 | REAL2STR
	 | BOOL2STR
	 | STR2REAL
	 | STR2CHR
	 | CHR2STR
	 | INT2REAL
	 | REAL2INT
	 | VMPTR
         | STRLENGTH
	 | DUMMY
	 | ERROR
	   -- Debug info
	 | LINENO String Int
	 | PUSHBT String String Int
	 | INLAM String
	 | POPBT
	 | CHECKCACHE Int
	 | STORECACHE [Var]
   deriving Show

data PushItem = NAME Mangled Int -- Remember the arity for making closure space
	      | VAL Int
	      | RVAL Double
	      | STR String
	      | INT Tmp
	      | REAL Tmp
	      | VAR Var
              | EMPTYSTR
   deriving (Show, Eq)

loc i = "l" ++ show i
tmp i = "t" ++ show i
tmpval i = UN ("tv" ++ show i)

type Local = Int

comp :: Monad m => Program -> Expr Name -> Bool -> Name -> m Code
comp ds e pure mod = do (Code pop args code) <- fcompile ds e pure mod
			return (Code [] args (peephole' (decls (peephole (pop++code)))))

-- Metadata for compilation of an expression
data CompData = CD {
		    modulename :: Name,
		    locs :: [Var],
		    tmpvar :: Int,
		    breaklabel :: Maybe Name,
		    tries :: Int,
		    cacheargs :: Maybe Int, -- number to cache
		    alldecls :: Program,
                    varargs :: [Var],
                    -- All we've learned from dataflow analysis
                    varInfo :: [Int] -- just unused variables currently
		   }

data Code = Code {
                  argPop :: [TAC], -- argument popping code
                  argNames :: [(ArgType, Var)],
                  mainCode :: [TAC] -- main body
                 }
   deriving Show

getUnused = id -- for now

unusedVar :: Int -> CompData -> Bool
unusedVar v cd = v `elem` (getUnused (varInfo cd))

-- Decorate names looked up from the context with their de bruijn index
getVar :: [Var] -> Int -> Var
getVar xs i | i<length xs = decorate (xs!!i) i
            | otherwise = error $ "Name lookup error, this is probably a bug: " ++ (show xs) ++ show i

decorate nm i = case nm of
                 (UN n, b) -> (UN (n++show i), b)
                 (MN n, b) -> (MN n, b)

infixl 5 !>

(!>) = getVar

fcompile :: Monad m => Program -> Expr Name -> Bool -> Name -> m Code
fcompile ds (Annotation a e) pure mod =
   do (Code pop args def) <- fcompile ds e pure mod
      (acode, lab, _) <- annotcompile a def 0 0
      return (Code pop args acode)
fcompile ds (Lambda ivs args sc) pure mod 
    = do let vs' = getvarargs 0 (zip ivs (map fst args))
         let (alllocs, vinfo) = findInfo sc
         let unused = neverUsed alllocs vinfo
         (code,lab',stmp') <- fcompile' sc 
                        (CD mod 
                            (map (\x -> (fst x, False)) args)
                            0 Nothing 0 (updatecache cache (length args)) ds vs' unused) 
                          0 0
	 let acode = acompile ivs args
	 let cachecode = case cache of
			    Nothing -> []
			    (Just x) -> [CHECKCACHE (length args)]
	 return $ Code acode (aget ivs args) (cachecode ++ code ++ remember vs' ++ [DISCARD])
  where updatecache Nothing l = Nothing
	updatecache (Just _) l = Just l
        getvarargs i [] = []
        getvarargs i ((Var,x):xs) = (decorate (x, False) i):getvarargs (i+1) xs
        getvarargs i (_:xs) = getvarargs (i+1) xs
        cache | pure = Just 0
              | otherwise = Nothing
fcompile ds Noop pure mod = return $ Code [] [] []
fcompile ds NoInit pure mod = return $ Code [] [] [DISCARD]
fcompile ds exp pure mod = fail $ "No function! Can't happen! " ++ show exp

-- fcompile' needs to keep track of:
--   local variable names ([Var])
--   labels used so far (first int, also returned)
--   temporary variables in the current block (second int)

-- It should probably also take the code so far in an accumulator, as there's
-- a lot of recursion going on. Or maybe ghc takes care of that if we're really
-- lucky....

-- FIXME: Need to add a flag for whether the compiled thing has to go on
-- the heap or not - the current hack leads to horrible code duplication, but
-- it's too late at night to do anything better for now. (Specifically, the
-- If and Case cases are duplicated)

-- Special case where we need to make sure we're making a heap allocated thing.
fMkHeap (Loc l) cd lab stmp = return (hpalloc++[PUSH (VAR ((locs cd)!>l))], lab, stmp)
    where hpalloc | snd ((locs cd)!!l) == False = [HEAPVAL ((locs cd)!>l)]
                  | otherwise = []
fMkHeap (GConst c) _ lab stmp = return ((ccompile c), lab, stmp)
fMkHeap (Infix op x y) cd lab stmp = icompile True (Infix op x y) cd lab stmp
fMkHeap (Unary op x) cd lab stmp = icompile True (Unary op x) cd lab stmp
fMkHeap (If c t e) cd lab stmp
     = do (ccode, lab'', stmp') <- fcompile' c cd lab' stmp
	  (tcode, lab''', stmp'') <- fMkHeap t cd lab'' stmp'
	  (ecode, lab'''', stmp''') <- fMkHeap e cd lab''' stmp''
	  return (ccode ++ [JFALSE false] ++ tcode ++
		  [JUMP end,LABEL false] ++ ecode ++ [LABEL end], lab'''', stmp''')
  where false = MN ("l",lab)
	end = MN ("l",lab+1)
	lab' = lab+2
fMkHeap (Coerce t1 t2 v) cd lab stmp = 
   do (vcode,lab',stmp') <- fMkHeap v cd lab stmp
      cocompile t1 t2 vcode cd lab' stmp'
fMkHeap (Case v alts) cd lab stmp = 
   do (vcode,lab',stmp') <- fcompile' v cd lab stmp
      (acode,lab'',stmp'') <- altcompile True alts cd lab' stmp'
      return (vcode ++ acode, lab'', stmp'')
fMkHeap (Annotation a e) cd lab stmp =
   do (ecode, lab', stmp') <- fMkHeap e cd lab stmp
      annotcompile a ecode lab' stmp'
fMkHeap x cd lab stmp = fcompile' x cd lab stmp


fRetcompile (Annotation a e) cd lab stmp =
   do (ecode, lab', stmp') <- fRetcompile e cd lab stmp
      annotcompile a ecode lab' stmp'
fRetcompile (Apply f as) cd lab stmp
    = do (pcode, lab', stmp') <- pushargs as cd lab stmp
	 (ccode, lab'', stmp'') <- tailcall f cd lab' stmp'
	 return (pcode ++ ccode, lab'', stmp)
fRetcompile x cd lab stmp = 
   do (xcode, lab', stmp') <- fcompile' x cd lab stmp
      return (xcode ++ [SETRV], lab', stmp')

-- Decide on a function based on whether the result goes on the heap or not
fHeapComp True = fMkHeap
fHeapComp False = fcompile'

{-
argcompile :: Monad m => Expr Name -> CompData -> Int -> Int -> m ([TAC], Int, Int)
argcompile (Annotation a e) cd lab stmp =
   do (ecode, lab', stmp') <- argcompile e cd lab stmp
      annotcompile a ecode lab' stmp'

    where atmpname = ((MN ("stmp",stmp)),True)
                                          
argcompile x cd lab stmp = fcompile' x cd lab stmp
-}

-- first Int is the label
-- second is the stacktmp number
-- most expressions reset the stacktmp number to the initial once
-- they're finished, but the labels are unique within a function
fcompile' :: Monad m => Expr Name -> CompData -> Int -> Int -> m ([TAC], Int, Int)
fcompile' (Global n mangling arity) _ lab stmp
    = return ([PUSH (NAME (cname n mangling) arity)], lab, stmp)
fcompile' (Loc l) cd lab stmp = return ([PUSH (VAR ((locs cd)!>l))], lab, stmp)
fcompile' (GVar i) cd lab stmp = return ([PUSHGLOBAL (show (modulename cd)) i], lab, stmp)
fcompile' (GConst c) _ lab stmp = ccompile_stack c stmp lab
fcompile' (Closure _ _ _) cd lab stmp = fail "Encountered a closure. Something is probably broken."
fcompile' (Bind n t v sc) (CD mod vars tmp b tr cache ds vs vinfo) lab stmp
    = do (vcode, lab', stmp') <- fcompile' v (CD mod vars tmp b tr cache ds vs vinfo) lab stmp
         (sccode, lab'', stmp'') <- fcompile' sc (CD mod (vars++[(n,False)]) tmp b tr cache ds vs vinfo) lab' stmp'
         return ((vcode ++ (POP (decorate (n,False) (length vars))):sccode), lab'', stmp)
fcompile' (Declare f l n t sc) (CD mod vars tmp b tr cache ds vs vinfo) lab stmp
    = fcompile' sc (CD mod (vars++[n]) tmp b tr cache ds vs vinfo) lab stmp
-- modify this to handle tail calls by using a new fRetcompile rather than
-- fMkHeap
fcompile' (Return v) cd lab stmp
    = do (vcode, lab', stmp') <- fRetcompile v cd lab stmp
	 return (vcode ++ (take (tries cd) (repeat TRIED)) ++ 
        	[POPBT] ++ (storecache (cacheargs cd) (locs cd)) ++ 
		remember (varargs cd) ++ [RETURN], lab', stmp)
  where storecache Nothing _ = []
	storecache (Just n) vs = [STORECACHE (take n vs)]
fcompile' (VoidReturn) cd lab stmp
    = return (remember (varargs cd) ++ [POPBT, DISCARD, RETURN], lab, stmp)
fcompile' (Assign l v) cd lab stmp
   -- if it's not used, evaluate v in case it's side effecting, but
   -- discard the result (might save a bit of stack space)
     = do let unused = case l of 
                 (AName var) -> unusedVar var cd
                 _ -> False
          (code, lab', stmp') <- (complex l) v cd lab stmp
	  (fpcode, lab'', stmp'') <- findplace l lab' stmp'
          if unused 
             then return (code ++ [DISCARD], lab'', stmp)
             else return (code ++ fpcode ++ [SETTOP], lab'', stmp)
  where -- complex (AGlob _) = fMkHeap
        complex _ = fcompile'
        findplace (AName i) lab stmp = return ([PUSH (VAR ((locs cd)!>i))], lab, stmp)
	findplace (AGlob i) lab stmp = return ([PUSHGLOBAL (show (modulename cd)) i], lab, stmp)
	findplace (AIndex l i) lab stmp
            = do (fpcode,lab', stmp') <- findplace l lab stmp
		 (code,lab'', stmp'') <- fcompile' i cd lab' stmp'
		 return (fpcode ++ code ++ [TOINDEX], lab'', stmp'')
	findplace (AField l f a t) lab stmp
            = do (fpcode,lab',stmp') <- findplace l lab stmp
		 return (fpcode ++ [PROJARG a t], lab',stmp')
fcompile' (AssignOp op l v) cd lab stmp
     = do (code, lab', stmp') <- complex l v cd lab stmp
	  (fpcode, lab'', stmp'') <- findplace l lab' stmp'
	  return (fpcode ++ code ++ [doToTop op], lab'', stmp)
  where -- complex (AGlob _) = fMkHeap
        complex _ = fcompile'
        doToTop Plus = ADDTOP
	doToTop Minus = SUBTOP
	doToTop Times = MULTOP
	doToTop Divide = DIVTOP
	findplace (AName i) lab stmp = return ([PUSH (VAR ((locs cd)!>i))], lab, stmp)
	findplace (AGlob i) lab stmp = return ([PUSHGLOBAL (show (modulename cd)) i], lab, stmp)
	findplace (AIndex l i) lab stmp
            = do (fpcode,lab',stmp') <- findplace l lab stmp
		 (code,lab'',stmp'') <- fcompile' i cd lab' stmp'
		 return (fpcode ++ code ++ [TOINDEX], lab'',stmp'')
	findplace (AField l f a t) lab stmp
            = do (fpcode,lab',stmp') <- findplace l lab stmp
		 return (fpcode ++ [PROJARG a t], lab',stmp')
fcompile' (AssignApp l v) cd lab stmp
     = do (code, lab', stmp') <- complex l v cd lab stmp
	  (fpcode, lab'', stmp'') <- findplace l lab' stmp'
	  return (fpcode ++ code ++ [APPENDTOP, DISCARD], lab'',stmp)
  where -- complex (AGlob _) = fMkHeap
        complex _ = fcompile'
        findplace (AName i) lab stmp = return ([PUSH (VAR ((locs cd)!>i))], lab, stmp)
	findplace (AGlob i) lab stmp = return ([PUSHGLOBAL (show (modulename cd)) i], lab, stmp)
	findplace (AIndex l i) lab stmp
            = do (fpcode,lab',stmp') <- findplace l lab stmp
		 (code,lab'',stmp'') <- fcompile' i cd lab' stmp'
		 return (fpcode ++ code ++ [TOINDEX], lab'',stmp'')
	findplace (AField l f a t) lab stmp
            = do (fpcode,lab',stmp') <- findplace l lab stmp
		 return (fpcode ++ [PROJARG a t], lab', stmp')
--	   (code ++ [POP (MN ("tmp",0))] ++ fpcode, lab'')
{-
  where findplace (AName i) j lab = ([SET (vars!>i) j (MN ("tmp",0))], lab)
	findplace (AIndex l i) j lab 
	    = let (code, lab') = fcompile' i vars lab tmp
		  (fpcode, lab'') = findplace l (j+1) lab' in
		  (code ++ fpcode, lab'')
	findplace (AField l f a t) j lab = error "Broken" -}


--fcompile' (IndexAssign l i v) vars lab
--     = fcompile' v vars lab ++ fcompile' i vars lab ++ [POPINDEX (vars!>l)]
fcompile' (Seq x y) cd lab stmp
    = do (xcode, lab', stmp') <- fcompile' x cd lab stmp
	 (ycode, lab'', stmp'') <- fcompile' y cd lab' stmp'
	 return (xcode ++ ycode, lab'', stmp)
-- If we know the function name and it's fully applied, use the fast entry 
-- point to save doing some unnecessary stack work
{- but it's not quite right yet
fcompile' (Apply (Global n m ar) as) cd lab
    = do (argcode, args, lab') <- makeargs as cd lab
         return (argcode ++ [FASTCALL (cname n "_fast_"++m) args], lab')
-}
-- ///CIM
fcompile' (Apply f as) cd lab stmp
    = do (rcode, lab', stmp') <- mkRcode lab stmp
         (pcode, lab'', stmp'') <- pushargs as cd lab' stmp'
	 (ccode, lab''', stmp''') <- call f cd lab'' stmp''
	 return (rcode ++ pcode ++ ccode, lab''', stmp')
    where mkRcode lab stmp = return ([USETMP newtmp,PUSH (VAR newtmp)],lab,(stmp+1)) -- make space for return value
          newtmp = ((MN ("stmp",stmp)),True)

fcompile' (Partial f as i) cd lab stmp
    = do (pcode, lab', stmp') <- pushargs as cd lab stmp
	 (ccode, lab'', stmp'') <- closure f (length as) i cd lab' stmp'
	 return (pcode ++ ccode, lab'', stmp)
fcompile' (Foreign ty f as) cd lab stmp = foreigncomp ty f as cd lab stmp
fcompile' (While c sc) cd@(CD mod vars tmp b tr cache ds vs vinfo) lab stmp
     = do let cd' = (CD mod vars tmp (Just end) tr cache ds vs vinfo)
	  (ccode, nlab, stmp') <- fcompile' c cd lab' stmp
	  (sccode, nlab', stmp'') <- fcompile' sc cd' nlab stmp'
	  return ([LABEL start] ++ ccode ++
		  [JFALSE end] ++ sccode ++ [JUMP start, LABEL end], nlab', stmp)
  where start = MN ("l",lab)
	end = MN ("l",(lab+1))
	lab' = lab+2
fcompile' (DoWhile sc c) cd@(CD mod vars tmp b tr cache ds vs vinfo) lab stmp
     = do let cd' = (CD mod vars tmp (Just end) tr cache ds vs vinfo)
          (ccode, nlab, stmp') <- fcompile' c cd lab' stmp
	  (sccode, nlab', stmp'') <- fcompile' sc cd' nlab stmp'
	  return ([LABEL start] ++ sccode ++ ccode ++
		  [JTRUE start,LABEL end], nlab', stmp)
  where start = MN ("l",lab)
	end = MN ("l",(lab+1))
	lab' = lab+2
fcompile' (For x uname y l ar e) (CD mod vars tmp b tr cache ds vs vinfo) lab stmp
     = -- Start of loop, set l to first element of ar
       do let cd' = (CD mod vars' tmp (Just end) tr cache ds vs vinfo)
          (xcode, labs1, stmp1) <- fcompile' (Assign (AName x) (GConst (Num 0))) cd' lab' (stmp+1)
	  (ycode, labs2, stmp2) <- fcompile' (Assign (AName y) ar) cd' labs1 stmp1
-- Check if x equals the length of the array
          let arrlen = Apply (Global sizefn sizemangle 1) [(Loc y)]
          let arrmod = arrModified ar e
          (lencode, labs3, stmp3) <- 
              if arrmod then return ([], labs2, stmp2) else
                 do (alcode, l', s') <- fcompile' arrlen cd' labs2 stmp2
                    return ([USETMP sizetmpvar] ++ alcode ++
                            [PUSHSETTOP (VAR sizetmpvar)], l', s')
          (testcode, labs4, stmp4) <- -- trace (show (ar, e, arrmod)) $
              if arrmod then fcompile' 
	          (Infix Equal (Loc x) arrlen) cd' labs3 stmp3
                else 
                  do return ([PUSH (VAR (vars'!>x)),
                              GETVAL tmp,
                              PUSH (VAR sizetmpvar), 
                              GETVAL (tmp+1),
                              INFIX (tmp+2) Equal tmp (tmp+1),
                              PUSH (INT (tmp+2))], labs3, stmp3)
          (acode, labs5, stmp5) <- fcompile' (Assign l (Index (Loc y) (Loc x))) cd' labs4 stmp4
       -- Compile the body
	  (bodycode, labs6, stmp6) <- fcompile' e cd' labs5 stmp5
       -- Point to the next element of ar, jump to start
	  (inccode, labs7, stmp7) <- fcompile' 
	       (Assign (AName x) (Infix Plus (Loc x) (GConst (Num 1)))) 
	       cd' labs6 stmp6
	  return (xcode ++ ycode ++ lencode ++
                  [LABEL start] ++ testcode ++ [JTRUE end] ++
		  acode ++ bodycode ++ inccode ++ 
		  [JUMP start,LABEL end], labs7, stmp)
  where	start = MN ("l", lab)
	end = MN ("l",lab+1)
	lcx = case uname of
                 Nothing -> MN ("i", lab+2)
                 (Just n) -> n
	lcy = MN ("i", lab+3)
        sizetmpvar = ((MN ("stmp",stmp)),True)
	vars' = vars ++ [(lcx,True),(lcy,True)]
	lab' = lab+5
   -- Array is modified if the local where the array lives in modified, or
   -- if it's something else we don't know about.
        arrModified (Loc i) block = elem i (modified block)
        arrModified (Apply _ _) block = False
        arrModified (Foreign _ _ _) block = False
        arrModified (ArrayInit _) block = False
        arrModified (Annotation a e) block = arrModified e block
        arrModified _ block = True
fcompile' (TryCatch tr ca (Loc err) fin) cd@(CD mod vars tmp b trys cache ds vs vinfo) lab stmp = 
        do let cd' = (CD mod vars tmp b (trys+1) cache ds vs vinfo)
	   (trcode, lab'', stmp') <- fcompile' tr cd' lab' stmp
	   (cacode, lab''', stmp'') <- fcompile' ca cd lab'' stmp'
	   (fincode, lab'''', stmp''') <- fcompile' fin cd lab''' stmp''
	   let tcode = ((TRY catchcode):trcode) ++ [TRIED, JUMP okay] ++
		    ((LABEL catchcode):(PUSH (VAR ((locs cd)!>err))):SETTOP:
		    RESTORE:TRIED:cacode) ++ fincode ++ [LABEL okay]
	   return (tcode, lab'''', stmp)
   where catchcode = MN ("l",lab)
	 okay = MN ("l",lab+1)
	 lab' = lab+2
fcompile' (NewTryCatch tr cs) cd@(CD mod vars tmp b trys cache ds vs vinfo) lab stmp = 
        do let cd' = (CD mod vars tmp b (trys+1) cache ds vs vinfo)
	   (trcode, lab'', stmp') <- fcompile' tr cd' lab' stmp
	   (catchblocks, lab''', stmp'') <- catchcompile cs okay cd lab'' stmp'
	   let tcode = ((TRY catchcode):trcode) ++ [TRIED, JUMP okay] ++
		       (LABEL catchcode):catchblocks ++ [LABEL okay]
	   return (tcode, lab''', stmp)
   where catchcode = MN ("l",lab)
	 okay = MN ("l",lab+1)
	 lab' = lab+2
fcompile' (Throw err) cd lab stmp = 
        do (errcode, lab', stmp') <- fcompile' err cd lab stmp
	   return (errcode ++ [THROW], lab', stmp)
fcompile' (Except err code) cd lab stmp = 
        do (errcode, lab', stmp') <- fcompile' err cd lab stmp
	   (ccode, lab'', stmp'') <- fcompile' code cd lab' stmp'
	   return (errcode ++ ccode ++ [MKEXCEPT], lab'', stmp)
fcompile' (Break f l) cd lab stmp = case (breaklabel cd) of
			    Nothing -> fail $ f++":"++show l++":"++
				       "Can't break out of function - use 'return'"
			    Just l -> return ([JUMP l],lab,stmp)
fcompile' (PrintNum x) cd lab stmp
     = do (code, lab',stmp') <- fcompile' x cd lab stmp
	  return (code ++ [PRINTINT], lab',stmp)
fcompile' (PrintStr x) cd lab stmp
     = do (code, lab',stmp') <- fcompile' x cd lab stmp
	  return (code ++ [PRINTSTR], lab',stmp)
fcompile' (PrintExc x) cd lab stmp
     = do (code, lab',stmp') <- fcompile' x cd lab stmp
	  return (code ++ [PRINTEXC], lab',stmp)
fcompile' (If c t e) cd lab stmp
     = do (ccode, lab'', stmp') <- fcompile' c cd lab' stmp
	  (tcode, lab''', stmp'') <- fcompile' t cd lab'' stmp'
	  (ecode, lab'''', stmp''') <- fcompile' e cd lab''' stmp''
	  return (ccode ++ [JFALSE false] ++ tcode ++
		  [JUMP end,LABEL false] ++ ecode ++ [LABEL end], lab'''',stmp)
  where false = MN ("l",lab)
	end = MN ("l",lab+1)
	lab' = lab+2
fcompile' (Infix op x y) cd lab stmp = icompile False (Infix op x y) cd lab stmp
fcompile' (RealInfix op x y) cd lab stmp = rcompile (RealInfix op x y) cd lab stmp
fcompile' (CmpExcept op x y) cd lab stmp = 
   do (xcode, lab', stmp') <- fcompile' x cd lab stmp
      (ycode, lab'', stmp'') <- fcompile' y cd lab' stmp'
      return (xcode ++ ycode ++ cmpcode op, lab'',stmp)
   where cmpcode Equal = [EQEXCEPT]
	 cmpcode NEqual = [NEEXCEPT]
fcompile' (CmpStr op x y) cd lab stmp = 
   do (xcode, lab', stmp') <- fcompile' x cd lab stmp
      (ycode, lab'', stmp'') <- fcompile' y cd lab' stmp
      return (xcode ++ ycode ++ cmpcode op, lab'', stmp)
   where cmpcode Equal = [EQSTRING]
	 cmpcode NEqual = [NESTRING]
fcompile' (Append x y) cd lab stmp = 
   do (xcode, lab', stmp') <- fcompile' x cd lab stmp
      (ycode, lab'', stmp'') <- fcompile' y cd lab' stmp'
      return (xcode ++ ycode ++ [APPEND], lab'', stmp)
fcompile' (AppendChain xs) cd lab stmp = 
   do (chcode, lab', stmp') <- chain xs cd lab stmp
      return ((PUSH EMPTYSTR):chcode, lab', stmp)
    where chain (x:xs) cd lab stmp
              = do (xcode, lab',stmp') <- fcompile' x cd lab stmp
                   (xscode, lab'',stmp'') <- chain xs cd lab' stmp'
                   return (xcode ++ APPENDTOP:xscode, lab'',stmp)
          chain [] cd lab stmp = return ([], lab,stmp)
fcompile' (Unary op x) cd lab stmp = icompile False (Unary op x) cd lab stmp
fcompile' (RealUnary op x) cd lab stmp = rcompile (RealUnary op x) cd lab stmp
fcompile' (Coerce t1 t2 v) cd lab stmp = 
   do (vcode,lab',stmp') <- fcompile' v cd lab stmp
      cocompile t1 t2 vcode cd lab' stmp'
fcompile' (Case v alts) cd lab stmp = 
   do (vcode,lab',stmp') <- fcompile' v cd lab stmp
      (acode,lab'',stmp'') <- altcompile False alts cd lab' stmp'
      return (vcode ++ acode, lab'',stmp)

fcompile' (Index v i) cd lab stmp = 
   do (vcode, lab', stmp') <- fcompile' v cd lab stmp
      (icode, lab'', stmp'') <- fcompile' i cd lab' stmp'
      return (vcode ++ icode ++ [GETINDEX], lab'', stmp)
fcompile' (Field v n a t) cd lab stmp =
   do (vcode, lab', stmp') <- fcompile' v cd lab stmp
      return (vcode ++ [PROJARG a t], lab', stmp)
fcompile' (ArrayInit xs) cd lab stmp =
   do (xscode, lab', stmp') <- pushargs xs cd lab stmp
      return (xscode ++ [MKARRAY (length xs)], lab', stmp)

--fcompile' (GConst (Num i)) cd lab
--   where i = getFunID n (alldecls cd)
    
fcompile' (Error str) cd lab stmp = return ([PUSH (STR str), ERROR], lab, stmp)
fcompile' Noop cd lab stmp = return ([], lab, stmp)
fcompile' NoInit cd lab stmp = return ([], lab, stmp)
fcompile' VMPtr cd lab stmp = return ([VMPTR],lab,stmp)
fcompile' (Length s) cd lab stmp = 
   do (scode, lab', stmp') <- fcompile' s cd lab stmp
      return (scode ++ [STRLENGTH], lab', stmp)
fcompile' (Metavar f l i) cd lab stmp = fail $ f++":"++show l++":Can't compile a metavariable"
fcompile' (Annotation a e) cd lab stmp =
   do (ecode, lab', stmp') <- fcompile' e cd lab stmp
      annotcompile a ecode lab' stmp'

fcompile' x _ lab stmp
--    = fail "FOO!"
    = return (trace ("Warning, unknown parse tree entry, please report:\n" ++ show x) $ ([],lab,stmp))

remember [] = []
remember (x:xs) = (REMEMBER x) : remember xs


-- If the name is a constructor function, 
-- return the tag and arity so that we don't actually have to *run* the
-- constructor function at run-time. This is distressingly slow.
isConApp :: Name -> Program -> Maybe (Int, Int)
isConApp nm [] = Nothing
isConApp nm ((FunBind (_,_,n,_,_,DataCon t a _) _ _):xs)
        | n == nm = Just (t,a)
        | otherwise = isConApp nm xs
isConApp nm (x:xs) = isConApp nm xs

--acompile ivs xs = map (\x -> POPARG (x,False)) (map fst xs)

acompile ivs xs = popargs (aget ivs xs)

-- The 'repeat Copy' is in case we don't have the same length information
-- about argument varness. This is a dodgy hack. Sorry.
aget ivs xs = acomp' 0 (zip (ivs++(repeat Copy)) (map fst xs))
  where acomp' _ [] = []
        acomp' idx ((v,x):xs) 
            = (v, decorate (x,False) idx):(acomp' (idx+1) xs)

--            = (poparg (v, decorate (x,False) idx)) ++ (acomp' (idx+1) xs)

popargs xs = concat $ map poparg xs

poparg (Var, x) = [POPARG x, NOTEVAR x, COPYARG x]
poparg (Copy, x) = [POPANDCOPYARG x]

--poparg (Var, x) = [DECLAREARG x, NOTEVAR x, COPYARG x]
--poparg (Copy, x) = [DECLAREARG x, COPYARG x]


-- For catchcompile, the exception is already at the top of the stack.

-- No handlers found, rethrow the exception. Rethrow case is inserted
-- by the parser, but should really be done here!
catchcompile [] end cd lab stmp = return ([], lab, stmp)
-- If value at top of expression stack is 'ex', execute handler, otherwise
-- jump to next
catchcompile ((Catch (Left (ex,args)) handler):xs) end cd lab stmp =
    do (hcode, lab'',stmp') <- fcompile' handler cd lab' stmp
       (xscode, lab''',stmp'') <- catchcompile xs end cd lab'' stmp'
       let tcode = [JEXNE ex nexthandler] ++ 
                   bindargs args 0 ++
                   [RESTORE, TRIED] ++
                   hcode ++ 
                   [JUMP end, LABEL nexthandler] ++ xscode
       return (tcode, lab''',stmp)
   where nexthandler = MN ("l",lab)
         lab' = lab+1
         bindargs [] i = []
         bindargs ((Loc x):xs) i = (EXPROJARG ((locs cd)!>x) i):(bindargs xs (i+1))
-- Catch all case, don't bother compiling the rest since they won't be used
catchcompile ((Catch (Right (Loc err)) handler):xs) end cd lab stmp =
    do (hcode, lab', stmp') <- fcompile' handler cd lab stmp
       let tcode = [(PUSH (VAR ((locs cd)!>err))), SETTOP, RESTORE, TRIED] ++
                    hcode ++ [JUMP end]
       return (tcode, lab',stmp)

ccompile (Num x) = [PUSH (VAL x)]
ccompile (Re x) = [PUSH (RVAL x)]
ccompile (Ch x) = [PUSH (VAL (fromEnum x))]
ccompile (Bo True) = [PUSH (VAL 1)]
ccompile (Bo False) = [PUSH (VAL 0)]
ccompile (Str str) = [PUSH (STR str)]
ccompile (Exc str i) = [PUSH (STR str), PUSH (VAL i), MKEXCEPT]

ccompile_stack (Num x) stmp lab 
    = return ([USETMP tmpvar, STACKINT tmpvar x], lab, stmp+1)
   where tmpvar = ((MN ("stmp",stmp)),True)
ccompile_stack (Re x) stmp lab 
    = return ([USETMP tmpvar, STACKREAL tmpvar (RVAL x)], lab, stmp+1)
   where tmpvar = ((MN ("stmp",stmp)),True)
ccompile_stack (Ch x) stmp lab 
    = return ([USETMP tmpvar, STACKINT tmpvar (fromEnum x)], lab, stmp+1)
   where tmpvar = ((MN ("stmp",stmp)),True)
--    = return ([PUSH (VAL (fromEnum x))], lab)
ccompile_stack (Bo True) stmp lab 
    = return ([USETMP tmpvar, STACKINT tmpvar 1], lab, stmp+1)
   where tmpvar = ((MN ("stmp",stmp)),True)
--    = return ([PUSH (VAL 1)], lab)
ccompile_stack (Bo False) stmp lab 
    = return ([USETMP tmpvar, STACKINT tmpvar 0], lab, stmp+1)
   where tmpvar = ((MN ("stmp",stmp)),True)
--    = return ([PUSH (VAL 0)], lab)
ccompile_stack (Str str) stmp lab 
    = return ([PUSH (STR str)], lab, stmp)
-- doesn't quite work yet
{-    = return ([USETMP tmpvar, STACKSTR tmpvar (STR str)], lab, stmp+1)
   where tmpvar = ((MN ("stmp",stmp)),True) -}
ccompile_stack (Exc str i) stmp lab 
    = return ([PUSH (STR str), PUSH (VAL i), MKEXCEPT], lab, stmp)

makeargs :: Monad m => [Expr Name] -> CompData -> Int -> Int -> m ([TAC],[Var],Int,Int)
makeargs [] _ lab stmp = return ([], [], lab, stmp)
makeargs (x:xs) cd lab stmp
    = do (pcode, nms, lab', stmp') <- makeargs xs cd (lab+1) stmp
	 (xcode, lab'', stmp'') <- fcompile' x cd lab' stmp'
	 return (pcode ++ xcode ++ 
                 [USETMP tmpvar, PUSH (VAR tmpvar), SETTOP], 
                 nms ++ [tmpvar], 
                 lab'', stmp)
  where tmpvar = ((MN ("apptmp",lab)), True)

-- ///CIM
pushargs :: Monad m => [Expr Name] -> CompData -> Int -> Int -> m ([TAC],Int,Int)
pushargs [] _ lab stmp = return ([], lab, stmp)
pushargs (x:xs) cd lab stmp
    = do (pcode, lab', stmp') <- pushargs xs cd lab stmp
	 (xcode, lab'', stmp'') <- fcompile' x cd lab' stmp'
	 return (pcode ++ xcode, lab'', stmp'')

pushargsHeap :: Monad m => [Expr Name] -> CompData -> Int -> Int -> m ([TAC],Int,Int)
pushargsHeap [] _ lab stmp = return ([], lab, stmp)
pushargsHeap (x:xs) cd lab stmp
    = do (pcode, lab', stmp') <- pushargsHeap xs cd lab stmp
	 (xcode, lab'', stmp'') <- fMkHeap x cd lab' stmp'
	 return (pcode ++ xcode, lab'', stmp'')

call (Global n m ar) cd lab stmp 
    | Just (tag,0) <- isConApp n (alldecls cd)
      = return ([MKCONZERORV tag], lab, stmp)
    | Just (tag,arity) <- isConApp n (alldecls cd)
      = return ([MKCONRV tag arity], lab, stmp)
    | otherwise = return ([CALLNAME (cname n m)],lab,stmp)
call (Loc i) cd lab stmp = return ([CALL ((locs cd)!>i)],lab,stmp)
call x cd lab stmp = do (ccode,lab',stmp') <- fcompile' x cd lab stmp
		        return ((ccode ++ [CALLTOP]),lab',stmp')

tailcall (Global n m ar) cd lab stmp 
    | Just (tag,0) <- isConApp n (alldecls cd)
      = return ([MKCONZERORV tag], lab, stmp)
    | Just (tag,arity) <- isConApp n (alldecls cd)
        = return ([MKCONRV tag arity], lab, stmp)
    | otherwise = return ([TAILCALLNAME (cname n m)],lab,stmp)
tailcall (Loc i) cd lab stmp = return ([TAILCALL ((locs cd)!>i)],lab,stmp)
tailcall x cd lab stmp = do (ccode,lab',stmp') <- fcompile' x cd lab stmp
		            return ((ccode ++ [TAILCALLTOP]),lab',stmp')

closure (Global n m ar) i left cd lab stmp
    = return ([CLOSURE (cname n m) i ar],lab,stmp)
closure (Loc l) i left cd lab stmp = return ([CLOSURELOC ((locs cd)!>l) i],lab,stmp)
closure x i left cd lab stmp = do (ccode,lab',stmp') <- fcompile' x cd lab stmp
		                  return ((ccode ++ [CLOSURETOP i]),lab',stmp)

{-
closure x cd lab = let (ccode,lab') = fcompile' x cd lab in
		    ((ccode ++ [CALLTOP]),lab')
-}

icompile :: Monad m => Bool -> -- True if heap allocation needed
                      Expr Name -> CompData -> Int -> Int -> m ([TAC], Int, Int)
icompile hp e cd@(CD mod vars tmp b tr cache ds vs vinfo) lab stmp = 
     do (ecode, lab', stmp') <- icomp' e lab tmp stmp
        cg hp tmp lab' stmp' ecode 
  where
     cg True tmp lab stmp ecode = return (ecode ++ [PUSH (INT tmp)], lab, stmp)
     cg False tmp lab stmp ecode
         = let tmpvar = ((MN ("stmp",stmp)),True) in
               return (ecode ++ [USETMP tmpvar, STACKTMP tmpvar tmp], lab+1, stmp+1)
 -- icomp compiles an expression and puts the result of its evaluation in <tmp>
 -- Shortcut special case
     icomp' (Infix OpAndBool x y) lab tmp stmp =
          do (cx,lab',stmp') <- icomp' x nextlab (tmp+1) stmp
             (cy,lab'',stmp'') <- icomp' y lab' (tmp+2) stmp'
	     return (cx ++ (JTFALSE (tmp+1) shortcut1):cy ++ 
		     [INFIX tmp OpAndBool (tmp+1) (tmp+2), JUMP shortcut2,
		      LABEL shortcut1, TMPSET tmp 0,
	              LABEL shortcut2], lab'',stmp)
        where nextlab = lab+2
	      shortcut1 = MN ("l",lab)
	      shortcut2 = MN ("l",lab+1)
 -- Shortcut special case
     icomp' (Infix OpOrBool x y) lab tmp stmp =
          do (cx,lab',stmp') <- icomp' x nextlab (tmp+1) stmp
	     (cy,lab'',stmp'') <- icomp' y lab' (tmp+2) stmp'
	     return (cx ++ (JTTRUE (tmp+1) shortcut1):cy ++ 
		     [INFIX tmp OpOrBool (tmp+1) (tmp+2), JUMP shortcut2,
		      LABEL shortcut1, TMPSET tmp 1,
	              LABEL shortcut2], lab'',stmp)
        where nextlab = lab+2
	      shortcut1 = MN ("l",lab)
	      shortcut2 = MN ("l",lab+1)
 -- Power operator
     icomp' (Infix Power x y) lab tmp stmp =
          do (cx,lab',stmp') <- icomp' x lab (tmp+1) stmp
	     (cy,lab'',stmp'') <- icomp' y lab' (tmp+2) stmp'
	     return (cx ++ cy ++ [INTPOWER tmp (tmp+1) (tmp+2)], lab'',stmp)
 -- General case
     icomp' (Infix op x y) lab tmp stmp = 
          do (cx,lab',stmp') <- icomp' x lab (tmp+1) stmp
	     (cy,lab'',stmp'') <- icomp' y lab' (tmp+2) stmp'
	     return (cx ++ cy ++ [INFIX tmp op (tmp+1) (tmp+2)], lab'',stmp)
     icomp' (Unary op x) lab tmp stmp =
	  do (cx,lab',stmp') <- icomp' x lab (tmp+1) stmp
	     return (cx ++ [UNARY tmp op (tmp+1)], lab',stmp)
     icomp' (GConst (Num x)) lab tmp stmp = return ([TMPSET tmp x], lab, stmp)
     icomp' x lab tmp stmp = 
	  do (xcode, lab',stmp') <- fcompile' x (CD mod vars (tmp+1) b tr cache ds vs vinfo) lab stmp
	     return (xcode ++ [GETVAL tmp], lab',stmp)

rcompile :: Monad m => Expr Name -> CompData -> Int -> Int -> m ([TAC], Int, Int)
rcompile e cd@(CD mod vars tmp b tr cache ds vs vinfo) lab stmp = 
     do (ecode, lab', pushit, stmp') <- rcomp' e lab (tmp+10000) stmp -- TMP HACK!
	if pushit 
--	    then return (ecode ++ [PUSH (REAL (tmp+10000))], lab', stmp)
            then return (ecode ++ [USETMP tmpvar, STACKREAL tmpvar (REAL (tmp+10000))], lab', stmp+1)
	    else return (ecode, lab', stmp)
  where
     tmpvar = ((MN ("stmp",stmp)),True)
 -- rcomp compiles an expression and puts the result of its evaluation in <tmp>
 -- Power operator
     rcomp' (RealInfix Power x y) lab tmp stmp =
          do (cx,lab',_,stmp') <- rcomp' x lab (tmp+1) stmp
	     (cy,lab'',_,stmp'') <- rcomp' y lab' (tmp+2) stmp'
	     return (cx ++ cy ++ [REALPOWER tmp (tmp+1) (tmp+2)], lab'', True, stmp)
     rcomp' (RealInfix op x y) lab tmp stmp
       | op `elem` boolops = -- Put result in boolean, so ignore tmp
          do (cx,lab',_,stmp') <- rcomp' x lab (tmp+1) stmp
	     (cy,lab'',_,stmp'') <- rcomp' y lab' (tmp+2) stmp'
	     return (cx ++ cy ++ [REALINFIXBOOL op (tmp+1) (tmp+2)], lab'', False, stmp)
       | otherwise = 
          do (cx,lab',_,stmp') <- rcomp' x lab (tmp+1) stmp
	     (cy,lab'',_,stmp'') <- rcomp' y lab' (tmp+2) stmp'
	     return (cx ++ cy ++ [REALINFIX tmp op (tmp+1) (tmp+2)], lab'', True, stmp)
     rcomp' (RealUnary op x) lab tmp stmp =
	  do (cx,lab',_,stmp') <- rcomp' x lab (tmp+1) stmp
	     return (cx ++ [REALUNARY tmp op (tmp+1)], lab', True, stmp)
     rcomp' (GConst (Re x)) lab tmp stmp = return ([RTMPSET tmp x], lab, True,stmp)
     rcomp' x lab tmp stmp = 
	 do (xcode, lab',stmp') <- fcompile' x (CD mod vars (tmp+1-10000) b tr cache ds vs vinfo) lab stmp
	    return (xcode ++ [GETRVAL tmp], lab', True, stmp)

cocompile :: Monad m => Type -> Type -> [TAC] -> CompData -> Int -> Int ->
	                m ([TAC], Int, Int)
cocompile (Prim StringType) (Prim Number) vcode cd lab stmp =
       return (vcode ++ [STR2INT], lab, stmp)
cocompile (Prim Number) (Prim StringType) vcode cd lab stmp =
       return (vcode ++ [INT2STR], lab, stmp)
cocompile (Prim RealNum) (Prim StringType) vcode cd lab stmp =
       return (vcode ++ [REAL2STR], lab, stmp)
cocompile (Prim StringType) (Prim RealNum) vcode cd lab stmp =
       return (vcode ++ [STR2REAL], lab, stmp)
cocompile (Prim StringType) (Prim Character) vcode cd lab stmp =
       return (vcode ++ [STR2CHR], lab, stmp)
cocompile (Prim Character) (Prim StringType) vcode cd lab stmp =
       return (vcode ++ [CHR2STR], lab, stmp)
cocompile (Prim Boolean) (Prim StringType) vcode cd lab stmp =
       return (vcode ++ [BOOL2STR], lab, stmp)
cocompile (Prim Number) (Prim RealNum) vcode cd lab stmp =
       return (vcode ++ [INT2REAL], lab, stmp)
cocompile (Prim RealNum) (Prim Number) vcode cd lab stmp =
       return (vcode ++ [REAL2INT], lab, stmp)
cocompile (Prim Character) (Prim Number) vcode cd lab stmp = return (vcode, lab, stmp)
cocompile (Prim Number) (Prim Character) vcode cd lab stmp = return (vcode, lab, stmp)
cocompile _ _ v _ _ _ = fail "Internal error, can't coerce"

altcompile :: Monad m => Bool -> -- Put on the heap
                         [CaseAlt Name] -> CompData -> Int -> Int -> m ([TAC], Int, Int)
-- INVARIANT: On entry, alternatives are already sorted.
-- Special case; they're all default cases, take the first one.
altcompile hp ((Default r):_) cd lab stmp = do (code,lab',stmp') <- fHeapComp hp r cd lab stmp
                                               return ((DISCARD:code), lab', stmp)
altcompile hp alts@((ConstAlt _ _ _):_) cd labs stmp =
    constaltcompile hp alts cd labs stmp
altcompile hp alts@((ArrayAlt _ _):_) cd labs stmp =
    arrayaltcompile hp alts cd labs stmp
altcompile hp alts cd lab stmp = do (codes, lab',stmp') <- mkcase sortalts lab stmp
			            return ([CASE codes], lab',stmp) 
  where mkcase [] lab stmp = return ([], lab,stmp)
	mkcase (x:xs) lab stmp = do (xcode, lab', stmp') <- ac x lab stmp
			            (codes, lab'',stmp'') <- mkcase xs lab' stmp'
			            return (xcode:codes,lab'',stmp)

        sortalts = insertdefault alts (getdef alts)
	ac (Alt t _ as r) lab stmp = 
	    do (rcode, lab',stmp') <- (fHeapComp hp r cd lab stmp)
	       return (bindargs 0 as ++ (DISCARD:rcode), lab', stmp)
	bindargs arg [] = []
        -- Don't project it out if it's never used.
	bindargs arg ((Loc i):xs) 
                 | unusedVar i cd = bindargs (arg+1) xs
                 | otherwise = (PROJ ((locs cd)!>i) arg):(bindargs (arg+1) xs)
	insertdefault xs def = idef 0 def (findtot xs) xs 

        findtot ((Alt t tot _ _):_) = tot
        findtot ((Default _):xs) = findtot xs
        findtot _ = error "Can't happen, findtot" -- all default; can't happen

	idef next def tot all@((alt@(Alt t _ _ _)):alts)
              | next == t = alt:(idef (next+1) def tot alts)
	      | next-1 == t = error "Overlapping cases"
	      | otherwise = (errorcase next tot def):(idef (next+1) def tot all)
        idef next def tot ((Default _):_) = idef next def tot []
        idef next def tot []
	      | next == tot = []
	      | otherwise = (errorcase next tot def):(idef (next+1) def tot [])
	errorcase t tot = Alt t tot []

        getdef [] =  throwerror
        getdef ((Default exp):xs) = exp
        getdef (_:xs) = getdef xs

        throwerror = (Throw (Apply (Global missingCase missingCaseMangling 0) []))

-- We have the value we're testing on the top of the stack. Compile
-- as a jump table.
constaltcompile :: Monad m => Bool -> -- Put on heap
                   [CaseAlt Name] -> CompData -> Int -> Int ->
		   m ([TAC], Int, Int)
constaltcompile hp cs cd lab stmp = ca' cs Number lab [] stmp
  where
    ca' ((ConstAlt pt c e):xs) _ lab acc stmp = 
	do (ecode,lab',stmp') <- fcompile' e cd lab stmp
	   ca' xs pt lab' ((c,DISCARD:ecode):acc) stmp
    ca' [] pt lab acc stmp =
	do (ecode,lab',stmp') <- fHeapComp hp throwerror cd lab stmp
	   mkConstAlts pt acc (DISCARD:ecode) cd lab' stmp
    ca' ((Default e):xs) pt lab acc stmp =
	do (ecode,lab',stmp') <- fHeapComp hp e cd lab stmp
	   mkConstAlts pt acc (DISCARD:ecode) cd lab' stmp

    throwerror = (Throw (GConst (Exc "Missing default" (127))))

    mkConstAlts pt acc def cd lab' stmp =
	return ([CONSTCASE pt acc def], lab', stmp)

-- We have the array we're testing on the top of the stack. 
-- Find its length, then compile as a jump table.
arrayaltcompile :: Monad m => Bool -> -- Put on heap
                   [CaseAlt Name] -> CompData -> Int -> Int ->
		   m ([TAC], Int, Int)
arrayaltcompile hp cs cd lab stmp = do (code, l, stmp') <- ca' cs lab [] stmp
                                       return ((GETLENGTH:code), l, stmp)
  where
    ca' ((ArrayAlt arr e):xs) lab acc stmp = 
	do -- compile code to project each variable out of the array
           let projcode = projArgs 0 arr
           (ecode,lab',stmp') <- fcompile' e cd lab stmp
           -- Discard length, project args, discard array
	   ca' xs lab' ((Num (length arr),(DISCARD:projcode) ++ (DISCARD:ecode)):acc) stmp
    ca' [] lab acc stmp =
	do (ecode,lab',stmp') <- fHeapComp hp throwerror cd lab stmp
           -- Discard length and array
	   mkArrAlts acc (DISCARD:DISCARD:ecode) cd lab' stmp
    ca' ((Default e):xs) lab acc stmp =
	do (ecode,lab',stmp') <- fHeapComp hp e cd lab stmp
           -- Discard length and array
	   mkArrAlts acc (DISCARD:DISCARD:ecode) cd lab' stmp

    projArgs i [] = []
    -- array at top of stack. Project element i and store it in v,
    -- leaving array where it is. Don't bother if v is unused though.
    projArgs i ((Loc v):xs) 
             | unusedVar v cd = projArgs (i+1) xs
             | otherwise = (SETLOOKUP i ((locs cd)!>v)):(projArgs (i+1) xs)

    throwerror = (Throw (GConst (Exc "Missing default" (127))))

    mkArrAlts acc def cd lab' stmp =
	return ([CONSTCASE Number acc def], lab', stmp)

annotcompile :: Monad m => Annotation -> [TAC] -> Int -> Int ->
		m ([TAC], Int, Int)
annotcompile (Line f l) code lab stmp = return $ ((LINENO f l):code, lab, stmp)
annotcompile (FnBody fn f l) code lab stmp
    = return $ (((PUSHBT fn f l):code++ [POPBT]), lab, stmp)
annotcompile (LamBody f) code lab stmp
    = return $ (((INLAM f):code++[POPBT]), lab, stmp)
annotcompile (DynCheck t) code lab stmp = return (code,lab,stmp)

decls :: [TAC] -> [TAC]
decls c = add_decls (find_decls c ([],[],[],[],[])) c

add_decls (ts,ts',vs,vs',as) c 
    = (map TMPINT (removedups ts)) ++
      (map TMPREAL (removedups ts')) ++
      (map dodec (removedups (removeothers vs' vs))) ++ 
      (map DECLAREQUICK (removedups vs')) ++
--      (map DECLAREARG (removedups vs')) ++
      (map ARRAY (removedups as)) ++ c

dodec (v,True) = DECLAREQUICK (v, True)
dodec (v,False) = DECLAREQUICK (v, False)
--dodec (v,False) = DECLARE (v, False)

removedups [] = []
removedups (x:xs) | x `elem` xs = removedups xs
		  | otherwise = x:(removedups xs)

removeothers vs [] = []
removeothers vs (x:xs) | x `elem` vs = removeothers vs xs
		       | otherwise = x:(removeothers vs xs)

-- FIXME: Yeeeuuch! There are much neater ways of doing this, probably with
-- a State monad.

find_decls :: [TAC] -> ([Int],[Int],[Var],[Var],[Var]) -> 
	               ([Int],[Int],[Var],[Var],[Var])
find_decls [] (ts,ts',vs,vs',as) = (ts,ts',vs,vs',as)
find_decls ((SET x i y):xs) (ts,ts',vs,vs',as) 
    = find_decls xs (ts,ts',x:y:vs,vs',as)
--find_decls ((USETMP v):xs) (ts,ts',vs,vs',as) 
--    = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((TMPSET x i):xs) (ts,ts',vs,vs',as) 
    = find_decls xs (x:ts,ts',vs,vs',as)
find_decls ((GETVAL x):xs) (ts,ts',vs,vs',as) 
    = find_decls xs (x:ts,ts',vs,vs',as)
find_decls ((GETRVAL x):xs) (ts,ts',vs,vs',as) 
    = find_decls xs (ts,x:ts',vs,vs',as)
find_decls ((RTMPSET x i):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,x:ts',vs,vs',as)
find_decls ((CALL v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((TAILCALL v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((INFIX t1 _ t2 t3):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (t1:t2:t3:ts,ts',vs,vs',as)
find_decls ((INTPOWER t1 t2 t3):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (t1:t2:t3:ts,ts',vs,vs',as)
find_decls ((UNARY t1 _ t2):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (t1:t2:ts,ts',vs,vs',as)
find_decls ((REALINFIX t1 _ t2 t3):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (ts,t1:t2:t3:ts',vs,vs',as)
find_decls ((REALINFIXBOOL _ t2 t3):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (ts,t2:t3:ts',vs,vs',as)
find_decls ((REALPOWER t1 t2 t3):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (ts,t1:t2:t3:ts',vs,vs',as)
find_decls ((REALUNARY t1 _ t2):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (ts,t1:t2:ts',vs,vs',as)
find_decls ((PUSH (VAR v)):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((PUSHSETTOP (VAR v)):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((PUSHGETVAL (VAR v) t):xs) (ts,ts',vs,vs',as) = find_decls xs (t:ts,ts',v:vs,vs',as)
find_decls ((PUSHGETRVAL (VAR v) t):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,t:ts',v:vs,vs',as)
find_decls ((PUSHGETINDEX (VAR v)):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((PUSHTOINDEX (VAR v)):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((SETLOOKUP _ v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((POP v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((PUSHSETINT (VAR v) x):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((SETINT v x):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((SETVAR v x):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:x:vs,vs',as)
find_decls ((POPARG v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',vs,v:vs',as)
find_decls ((POPANDCOPYARG v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',vs,v:vs',as)
find_decls ((USETMP v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',vs,v:vs',as)
find_decls ((POPINDEX v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',v:as)
find_decls ((CASE alts):xs) (ts,ts',vs,vs',as) = find_decls xs 
					    (fd_alts alts (ts,ts',vs,vs',as))
   where fd_alts [] acc = acc
	 fd_alts (xs:xss) acc = find_decls xs (fd_alts xss acc)
find_decls ((CONSTCASE _ ccs defcode):xs) (ts,ts',vs,vs',as) 
    = find_decls xs (find_decls defcode 
                     (fd_alts (map snd ccs) (ts,ts',vs,vs',as)))
   where fd_alts [] acc = acc
	 fd_alts (xs:xss) acc = find_decls xs (fd_alts xss acc)
find_decls ((PROJ v i):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((EXPROJARG v i):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls (_:xs) acc = find_decls xs acc


-- compileDecls :: [ConDecl] -> [CompileResult]
-- compileDecls cs = cd 0 cs
--   where cd i [] = []
-- 	cd i ((Con n t):xs) = (ByteCode (n,[MKCON i (numargs t)])):
-- 			          (cd (i+1) xs)
-- 	numargs (Fn _ ts t) = length ts
-- 	numargs _ = 0

mkGlobInit :: Program -> Expr Name
mkGlobInit [] = NoInit
mkGlobInit ((Glob (n,st,gid,Nothing)):xs) = mkGlobInit xs
mkGlobInit ((Glob (n,st,gid,Just val)):xs) = 
    Seq (Assign (AGlob gid) val) (mkGlobInit xs)
mkGlobInit (_:xs) = mkGlobInit xs

addGlobInit :: Program -> Name -> Program
addGlobInit prg mod 
    = let expr = exceptGuard mod $ mkGlobInit prg 
          ty = Fn [] [] (Prim Void) in
          (FunBind ("(autogenerated)",0,(NS mod (UN "__init")),
                    ty,[StartupFn],Defined expr)
                   "" ty):prg

-- Stick a try...catch block around global initialisation
exceptGuard mod Noop = Noop
exceptGuard mod NoInit = NoInit
exceptGuard mod expr = Lambda [] [] $
  Declare "(autogenerated)" 0 
     (MN ("exc",0), True) (Prim Exception) $
       TryCatch expr dumpex (Loc 0) Noop
      where dumpex = Seq printfail $
                     Seq 
                     (Apply (Global backtracefun backtracemangle 1) 
                        [Loc 0])
                     bombout
            bombout = Apply (Global exitfun exitmangle 1) 
                        [GConst (Num 0)]
            printfail = Apply (Global putstrlnfun putstrlnmangle 1) 
                        [GConst (Str "Global initialisation failed")]

conZeroCode t = Code [] [] [MKCONZERO t, SETRV]
conCode t i = Code [] [] [MKCON t i, SETRV]
exceptCode nm i = Code [] [] [MKNEWEXCEPT nm i]

compileAll :: Monad m => Program -> Name -> m [CompileResult]
compileAll ds n = ca' ds n
  where
    ca' [] _ = return []
    ca' (CInclude str:xs) mod = do
        rest <- ca' xs mod
	return $ (RawCode ("extern \"C\" {\n#include <"++str++">\n}")):rest
    ca' (FMName n:xs) mod = do
        rest <- (ca' xs mod)
        return $ (RawCode ("void " ++ n ++ "(VMState* vm);\n")):rest
    ca' ((FunBind (_,_,n,ty,fopts,Defined e) _ _):xs) mod = do
        rest <- ca' xs mod
	code <- comp ds e (elem Pure fopts) mod
	return $ (ByteCode (n,ty,code)):rest
    ca' ((FunBind (_,_,n, ty,_,Unbound) _ _):xs) mod = do
        rest <- ca' xs mod
	return $ (ExternDef (n, ty)):rest
    ca' ((FunBind (_,_,n, ty,_,ExtInlinable _) _ _):xs) mod = do
        rest <- ca' xs mod
	return $ (ExternDef (n, ty)):rest
    ca' ((FunBind (_,_,n,ty,_,DataCon t 0 True) _ _):xs) mod = do
        rest <- ca' xs mod    
	return $ (ByteCode (n,ty,conZeroCode t)):rest
    ca' ((FunBind (_,_,n,ty,_,DataCon t i True) _ _):xs) mod = do
        rest <- ca' xs mod    
	return $ (ByteCode (n,ty,conCode t i)):rest
    ca' ((FunBind (_,_,n,ty,_,DataCon t i False) _ _):xs) mod = do
        rest <- ca' xs mod    
	return $ (ExternDef (n, ty)):rest
    ca' ((FunBind (_,_,n,ty,_,ExceptionFn nm i True) _ _):xs) mod = do
        rest <- ca' xs mod
        return $ (ExcCode nm True):(ByteCode (n,ty,exceptCode nm i)):rest
    ca' ((FunBind (_,_,n,ty,_,ExceptionFn nm i False) _ _):xs) mod = do
        rest <- ca' xs mod
        return $ (ExcCode nm False):(ExternDef (nm ,ty)):rest
--ca' ((DataDecl n tys cs):xs) = (compileDecls cs) ++ (ca' xs)
    ca' ((Glob (n,st,gid,_)):xs) mod = do
        rest <- ca' xs mod
	return $ (GlobCode gid):rest
--    ca' ((ExceptDecl f l n ty comm):xs) mod = do
--        rest <- ca' xs mod
--        let ext = if defhere then "" else "extern "
--        return $ (RawCode $ ext ++ "void* " ++show n++";\n"):rest
    ca' (x:xs) mod = ca' xs mod

-- 'peephole' optimisation of bytecode; combines common sequences of
-- instructions into faster instructions.

-- INVARIANT: code modified by peephole should not have different semantics
-- (i.e. value returned is the same and side effects happen in the same order).

peephole :: [TAC] -> [TAC]
peephole [] = []
-- Add argument declaration.
--peephole ((POPARG x):xs) = (DECLAREARG x):(POPARG x):(peephole xs)
--peephole ((PUSHVAL x):(POP v):xs) = (SETVAL v x):(peephole xs)
--peephole ((DECLARE n@(MN _)):xs) = (DECLAREQUICK n):(peephole xs)
peephole ((PUSH x):(GETVAL t):(TMPSET tv v):(INFIX tv2 Plus vx vy):(USETMP b):(STACKTMP a s):(PUSH y):SETTOP:xs)
    | x == y = (ADDINPLACE x v):(peephole xs)
peephole ((PUSH x):(GETVAL t):(TMPSET tv v):(INFIX tv2 Minus vx vy):(USETMP b):(STACKTMP a s):(PUSH y):SETTOP:xs)
    | x == y = (SUBINPLACE x v):(peephole xs)
peephole ((PUSH (INT x)):(POP v):xs) = (SETINT v x):(peephole xs)
peephole ((PUSH (VAR x)):(POP v):xs) = (SETVAR v x):(peephole xs)
peephole ((PUSH (INT t)):(PUSH x):SETTOP:xs) = (PUSH x):(TMPSETTOP t):(peephole xs)
peephole ((USETMP b):(STACKTMP a s):(PUSH x):SETTOP:xs) = (PUSHSETINT x s):(peephole xs)
peephole ((STACKINT a i):(GETVAL t):xs) = (TMPSET t i):(peephole xs)
peephole ((STACKTMP a i):(JFALSE l):xs) = (JTFALSE i l):(peephole xs)
peephole ((STACKTMP a i):(JTRUE l):xs) = (JTTRUE i l):(peephole xs)
peephole ((PUSH x):SETTOP:xs) = (PUSHSETTOP x):(peephole xs)
peephole ((PUSH (INT x)):(JFALSE l):xs) = (JTFALSE x l):(peephole xs)
peephole ((PUSH (INT x)):(JTRUE l):xs) = (JTTRUE x l):(peephole xs)
peephole (CHR2STR:APPENDTOP:xs) = APPENDTOPINT:(peephole xs)
peephole ((PUSH (VAL x)):CHR2STR:APPENDTOP:xs) = (APPENDCHAR x):(peephole xs)
peephole ((PUSH (INT x)):CHR2STR:APPENDTOP:xs) = (APPENDTMP x):(peephole xs)
peephole ((PUSH (STR x)):APPENDTOP:xs) = (APPENDSTR x):(peephole xs)
peephole ((PUSH (STR x)):EQSTRING:(JFALSE l):xs) = (JNESTRINGW l x):(peephole xs)
peephole ((PUSH (STR x)):EQSTRING:(JTRUE l):xs) = (JEQSTRINGW l x):(peephole xs)
peephole ((PUSH (STR x)):NESTRING:(JFALSE l):xs) = (JEQSTRINGW l x):(peephole xs)
peephole ((PUSH (STR x)):NESTRING:(JTRUE l):xs) = (JNESTRINGW l x):(peephole xs)
peephole ((PUSH (STR x)):EQSTRING:xs) = (EQSTRINGW x):(peephole xs)
peephole ((PUSH (STR x)):NESTRING:xs) = (NESTRINGW x):(peephole xs)
peephole (EQSTRING:(JFALSE l):xs) = (JNESTRING l):(peephole xs)
peephole (EQSTRING:(JTRUE l):xs) = (JEQSTRING l):(peephole xs)
peephole (NESTRING:(JFALSE l):xs) = (JEQSTRING l):(peephole xs)
peephole (NESTRING:(JTRUE l):xs) = (JNESTRING l):(peephole xs)
peephole ((PUSH x):(GETVAL t):xs) = (PUSHGETVAL x t):(peephole xs)
peephole ((PUSH x):(GETRVAL t):xs) = (PUSHGETRVAL x t):(peephole xs)
peephole (INT2REAL:(GETRVAL t):xs) = (INTGETRVAL t):(peephole xs)
peephole ((PUSH x):(GETINDEX):xs) = (PUSHGETINDEX x):(peephole xs)
peephole ((PUSH x):(TOINDEX):xs) = (PUSHTOINDEX x):(peephole xs)
{- CIM: optimise (in)equality testing -}
peephole ((INFIX t op x y):(STACKINT a s):(JFALSE l):xs) = (INFIXJFALSE op x y l):(peephole xs)
peephole ((INFIX t op x y):(STACKINT a s):(JTRUE l):xs) = (INFIXJTRUE op x y l):(peephole xs)
peephole ((INFIX t op x y):(USETMP b):(STACKTMP a s):(JFALSE l):xs) = (INFIXJFALSE op x y l):(peephole xs)
peephole ((INFIX t op x y):(USETMP b):(STACKTMP a s):(JTRUE l):xs) = (INFIXJTRUE op x y l):(peephole xs)
peephole ((UNARY t op x):(USETMP b):(STACKTMP a s):(JFALSE l):xs) = (UNARYJFALSE op x l):(peephole xs)
peephole ((UNARY t op x):(USETMP b):(STACKTMP a s):(JTRUE l):xs) = (UNARYJTRUE op x l):(peephole xs)
peephole ((REALINFIXBOOL op x y):(JFALSE l):xs) = (INFIXJFALSE op x y l):(peephole xs)
peephole ((REALINFIXBOOL op x y):(JTRUE l):xs) = (INFIXJTRUE op x y l):(peephole xs)
{- CIM: optimise while(true|false) -}
peephole ((STACKINT a 0):(JFALSE l):xs) = (JUMP l):(peephole xs)
peephole ((STACKINT a _):(JFALSE l):xs) = (peephole xs)
peephole ((STACKINT a 0):(JTRUE l):xs) = (peephole xs)
peephole ((STACKINT a _):(JTRUE l):xs) = (JUMP l):(peephole xs)

-- optimise return values
peephole ((PUSH (REAL r)):SETRV:xs) = (SETRVREAL r):(peephole xs)

-- Slightly trickier string-in-place optimisations
{-
peephole ((PUSH x1):(PUSH (STR y)):APPEND:(PUSH x2):SETTOP:xs) 
         | x1 == x2 = (PUSH x1):(APPENDSTR y):DISCARD:(peephole xs)
peephole ((PUSH x1):(PUSH (VAL y)):CHR2STR:APPEND:(PUSH x2):SETTOP:xs) 
         | x1 == x2 = (PUSH x1):(APPENDCHAR y):DISCARD:(peephole xs)
peephole ((PUSH x1):(PUSH (INT y)):CHR2STR:APPEND:(PUSH x2):SETTOP:xs) 
         | x1 == x2 = (PUSH x1):(APPENDTMP y):DISCARD:(peephole xs)
-}

--peephole ((POPARG x):(PUSH (VAR y)):xs) | x==y = peephole xs
{-
peephole ((PUSH x):(PUSH y):(PUSH z):(PUSH w):xs) 
    = (PUSH4 x y z w):(peephole xs)
peephole ((PUSH x):(PUSH y):(PUSH z):xs) = (PUSH3 x y z):(peephole xs)
peephole ((PUSH x):(PUSH y):xs) = (PUSH2 x y):(peephole xs)
-}
peephole ((PUSH d@(VAL v)):(PUSH s):(PROJARG a t):SETTOP:xs) = (SETPROJARG s a t d):(peephole xs)

peephole ((SET v 0 x):xs) = (SETVAR v x):(peephole xs)
--peephole (RETURN:[]) = []

-- peephole (TAILCALL n:SETRV:TRIED:POPBT:xs) = (CALL n):TRIED:POPBT:(peephole xs)
-- peephole (TAILCALLNAME n:SETRV:TRIED:POPBT:xs) = (CALLNAME n):TRIED:POPBT:(peephole xs)
-- peephole (TAILCALLTOP:SETRV:TRIED:POPBT:xs) = CALLTOP:TRIED:POPBT:(peephole xs)
-- peephole (TAILCALL n:SETRV:POPBT:xs) = POPBT:(TAILCALL n):(peephole xs)
-- peephole (TAILCALLNAME n:SETRV:POPBT:xs) = POPBT:(TAILCALLNAME n):(peephole xs)
-- peephole (TAILCALLTOP:SETRV:POPBT:xs) = POPBT:TAILCALLTOP:(peephole xs)

peephole (TAILCALL n:POPBT:xs) = POPBT:(TAILCALL n):(peephole xs)
peephole (TAILCALLNAME n:POPBT:xs) = POPBT:(TAILCALLNAME n):(peephole xs)
peephole (TAILCALLTOP:POPBT:xs) = POPBT:TAILCALLTOP:(peephole xs)

peephole ((MKCON t a):SETRV:xs) = (MKCONRV t a):(peephole xs)
peephole ((MKCONZERO t):SETRV:xs) = (MKCONZERORV t):(peephole xs)
-- are these three really tail recursive?
-- probably not, but from the point of view of POPBT, yes.
-- on the other hand, if these actually get used, there's an error elsewhere
peephole (CALL n:SETRV:POPBT:xs) = POPBT:(TAILCALL n):SETRV:(peephole xs)
peephole (CALLNAME n:SETRV:POPBT:xs) = POPBT:(TAILCALLNAME n):SETRV:(peephole xs)
peephole (CALLTOP:SETRV:POPBT:xs) = POPBT:TAILCALLTOP:SETRV:(peephole xs)
-- these should always be void function calls
peephole (CALL n:POPBT:xs) = POPBT:(TAILCALL n):(peephole xs)
peephole (CALLNAME n:POPBT:xs) = POPBT:(TAILCALLNAME n):(peephole xs)
peephole (CALLTOP:POPBT:xs) = POPBT:TAILCALLTOP:(peephole xs)
peephole (CALL n:DISCARD:POPBT:xs) = POPBT:(TAILCALL n):DISCARD:(peephole xs)
peephole (CALLNAME n:DISCARD:POPBT:xs) = POPBT:(TAILCALLNAME n):DISCARD:(peephole xs)
peephole (CALLTOP:DISCARD:POPBT:xs) = POPBT:TAILCALLTOP:DISCARD:(peephole xs)
peephole (CASE cs:xs) = CASE (map peephole cs):(peephole xs)
peephole ((CONSTCASE ty cs defs):xs) 
    = CONSTCASE ty (map (\ (c,cd) -> (c, peephole cd)) cs) (peephole defs):
        (peephole xs)
peephole ((LINENO _ _):cs@((LINENO _ _):_)) = peephole cs
peephole (x:xs) = x:(peephole xs)

peephole' ((USETMP _):xs) = peephole' xs
peephole' (x:xs) = x:(peephole' xs)
peephole' [] = []

foreigncomp :: Monad m => Type -> Name -> [(Expr Name,Type)] -> CompData -> 
	       Int -> Int -> m ([TAC], Int, Int)
foreigncomp ty n as cd lab stmp = 
    do (pcode, lab', stmp') <- fpushargs as cd lab stmp
       return (pcode ++ foreigncall ty n (map snd as), lab', stmp)
  where fpushargs [] _ lab stmp = return ([], lab, stmp)
        fpushargs ((x, ty):xs) cd@(CD mod vars tmp b tr cache ds vs vinfo) lab stmp
            = do (pcode, lab',stmp') <- fpushargs xs (CD mod vars tmp b tr cache ds vs vinfo) lab stmp
	         (xcode, lab'',stmp'') <- cmpfn ty x cd lab' stmp'
	         return (pcode ++ xcode, lab'',stmp'')
        cmpfn (Prim Exception) = fMkHeap
        cmpfn (Prim _) = fcompile'
        cmpfn _ = fMkHeap

foreigncall :: Type -> Name -> [Type] -> [TAC]
{- old way
foreigncall ty (UN n) args = [USETMP (tmpval n) | n <- [0..(length args)-1]]
			     ++ [RAWC ((popvals 0 args)++"\n\t"++
			        (conv ty)++"("++ n ++
			       "("++stackconv 0 args++")))")]
-}		     
foreigncall ty (UN n) args = [USETMP ((tmpval n), True) | n <- [0..(length args)-1]]
			     ++ [FOREIGNCALL n "nolibsyet" ty args]

{-   where conv (Prim Void) = "VOID("
	 conv (Prim Number) = "PUSH(MKINT"
	 conv (Prim RealNum) = "PUSH(MKREAL"
	 conv (Prim Boolean) = "PUSH(MKINT"
	 conv (Prim Character) = "PUSH(MKCHAR"
	 conv (Prim StringType) = "PUSH(MKSTR"
	 conv (Prim Pointer) = "PUSH(MKINT"
	 conv (TyVar _) = "PUSH(" -- Enough rope to hang yourself with!
	 conv (Array _) = "PUSH(MKARRAYVAL"
	 conv t = error $ "Can't deal with that type in foreign calls" ++ show t

         popvals n [] = ""
	 popvals n (x:xs) = show (tmpval n) ++ 
			    " = vm->doPop(); " ++ popvals (n+1) xs

         stackconv n [] = ""
	 stackconv n [x] = stackconv' n x
	 stackconv n (x:xs) = stackconv' n x ++ "," ++ stackconv (n+1) xs
	 stackconv' n (Prim Number) = show (tmpval n) ++ "->getInt()"
	 stackconv' n (Prim RealNum) = show (tmpval n) ++ "->getReal()"
	 stackconv' n (Prim Boolean) = show (tmpval n) ++ "->getInt()"
	 stackconv' n (Prim Character) = show (tmpval n) ++ "->getInt()"
	 stackconv' n (Prim StringType) 
	     = show (tmpval n) ++ "->getString()->getVal()"
	 stackconv' n (Prim Pointer) = show (tmpval n) ++ "->getRaw()"
	 stackconv' n (Array _) = show (tmpval n) ++ "->getArray()"
	 stackconv' n (TyVar _) = show (tmpval n)
	 
	 stackconv' _ t = error $ "Can't deal with that type (" ++ show t ++ ") in foreign calls"
-}

mkStartup :: Program -> CompileResult
mkStartup p = let startups = callStartups p in
                RawCode $ "void _my_runatstartup(VMState* vm) {\n\tDECLAREQUICK(stmp);\n" ++ startups ++ "}"

callStartups [] = ""
callStartups ((FunBind (_,_,n,ty,fopts,_) _ _):xs) 
  | StartupFn `elem` fopts =
      "\tPUSH(stmp);\n"++
      "\tCALL(" ++ show n ++ mangling ty ++ ");\n" ++ callStartups xs
callStartups (_:xs) = callStartups xs


type FunMap = [(String,Int)]

mkfnmap :: Program -> CompileResult
mkfnmap p = let fm = mkf' 0 p
                ifm = mkfinit fm
		mkfm = mkfcode fm in
		RawCode $ "void _my_initfunmap(VMState* vm) {\n " ++ ifm ++ mkfm ++ "}"

mkf' :: Int -> Program -> FunMap
mkf' i [] = []
-- Add any functions which take one argument and return nothing.
--mkf' i ((FunBind (_,_,n,Fn _ [x] (Prim Void),_,_) _):xs) = (n,i):(mkf' (i+1) xs)
-- No, we now add all functions, including lambda lifted ones.
mkf' i ((FunBind (_,_,n,ty@(Fn _ _ _),_,_) _ _):xs) 
    = (show n ++ mangling ty, i):(mkf' (i+1) xs)
mkf' i ((FMName x):xs) = (x, i):(mkf' (i+1) xs)
mkf' i (x:xs) = mkf' (i+1) xs

mkfinit :: FunMap -> String
mkfinit fm = mkfi' fm (hash (show fm))
  where mkfi' [] h = ""
        mkfi' ((str,i):[]) h = "\tinitFunMap(" ++ show (i+1) ++", (kint)" ++ 
                               show h ++ ");\n"
        mkfi' ((str,i):xs) h = mkfi' xs h

mkfcode :: FunMap -> String
mkfcode [] = ""
mkfcode ((str,i):xs) = "\taddToFunMap("++show i++", "++ str ++");\n" ++
		       mkfcode xs

mkaeskey :: Int -- Deterministic if nonzero
         -> IO CompileResult
mkaeskey 0 = do key <- mkkey False 32 0
		return (RawCode $ "const char* kaya_secret = \"KSB" ++ key ++ "KSE\";")
mkaeskey seed = do key <- mkkey True 32 seed
		   return 
		      (RawCode $ "const char* kaya_secret = \"KSB" ++ key ++ "KSE\";")
--	      return $ RawCode $ "char* kaya_secret = \"\\242\\052\\233\\025\\198\\096\\164\\249\\003\\060\\103\\245\\221\\110\\124\\061\\148\\090\\015\\149\\200\\240\\052\\020\";"

mkivec :: Int -> IO CompileResult
mkivec seed = do key <- mkkey (seed /= 0) 16 seed
		 return 
		   (RawCode $ "const char* kaya_ivec = \"KIB" ++ key ++ "KIE\";")
--	    return $ RawCode $ "char* kaya_ivec = \"\\208\\079\\091\\021\\123\\197\\092\\156\";"

showtac :: [CompileResult] -> String
showtac [] = ""
showtac ((ByteCode (n,ty,(Code pop _ main))):xs) = showuser n ++ ": " ++ mangling ty ++ "\n" ++ showbc 1 (pop ++ main) ++ 
				"\n\n" ++ showtac xs
  where showbc i [] = ""
	showbc i ((CASE bcs):xs) = (take i (repeat '\t')) ++ 
				  "CASE\n" ++ showcase (i+1) 0 bcs ++ showbc i xs
	showbc i (x:xs) = (take i (repeat '\t'))++show x ++ "\n" ++ showbc i xs
	showcase i c [] = ""
	showcase i c (bc:bcs) = (take (i-1) (repeat '\t')) ++ show c ++ 
				":\n" ++ showbc i bc ++ showcase i (c+1) bcs
showtac (x:xs) = showtac xs

hash [] = 0
hash (x:xs) = 131*(hash xs)+(fromEnum x)
