/***************************************************************************
 *   Copyright (C) 2004 by Stefano Salvador                                *
 *   salva_ste@tin.it                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef KATALOG_H
#define KATALOG_H

#include <qmap.h>
#include <qdom.h>
#include <qobject.h>
#include <qstringlist.h>

#include <kio/job.h>
#include <kurl.h>

#include "kataloguds.h"

class QDate;

class KatalogJobItem{
public:
  KatalogJobItem();
  
  KatalogJobItem(KURL& _u, QStringList& _p);
  
  KURL url() const;
  
  QStringList path() const;
  
  bool isEmpty();
  
  bool operator==( const KatalogJobItem & kji ) const;
  
private:
  KURL u;
  QStringList p;
};

typedef QValueList<KatalogJobItem> KatalogJobs;
typedef QValueListIterator<KatalogJobItem> KatalogJobsIterator;

/**
@author Stefano Salvador
*/
class Katalog : public QObject{
Q_OBJECT
public:
  Katalog();

  virtual  ~Katalog();

  enum Type { E_LINK = 1, E_DIR = 2, E_FILE = 4 };

  /** Enumerator for error codes in init document */
  enum InitErrors { UNREADABLE_URL = 1, NOT_KATALOG = 2, UNCOMPLETE = 4 };
  
  /** Enumerator for error codes in creating a new catalog */
  enum CatErrorrs { NAME_EMPTY = 2 };
  
  static const int SUCCESS = 0;
    
  /** Init the document for further operations */
  int initDocument(const KURL& url);
  
  /** Save the current document to file */
  bool saveDocument(const KURL& file, const char *format = "application/x-gzip");
  
  /** Adds the items found in mount to the catalog, creating it if not exist */
  int addItems(const KURL& mount,
               QString catalog,
               bool exploreArchives = false,
               bool getMetaInfo = false);
  
  bool isChanged();

  /** Get the list of contents in a node */
  KatalogUDSEntryList getNodeContent(QStringList& fullPath) const;
  
  /** Get the given entry */
  KatalogUDSEntry findEntry(QStringList& fullPath) const;
  
  /** Rename a node */
  bool rename(QStringList& path, QString& newName);
  
  /** Delete a node */
  void del(QStringList& path);
  
  /** Returns the original URL of the stored file */
  KURL sourceURL(QStringList& path) const;
  
  /** Returns the Meta Info of the specified file */
  QString readInfo(QStringList& path) const;  
       
private:
  /** Map of open jobs */
  KatalogJobs m_jobs;
  
  /** The DOM document */
  QDomDocument m_document;
  
  /** The root node of the DOM tree*/
  QDomElement m_rootElement;
  
  /** The url now listed*/
  QString m_currentUrl;

  /** flag: true for exploring compressed archives */
  bool m_exploreArchives;

  /** flag: true for reading and storing Meta Infos */
  bool m_getMetaInfo;
  
  /** flag: true if the document is changed since the initDocument call */
  bool m_changed;
  
  KatalogJobItem find(const KURL& url) const;

protected:
  /** Find node relative to path */
  QDomNode findNode(QStringList& path) const;
  
  KatalogUDSEntry createUDSEntry(QDomElement el) const;
  
  int itemsInNode(QDomNode& node) const;
  
  int totalItems() const;
  
  int totalCatalogs() const;
  
private slots:
  void slotEntries(KIO::Job *, const KIO::UDSEntryList &);
  void slotResult(KIO::Job *);
  void slotRedirection(KIO::Job *, const KURL&);

signals:
  void finished(QString);
};

#endif
