#include <qcstring.h>
#include <qsocket.h>
#include <qdatetime.h>
#include <qbitarray.h>
#include <qfile.h>
#include <qdom.h>


#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <iostream>

#include <kapplication.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kinstance.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <klocale.h>
#include <kurl.h>
#include <ksock.h>

#include "katalogslave.h"

#include <katalog.h>

using namespace KIO;

kio_katalogProtocol::kio_katalogProtocol(const QCString &pool_socket, const QCString &app_socket)
    : SlaveBase("kio_katalogslave", pool_socket, app_socket)
{
  attach();
  appId = registerAs("katalogSlave");
}

void kio_katalogProtocol::wakeDCOP()
{
  if(isApplicationRegistered("katalogdcop"))
  {
    return; 
  }
  
  QString app("katalogdcop.desktop");
  
  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  QStringList URLs;
  arg << app << URLs;
  call( "klauncher", "klauncher", "start_service_by_desktop_path(QString,QStringList)",  
        data, replyType, replyData);
}

kio_katalogProtocol::~kio_katalogProtocol()
{
  if(isApplicationRegistered("katalogdcop"))
  {
    QByteArray data;
    QDataStream arg(data, IO_WriteOnly);
    send("katalogdcop", "katalogdcopInterface", "saveDocuments()", data);
    send("katalogdcop", "katalogdcopInterface", "exit()", data);
  }
  
  detach();
}

bool kio_katalogProtocol::checkNewFile( const KURL & url, QString & path )
{
  wakeDCOP();
  
  QString fullPath = url.path();
  
  // Find where the katalog file is in the full path
  int pos = 0;
  QString katalogFile;
  path = QString::null;
  
  int len = fullPath.length();
  if ( len != 0 && fullPath[ len - 1 ] != '/' )
    fullPath += '/';

  while ( (pos=fullPath.find( '/', pos+1 )) != -1 )
  {
    QString tryPath = fullPath.left( pos );
    struct stat statbuf;
    if ( ::stat( QFile::encodeName(tryPath), &statbuf ) == 0 && !S_ISDIR(statbuf.st_mode) )
    {
      katalogFile = tryPath;
      m_mtime = statbuf.st_mtime;
      path = fullPath.mid( pos + 1 );
      len = path.length();
      if ( len > 1 )
      {
        if ( path[ len - 1 ] == '/' )
          path.truncate( len - 1 );
      }
      else
        path = QString::fromLatin1("/");
      break;
    }
  }
  
  if ( katalogFile.isEmpty() )
    return false;

  // Open new file
  m_katalogURL = new KURL();
  m_katalogURL->setProtocol("file");
  m_katalogURL->setPath(katalogFile);
  
  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  arg << *m_katalogURL;
  if (!call("katalogdcop", "katalogdcopInterface", "initDocument(KURL)",
            data, replyType, replyData))
  {
    return false;
  }
  else {
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "int") {
      int result;
      reply >> result;
      if(result == 0)
        return true;
    } else
      return false;
  }
  
  return false;
}

void kio_katalogProtocol::get(const KURL& url )
{}

// stat is used by the client to decide if the url
// refers to a file or a dir.
void kio_katalogProtocol::stat( const KURL & url )
{
  QString path;
  UDSEntry *entry;
  if ( !checkNewFile( url, path ) )
  {
    error( KIO::ERR_DOES_NOT_EXIST, url.prettyURL() );
    return;
  }
  
  if ( path.isEmpty() )
  {
    KURL redir( url.protocol() + QString::fromLatin1( ":/") );
    redir.setPath( url.path() + QString::fromLatin1("/") );
    redirection( redir );
    finished();
    return;
  }

  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  arg << *m_katalogURL << path;
  if (!call("katalogdcop", "katalogdcopInterface", "findEntry(KURL, QString)",
            data, replyType, replyData))
  {
    error( KIO::ERR_DOES_NOT_EXIST, url.prettyURL() );
    return;
  }
  else {
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "KatalogUDSEntry")
    {
      KatalogUDSEntry *result = new KatalogUDSEntry();
      reply >> *result;
      if(result->isEmpty())
      {
        error( KIO::ERR_DOES_NOT_EXIST, url.prettyURL() );
        return;
      }
      entry = (UDSEntry *)result;
      
    }
    else
    {
      error( KIO::ERR_DOES_NOT_EXIST, url.prettyURL() );
      return;
    }
  }
    
  statEntry( *entry );
  
  delete entry;
  
  finished();
}

// listDir list the contents of the url passed
// as argument (if it's a dir, see stat), to costruct a tree
// call recursively this method.
void kio_katalogProtocol::listDir( const KURL& url)
{
  QString path;
  if ( !checkNewFile( url, path ) )
  {
    error( KIO::ERR_DOES_NOT_EXIST, url.prettyURL() );
    return;
  }
    
  UDSEntryList *entries;
  KatalogUDSEntryList *result = new KatalogUDSEntryList();
  
  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  arg << *m_katalogURL << path;
  if (!call("katalogdcop", "katalogdcopInterface", "getNodeContent(KURL, QString)",
            data, replyType, replyData))
  {
    error( KIO::ERR_DOES_NOT_EXIST, url.prettyURL() );
    return;
  }
  else {
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "KatalogUDSEntryList")
    {
      reply >> *result;
      entries = (UDSEntryList *)result;
    }
    else
    {
      error( KIO::ERR_DOES_NOT_EXIST, url.prettyURL() );
      return;
    }
  }
    
  totalSize( entries->count() );

  UDSEntryListConstIterator it;
  for ( it = entries->begin(); it != entries->end(); ++it )
    listEntry( *it, false );
  
  listEntry( *it, true ); // ready
  
  delete entries;
  
  finished();
}

void kio_katalogProtocol::rename(const KURL &url, const KURL &dest, bool overwrite)
{

  QString pathIn = url.path();
  pathIn = pathIn.left(pathIn.findRev('/', -2)); 
  QString pathOut = dest.path();
  pathOut = pathOut.left(pathOut.findRev('/', -2)); 
  
  if(pathIn != pathOut)
  {
    error( KIO::ERR_CANNOT_RENAME, url.prettyURL() );
    return;
  }

  if(url.protocol() != "katalog" || dest.protocol() != "katalog")
  {
    error( KIO::ERR_UNSUPPORTED_PROTOCOL, url.prettyURL() );
    return;
  }
  
  if(url.fileName() == dest.fileName())
  {
    error( KIO::ERR_IDENTICAL_FILES, url.prettyURL() );
    return;
  }
  
  QString path;
  if ( !checkNewFile( url, path ) )
  {
    error( KIO::ERR_DOES_NOT_EXIST, url.prettyURL() );
    return;
  }
    
  // I assume that it's possible to rename only an item
  // in another item inside the same folder. Otherwise
  // some obscure error may happen.
  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  arg << *m_katalogURL << path << dest.fileName();
  if (!call("katalogdcop", "katalogdcopInterface", "rename(KURL, QString, QString)",
            data, replyType, replyData))
  {
    error( KIO::ERR_WRITE_ACCESS_DENIED, url.prettyURL() );
    return;
  }
  else {
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "bool")
    {
      bool result;
      reply >> result;
      if(!result)
      {
        error( KIO::ERR_CANNOT_RENAME, url.prettyURL() );
        return;
      }
    }
    else
    {
      error( KIO::ERR_UNKNOWN, url.prettyURL() );
      return;
    }
  }
  
  finished();
}

void kio_katalogProtocol::del(const KURL & url, bool b)
{
  QString path;
  if ( !checkNewFile( url, path ) )
  {
    error( KIO::ERR_DOES_NOT_EXIST, url.prettyURL() );
    return;
  }
      
  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  arg << *m_katalogURL << path;
  if (!call("katalogdcop", "katalogdcopInterface", "del(KURL, QString)",
            data, replyType, replyData))
  {
    error( KIO::ERR_WRITE_ACCESS_DENIED, url.prettyURL() );
    return;
  }
  else
  {
    finished();
  }
}

extern "C"
{
  int kdemain(int argc, char **argv)
  {
    KInstance instance( "kio_katalogslave" );
    if (argc != 4) {
      exit(-1);
    }
    
    kio_katalogProtocol slave(argv[2], argv[3]);
    slave.dispatchLoop();

    return 0;
  }
}
