/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef NOTIFICATION_H
#define NOTIFICATION_H

#include <qmap.h>
#include <qobject.h>
#include <qstring.h>
#include <qstringlist.h>

#include "userlist.h"

#include "../notify/notification.h"

class QTimer;

/**
	@class Notification
	@author Rafa 'Vogel' Malinowski
	@brief Klasa informujca uytkownika o zajciu pewnego zdarzenia.

	Zdarzenia identyfikowane s poprzez nazwy. Kade zdarzenie musi zosta zarejestrowane
	przed uyciem w klasie Notify a takze wyrejestrowane w przypadku wyadowania moduu,
	ktry wywoywa dane zdarzenie.

	Do kadego zdarzenia moe by przypisana ikona, lista kontaktw oraz lista akcji, ktre uytkownik
	moe podj w odpowiedzi na zdarzenie. Przykadowo, zdarzenie polegajce na odebraniu wiadomoci od
	anonimowego kontaktu, z ktrym nie prowadzimy aktualnie rozmowy moe mie posta:

	<ul>
		<li>nazwa: newChatWithAnonymous</li>
		<li>ikona: chat</li>
		<li>lista kontaktw: kontakt, ktry rozpocz rozmow</li>
		<li>akcja: odbierz komunikat</li>
		<li>akcja: ignoruj komunikat</li>
	</ul>

	Akcje uytkownika implementowane s w obiektach potomnych w postaci w postaci slotw.
	Domylnie zaimplementowane s dwa sloty wirtualne: callbackAccept i callbackDiscard,
	odpowiadajce zaakceptowaniu i odrzuceniu zdarzenia.

	Dodanie nowych akcji polega na dodaniu nowych slotw do klas potomnych. Aktywacja
	akcji (take tych dwch domylnych) dokonuje sie poprzez metod addCallback(caption, slot),
	na przykad:

	<code>
		addCallback(tr("Odbierz rozmow"), SLOT(callbackAccept()));
		addCallback(tr("Ignoruj rozmow"), SLOT(callbackDiscard()));
	</code>

	Obiekty mog automatycznie podejmowa akcje domylne po przekroczeniu pewnego
	okresu czasu:

	<code>
		setDefaultCallback(2000, SLOT(callbackAccept()));
	</code>

	Mona takze anulowa akcj domyln:

	<code>
		clearDefaultCallback();
	</code>

	Notyfikacja, co do ktrej pewna akcja zostaa wykonana, wywouje sygna closed().
 **/
class Notification : public QObject {
	Q_OBJECT

private:

	QString Type;
	UserListElements Ule;

	QString Title;
	QString Text;
	QString Details;
	QString Icon;

	QValueList<QPair<QString, const char *> > Callbacks;
	QTimer *DefaultCallbackTimer;

	int ReferencesCount;
	bool Closing;

public:

	/**
		Tworzy now notfikacj o zadanym typie, ikonie i zadanej asocjacji z kontaktami.
		Klasy potome zazwyczaj maj wasne kontruktory, ktre wypeniaj automatycznie
		parametry tego konstruktora.

		@arg type typ zdarzenia
		@arg icon nazwa ikony zdarzenia
		@arg userListElements lista kontaktw powizana ze zdarzeniem
	 **/
	Notification(const QString &type, const QString &icon, const UserListElements &userListElements);
	virtual ~Notification();

	/**
		Wywoywane przez notyfikator, ktry zajmuje si danym zdarzeniem.
	 **/
	void acquire();
	/**
		Wywoywane przez notyfikator, ktry przestaje zajmowa si danym zdarzeniem.
		Gdy aden notyfikator nie zajmuje si danym zdarzeniem, zdarzenie jest zwalniane.
		Wystpuje to na przykad w przypadku moduw dwikowych czy moduu hints, gdy
		dymek zniknie po upywie okrelonego czasu a nie przez zdarzenie wywoane przez uytkownika.
	 **/
	void release();

	/**
		Zamyka zdarzenie. Wywouje sygna closed() i usuwa obiekt.
	 **/
	void close();

	/**
		Usuwa akcje uytkownika
	 **/
	void clearCallbacks();
	/**
		Dodaje akcje uytkownika.

		@arg caption wywietlana nazwa akcji
		@arg slot slot wywoywany w przypadku wybrania przez uytkownika danej akcji
	 **/
	void addCallback(const QString &caption, const char *slot);

	/**
		Ustawia akcj domyln.

		@arg timeout liczba milisekund, jaka musi upyn przed wykonaniem domylnej akcji
		@arg slot slot wywoywany w przypadku upynicia czasu
	 **/
	void setDefaultCallback(int timeout, const char *slot);

	/**
		Typ zdarzenia.

		@return typ zdarzenia
	 **/
	QString type() const;
	/**
		Lista kontaktw powizana ze zdarzeniem/

		@return lista kontaktw powizana ze zdarzeniem
	 **/
	const UserListElements &userListElements() const;

	/**
		Ustawia tytu zdarzenia.
	 **/
	void setTitle(const QString &title);
	/**
		Tytu zdarzenia.

		@return tytu zdarzenia
	 **/
	QString title() const;

	/**
		Ustawia tre zdarzenia.
	 **/
	void setText(const QString &text);
	/**
		Tre zdarzenia.

		@return tre zdarzenia
	 **/
	QString text() const;

	/**
		Ustawia szczegy zdarzenia (jak na przykad tekst wiadomoci).
	 **/
	void setDetails(const QString &details);
	/**
		Szczegy zdarzenia
	 **/
	QString details() const;

	/**
		Ustawia ikon zdarzenia.
	 **/
	void setIcon(const QString &icon);
	/**
		Ikona zdarzenia.

		@return ikona zdarzenia
	 **/
	QString icon() const;

	/**
		Lista akcji.

		@return lista akcji
	 **/
	const QValueList<QPair<QString, const char *> > & getCallbacks();

public slots:
	/**
		Domylny slot wywoywany w przypadku zaakceptowania zdarzenia.
	 **/
	virtual void callbackAccept();
	/**
		Domylny slot wywoywany w przypadku odrzucenia zdarzenia.
	 **/
	virtual void callbackDiscard();

	/**
		Slot anulujcy domyln akcj - wywoywany rcznie przy wyborze dowolnej innej akcji.
	 **/
	void clearDefaultCallback();

signals:
	/**
		Sygna wysylany przy zamykaniu zdarzenia, po wyborze przez uytkownika dowolnej akcji.
	 **/
	void closed(Notification *);

};

#endif // NOTIFICATION_H
