#ifndef KADU_PROTOCOL_H
#define KADU_PROTOCOL_H

#include <qobject.h>
#include <qstring.h>
#include <qdatetime.h>

#include "status.h"
#include "usergroup.h"

typedef uint32_t UinType;

class Protocol : public QObject
{
	Q_OBJECT

	private:
		UserStatus &writeableStatus() { return *NextStatus; }
		friend class Kadu;

	protected:
		QDateTime ConnectionTime;

		QString ProtocolID;
		QString id;

		/**
			Biecy status. Zmieniany po poczeniu, oraz w przypadku zmiany statusu kiedy poczenie
			jest ju zainicjowane.

			@see login
			@see connected
			@see NextStatus
			@see UserStatus
			@see status
		**/
		UserStatus *CurrentStatus;

		/**
			Nastpny status. Ustalany zewntrznie przy wykorzystaniu metody status i odpowiednich
			slotw klasy UserStatus. Zmiana wywouje jedn z metod iWantGo... i w konsekwencji zmian
			statusu (w razie koniecznoci te zalogowanie).

			@see login
			@see connected
			@see CurrentStatus
			@see UserStatus
			@see status
		**/
		UserStatus *NextStatus;

	public:
		Protocol(const QString &proto, const QString &id, QObject *parent = 0, const char *name = 0);
		virtual ~Protocol();

		/**
			Status uytkownika. Za pomoc tej metody moemy go zmieni, pobra ikon statusu i wykona
			kilka innych ciekawych rzeczy.

			1. Zmiana statusu:
			<code>
				Protocol *proto;

				...

				proto->status().setOnline("Jestem zalogowany"); // zalogowanie i ustawienie opisu
				proto->status().setFriendsOnly(true);           // tryb tylko dla przyjaci
				...
				proto->status().setOffline();                   // wylogowanie, usunicie opisu
			</code>

			2. Sprawdzenie statusu:
			<code>
				Protocol *proto;

				if (proto->status().isOnline())                 // jestemy online
					...
				else if (proto->status().isInvisible())         // jestemy niewidzialni
					...

				// mona te:
				switch (proto->status().status())
				{
					case Online:
						break;
					case Busy:
						break;
					case Invisible:
						break;
					case Offline:
						break;
				}
			</code>

			3. Pobranie ikony i nazwy statusu
			<code>
				QPixmap pix;
				QString name;
				Protocol *proto;

				...

				pix = proto->status().pixmap();
				name = proto->status().name();
			</code>

			@see currentStatus
		**/
		const UserStatus &nextStatus() { return *NextStatus; }

		/**
			Rzeczywisty aktualny status. Mona go wykorzysta tylko w trybie do odczytu (pobranie
			ikony, nazwy, sprawdzenie rzeczywistego stanu poczenia).

			@see status
		**/
		const UserStatus &currentStatus() const { return *CurrentStatus; }

		QString protocolID() const { return ProtocolID; }

		QString ID() const { return id; }

		virtual bool validateUserID(QString& uid) = 0;

		virtual UserStatus *newStatus() const = 0;

		const QDateTime &connectionTime() const;

	public slots:
		/**
			Wysya wiadomo bez formatowania tekstu. Jeli adresatw jest wicej ni jeden, to wysyana
			jest wiadomo konferencyjna. Zwracany jest numer sekwencyjny wiadomoci, jeli
			przypadkiem bymy chcieli ledzi jej potwierdzenie.
			@param users lista uytkownikw, do ktrych wysyamy wiadomo
			@param mesg wiadomo, ktr wysyamy - kodowanie zmieniane wewntrz
		**/
		virtual QString sendMessage(UserListElements users, const QString &mesg) = 0;
		/**
			Wysya wiadomo bez formatowania tekstu. Zwracany jest numer sekwencyjny wiadomoci, jeli
			przypadkiem bymy chcieli ledzi jej potwierdzenie.
			@param users lista uytkownikw, do ktrych wysyamy wiadomo
			@param mesg wiadomo, ktr wysyamy - kodowanie zmieniane wewntrz
		**/
		QString sendMessage(UserListElement user, const QString &mesg);


	signals:

		/**
			udao si zalogowa
		**/
		void connected();

		/**
			rozpoczynamy procedur logowania si
		**/
		void connecting();

		/**
			rozczylimy si z serwerem
		**/
		void disconnected();

		/**
			wystpi bd poczenia
			@param protocol protok
			@param reason napis do wywietlenia dla uytkownika
		**/
		void connectionError(Protocol *protocol, const QString &server, const QString &reason);

		/**
			\fn void messageFiltering(const UserGroup *users, QCString& msg, bool& stop)
			Sygnal daje mozliwosc operowania na wiadomosci
			ktora ma byc wyslana do serwera juz w jej docelowej
			formie po konwersji z unicode i innymi zabiegami.
			Tresc wiadomosci mozna zmienic podmieniajac wskaznik
			msg na nowy bufor i zwalniajac stary (za pomoca free).
			Mozna tez przerwac dalsza jej obrobke ustawiajac
			wskaznik stop na true.
			\param users lista uytkownikw
			\param msg wiadomo
			\param stop zakoczenie dalszej obrbki sygnau
		**/
		void sendMessageFiltering(const UserListElements users, QString &msg, bool &stop);
		/**
			wiadomo nie zostaa dostaczona
		**/
		void messageNotDelivered(const QString &message);
		/**
			wiadomo zostaa przyjta przez serwer
			TODO: WTF??
		**/
		void messageAccepted();

		/**
			\fn receivedMessageFilter(Protocol *protocol, UserListElements senders, const QString &msg, time_t time, bool &ignore);
			Filtrujemy wiadomo. Mona j odrzuci albo i nie.
			\param protocol protok na ktrym otrzymalimy wiadomo
			\param senders lista nadawcw
			\param message komunikat w postaci Unicode HTML
			\param time czas nadania wiadomoci
			\param ignore po ustawieniu na true wiadomo jest ignorowana
		**/
		void receivedMessageFilter(Protocol *protocol, UserListElements senders, const QString &message, time_t time, bool &ignore);
		/**
			\fn messageReceived(Protocol *protocol, UserListElements senders, const QString &msg, time_t time);
			Otrzymalimy wiadomo.
			\param protocol protok na ktrym otrzymalimy wiadomo
			\param senders lista nadawcw
			\param message komunikat w postaci Unicode HTML
			\param time czas nadania wiadomoci
		**/
		void messageReceived(Protocol *protocol, UserListElements senders, const QString &message, time_t time);

	private:
		Protocol(const Protocol &) {}
		virtual Protocol &operator=(const Protocol &){return *this;}
};

#endif
