/**
* Displays the Mail User Agent (MUA) aka mail program
* which sent the current message.
*
* @author Christian Weiske <cweiske@cweiske.de>
*
* The icons have been made by myself (most of them),
* while some have been created by dispMUA users. They're
* too much to list here.
*
*
* @todo:
*  - option pane: display version as text in statusbar or so
*  - dispMUA is not thread safe - if you open a second tb window,
*       select another message which and click the icon in the first
*       window, the popup will show the wrong user agent.
*       But I doubt that anyone will bother :)
*
* @see useragents.txt for a list of known (not necessarily supported) MUAs
*/
var dispMuaGlobals = {
    strDispMUAstring: "",
    //thunderbird 1.0 doesn't notify onEndHeaders listener
    //this bug has been reported and is fixed in 1.1
    bTimeoutWorkaround: false,
    bLastUnknown: false,

    arDispMuaOverlay:    new Array(),//user overlay
    strOverlayFilename:  "dispMuaOverlay.csv",//user overlay filename

    /**
    *   The array has the following structure:
    *   - The array contains a number of subarrays.
    *   - The subarrays' keys are the strings which the user agent line has to contain to be
    *       associated to the icon in the value
    *   - The first entry, "presearch", contains some identifiers which may occur in the mid
    *       of an identification line and would not be recognized in the postsearch array, because
    *       other keys would match
    *   - There are 26 subarrays, one for each letter: The user agent keys in them *have to begin*
    *       with this letter
    *   - If there is a user agent identification line which doesn't begin with a
    *       determined letter, but contains a certain sequence after the first letter,
    *       it goes in the "others" subarray.
    *
    *   Now the search algorithm is the following:
    *   1. Search in mostwanted. If one of the keys matches, use the icon
    *   2. Get the first letter of the user agent identification and search through the
    *       corresponding subarray
    *   3. If it's still not found, search through the "others" array and find a matching
    *       key
    */
    // {{{
    arDispMuaAllocation: {
        //The user agents used at most - this array is accessed at first
        "presearch": {
            "thunderbird"              : "thunderbird.png",//is sometimes in the mid of the identifier
            "seamonkey"                : "seamonkey.png",
            "mozilla"                  : "mozilla.png"
        },
        //User agents beginning with A
        "a" : {
            "abvmail"                  : "abvmail.png",
            "accucast"                 : "accucast.png",
            "aconon mail software"     : "aconon.png",
            "acro media inc"           : "acromedia.png",
            "active! mail"             : "active_mail_jp.png",
            "active.mail"              : "active.mail.png",
            "activxperts activemail"   : "activxperts.png",
            "admin pro tools email list manager" : "apt_email_list_manager.png",
            "adobe photoshop services" : "photoshop.png",
            "adtoone"                  : "adtoone.png",
            "advalvas mailer"          : "advalvas.png",
            "aeromail"                 : "aeromail.png",
            "agendus mail"             : "agendus.png",
            "agnitas ag/conrad"        : "conrad.png",
            "agnitas emm"              : "agnitas.png",
            "aim webmail"              : "aol.png",
            "airmail for windows"      : "airmail.png",
            "airt"                     : "airt.png",
            "ak-mail"                  : "ak_mail.png",
            "akio mail center"         : "akio.png",
            "al-mail32"                : "almail.png",
            "alerts delivery system, marketwatch"   : "marketwatch.png",
            "ambit new media"          : "ambit.png",
            "amicron mailoffice"       : "amicron.png",
            "anglemail"                : "anglemail.png",
            "animexx mailritter"       : "animexx.png",
            "aol"                      : "aol.png",
            "apple mail"               : "apple_mail.png",
            "apple .mac mailer"        : "apple_store.png",
            "applixware"               : "applixware.png",
            "arclab maillist controller"    : "arclab.png",
            "arena internet mailer"    : "arena_internet_mailer.png",
            "asp component bundle"     : "aspqmail.png",
            "aspmail"                  : "aspqmail.png",
            "aspnetemail"              : "aspnetemail.png",
            "aspqmail"                 : "aspqmail.png",
            "asp smtpsvg"              : "aspqmail.png",
            "aster webmail"            : "aster.png",
            "at&t message center"      : "att.png",
            "atlantisworld"            : "atlantisworld.png",
            "atlas mailer"             : "netscape.png",
            "atmail "                  : "atmail.png",
            "aweber"                   : "aweber.png",
            "azureus"                  : "azureus.png"
        },
        //User agents beginning with B
        "b" : {
            "b1gmail"                  : "b1gmail.png",
            "balsa"                    : "balsa.png",
            "bank of america"          : "bankofamerica.png",
            "barca"                    : "barca.png",
            "bbc"                      : "bbc.png",
            "becky"                    : "becky.png",
            "bee alerts"               : "bee_alerts.png",
            "bemail"                   : "bemail.png",
            "beos-pynr"                : "beos.png",
            "betanews"                 : "betanews.png",
            "biabam"                   : "biabam.png",
            "bitdefender vshield"      : "bitdefender.png",
            "blitzmail"                : "blitzmail.png",
            "blizzard mailer"          : "blizzard.png",
            "bloomba"                  : "bloomba.png",
            "blue shield mail server"  : "tumbleweed.png",
            "blue world lasso"         : "lasso.png",
            "bluedragon"               : "bluedragon.png",
            "bluetie mailer"           : "bluetie.png",
            "bluewin"                  : "bluewin_ch.png",
            "bofh-mail"                : "bofhmail.png",
            "bookcrossing.com mailer"  : "bookcrossing.png",
            "bre massmailer"           : "bre.png"
        },
        //User agents beginning with C
        "c" : {
            "cacert.org"               : "cacert.png",
            "calypso"                  : "calypso.png",
            "cappuccino crm"           : "cappuchino.png",
            "caramail"                 : "lycos.png",
            "cas genesisworld"         : "genesis_world.png",
            "catchme@mail"             : "catchmeatmail.png",
            "cb javamail"              : "java.png",
            "cbfmail"                  : "cbfmail.png",
            "cbizone"                  : "cbizone.png",
            "centrum mail"             : "centrum_cz.png",
            "cerberus"                 : "cerberus_helpdesk.png",
            "chatteremail"             : "chatteremail.png",
            "chilkat software"         : "chilkat.png",
            "chorus by zerocode"       : "zerocode.png",
            "cisco"                    : "cisco.png",
            "citromail"                : "citromail.png",
            "claris emailer"           : "claris_emailer.png",
            "claws mail"               : "claws-mail.png",
            "clear net webmail"        : "clearnet.png",
            "codes-sources"            : "code-source.png",
            "coldfusion mx"            : "coldfusion.png",
            "columba"                  : "columba.png",
            "communigate"              : "communigate.png",
            "compuserve"               : "compuserve.png",
            "confixx"                  : "confixx.png",
            "cookie craft server"      : "cookie_craft.png",
            "coolblue-mailer"          : "coolblue.png",
            "copland mail"             : "aol.png",
            "coppermine photo gallery" : "coppermine.png",
            "cpjnssmtpconnection"      : "cpjnsmtpconnection.png",
            "crosspoint/freexp"        : "freexp.png",
            "crosspoint/openxp"        : "openxp.png",
            "crosspoint [openxp/"      : "openxp.png",
            "csmtpconnection"          : "cpjnsmtpconnection.png",
            "ctm powermail"            : "ctm_powermail.png",
            "cvsmailer"                : "cvsmailer.png",
            "cvsspam"                  : "cvs.png",
            "cybozu"                   : "cybozu.png"
        },
        //User agents beginning with D
        "d" : {
            "dada mail"                : "dada_mail.png",
            "danaos"                   : "danaos.png",
            "danger service"           : "danger.png",
            "datenkraft newsletter system"  : "datenkraft.png",
            "datula"                   : "datula.png",
            "daum web mailer"          : "daum_webmail.png",
            "directmail/"              : "toolmaker_de.png",
            "delano e-business"        : "delano.png",
            "denshin"                  : "denshin.png",
            "desknet's"                : "desknet.png",
            "deskpro"                  : "deskpro.png",
            "devmail.net"              : "devmailnet.png",
            "direct mail for mac os x" : "direct_mail_osx.png",
            "directemail xtra"         : "directmailxtra.png",
            "direto"                   : "direto.png",
            "distributed.net"          : "distributednet.png",
            "divmod-quotient"          : "divmod.png",
            "dmc - www.dmcdeutschland.de"   : "dmc.png",
            "domaincheck mailing list" : "domaincheck.png",
            "donorware sendbulk"       : "donorware.png",
            "dotclear"                 : "dotclear.png",
            "dotnetopenmail"           : "dotnetopenmail.png",
            "dovenews"                 : "dovenews.png",
            "dreamhost webmail"        : "dreamhost.png",
            "dreammail"                : "dreammailer.png",
            "drupal"                   : "drupal.png",
            "dsa-launch"               : "dsa_launch.png",
            "dtemapper"                : "dtemapper.png",
            "dtmail"                   : "dtmail.png",
            "dumbonet systems"         : "dumbonet.png",
            "dundas mailer"            : "dundas.png",
            "dvise"                    : "dvise.png",
            "dyndns"                   : "dyndns.png"
        },
        //User agents beginning with E
        "e" : {
            "e-undertown"              : "e_undertown.png",
            "earthlink"                : "earthlink.png",
            "ebizmailer"               : "ebizmailer.png",
            "ecn communicator"         : "ecn.png",
            "edmail"                   : "edmail.png",
            "edmax"                    : "edmax.png",
            "egroups-ew"               : "yahoo.png",
            "egroups message poster"   : "egroups.png",
            "eharmony mailer"          : "eharmony.png",
            "elaine"                   : "elaine.png",
            "element5"                 : "element5.png",
            "eloqua email"             : "eloqua.png",
            "elm"                      : "elm.png",
            "email v"                  : "email.png",
            "emill/"                   : "emill.png",
            "empal"                    : "empas.png",
            "emumail"                  : "emumail.png",
            "endymion mailman"         : "endymion.png",
            "ensemble email services"  : "ensemble.png",
            "erweiterungen.de"         : "erweiterungen_de.png",
            "epoc email"               : "epoc.png",
            "epoc e-mail"              : "epoc.png",
            "esmtp (eudora internet mail server"    : "eudora2.png",
            "eudora"                   : "eudora2.png",
            "evolution"                : "evolution.png",
            "exclamationsoft"          : "exclamationsoft.png",
            "expressomail"             : "expressomail.png",
            "exmh"                     : "exmh.png",
            "eyepin campaign server"   : "eyepin.png"
        },
        //User agents beginning with F
        "f" : {
            "f5mailer"                 : "f5mailer.png",
            "farmail"                  : "farmail.png",
            "firefox"                  : "firefox.png",
            "flicks softwares ocxmail" : "ocxmail.png",
            "firepass"                 : "f5.png",
            "firstclass"               : "firstclass.png",
            "forte agent"              : "forte_agent.png",
            "forte free agent"         : "forte_agent.png",
            "forteagent"               : "forte_agent.png",
            "fotka mailer"             : "fotka.png",
            "foxmail"                  : "foxmail.png",
            "freenetmail"              : "freenetmail.png",
            "freexp"                   : "freexp.png",
            "fronter.com"              : "fronter.png",
            "fudforum"                 : "fudforum.png",
            "fusemail"                 : "fusemail.png",
            "fuzzy admail"             : "fuzzy_admail.png"
        },
        //User agents beginning with G
        "g" : {
            "g2/"                      : "gmail.png",
            "gaucho"                   : "gaucho.png",
            "geeklog"                  : "geeklog.png",
            "gemini/"                  : "gemini.png",
            "geminisoft pimmy"         : "pimmy.png",
            "genesys e-mail"           : "genesys7.png",
            "gimp useless mail program": "gimp.png",
            "git-send-email"           : "git.png",
            "gnspool"                  : "gnspool.png",
            "gnumail"                  : "gnumail.png",
            "goldmine"                 : "goldmine.png",
            "goo mail"                 : "goo.png",
            "goto software sarbacane"  : "sarbacane.png",
            "gp software directory opus"    : "gp_opus.png",
            "grono mailing system"     : "grono.png",
            "group-office"             : "group_office.png",
            "groundspeak"              : "groundspeak.png",
            "gyazmail"                 : "gyazmail.png"
        },
        //User agents beginning with H
        "h" : {
            "hamster"                  : "hamster.png",
            "hastymail"                : "hastymail.png",
            "hhde webmail"             : "hhde.png",
            "hidemarumail"             : "hidemaru.png",
            "hogwasher"                : "hogwasher.png",
            "home.ro"                  : "home_ro.png",
            "home.pl"                  : "home_pl.png",
            "horde application framework"      : "horde.png",
            "http://www.courier-mta.org/cone/" : "cone.png",
            "http://newsgroup.virgilio.it"     : "virgilio.png",
            "http://www.letterit.de"   : "letterit.png",
            "http://www.phpclasses.org": "phpclasses.png",
            "http://www.spamcop.net/"  : "spamcop.png",
            "hula modweb module"       : "hula.png"
        },
        //User agents beginning with I
        "i" : {
            "i.scribe"                 : "i_scribe.png",
            "i-soft i-tems"            : "i_tems.png",
            "icewarp web mail"         : "icewarp.png",
            "icedove"                  : "icedove.png",
            "icompass-mail"            : "icompass.png",
            "ics smtp component"       : "internet_component_suite.png",
            "ientrymailer"             : "ientrymailer.png",
            "igmail"                   : "igmail.png",
            "ilohamail"                : "iloha_mail.png",
            "imp"                      : "imp.png",
            "imp/php"                  : "imp.png",
            "inbox.lv webmail"         : "inbox.png",
            "incredimail"              : "incredimail.png",
            "inmail by insite"         : "inmailinsite.png",
            "inmail insite"            : "inmailinsite.png",
            "inscribe"                 : "inscribe.png",
            "inteligo-smtp"            : "inteligo.png",
            "internet fax, mgcs"       : "panafax.png",
            "internet mail service"    : "ms_exchange.png",
            "internet messaging program"       : "imp.png",
            "internetpost for active platform" : "internetpost.png",
            "inxmail"                  : "inxmail.png",
            "ipb php mailer"           : "ipb.png",
            "iplanet"                  : "iplanet.png",
            "irfanview"                : "irfanview.png",
            "iskoninternet webmail"    : "iskon.png"
        },
        //User agents beginning with J
        "j" : {
            "jamail3"                  : "jamail.png",
            "jamail "                  : "jamail.png",
            "jana-server"              : "jana.png",
            "javamail"                 : "java.png",
            "jawmail"                  : "jawmail.png",
            "jbmail"                   : "jbmail.png",
            "jetbrains omea"           : "omea.png",
            "jmail"                    : "jmail.png",
            "juno"                     : "juno.png",
            "just send a mail"         : "justsendamail.png",
            "jsvmail"                  : "shuriken_pro.png"
        },
        //User agents beginning with K
        "k" : {
            "kajomimail"               : "kajomi.png",
            "kana response"            : "kana_response.png",
            "kana"                     : "kana.png",
            "karen's mailer"           : "karensmailer.png",
            "kaspersky"                : "kaspersky.png",
            "katamail"                 : "katamail.png",
            "kayako esupport"          : "kayako.png",
            "kayako supportsuite"      : "kayako.png",
            "ken! 3"                   : "ken3.png",
            "kerio mailserver"         : "kerio.png",
            "kerio outlook"            : "kerio.png",
            "kmail"                    : "kmail.png",
            "knews"                    : "knews.png",
            "knode"                    : "knode.png",
            "kopiemail"                : "kopiemail.png"
        },
        //User agents beginning with L
        "l" : {
            "lasso professional"       : "lasso.png",
            "listserv"                 : "listserv.png",
            "listrak"                  : "listrak.png",
            "loadmail smtp"            : "loadmail.png",
            "locamail"                 : "locamail.png",
            "log_mail"                 : "pear.png",
            "lotus domino"             : "lotus_domino6.png",
            "lotus notes"              : "lotus_notes.png",
            "lucy"                     : "lucy.png",
            "lycosmail"                : "lycos.png",
            "lyris"                    : "lyris.png"
        },
        //User agents beginning with M
        "m" : {
            "m5mailer.com"             : "m5_mailer.png",
            "mach5 mailer"             : "m5_mailer.png",
            "mach 5 mailer"            : "m5_mailer.png",
            "macsoup"                  : "macsoup.png",
            "mahogany"                 : "mahogany.png",
            "mail user's shell"        : "mush.png",
            "mail (gnu mailutils"      : "shell-bash.png",
            "mail/news client"         : "thunderbird.png",
            "mail/news"                : "thunderbird.png",
            "mail.bg webmail"          : "mail_bg.png",
            "mail::mailer"             : "perl.png",
            "mail::send"               : "perl.png",
            "mail602"                  : "602.png",
            "mailcity service"         : "lycos.png",
            "mailenable"               : "mailenable.png",
            "mailer at germany.ru"     : "germany_ru.png",
            "maillist king"            : "maillist_king.png",
            "mailman"                  : "mailman.png",
            "mailout"                  : "mailout.png",
            "mailreader.com"           : "mailreader.png",
            "mailroom"                 : "mailroom.png",
            "mailsite express"         : "mailsite_express.png",
            "mailsmith"                : "mailsmith.png",
            "majordomo"                : "greatcircle.png",
            "macintosh eudora"         : "eudora2.png",
            "mambo open source"        : "mambo.png",
            "manitou"                  : "manitou.png",
            "mapsolute map24"          : "map24.png",
            "mark/space mail"          : "markspacemail.png",
            "marlin"                   : "marlin.png",
            "max mailer"               : "maxbulk.png",
            "measuremail"              : "measuremail.png",
            "mediawiki"                : "mediawiki.png",
            "messagingengine.com"      : "messagingengine.png",
            "mercury/32"               : "mercury.png",
            "mesnews"                  : "mesnews.png",
            "message editor"           : "messageeditor.png",
            "messenger-pro"            : "messenger_pro.png",
            "mew version"              : "mew.png",
            "microdot"                 : "microdot.png",
            "microplanet gravity"      : "microplanet.png",
            "microplanet-gravity"      : "microplanet.png",
            "microsoft cdo"            : "ms_cdo.png",
            "microsoft exchange server": "ms_exchange.png",
            "microsoft internet e-mail": "ms_patch.png",
            "microsoft internet mail"  : "ms_patch.png",
            "microsoft office outlook" : "outlook_2003.png",
            "microsoft outlook express": "outlook_express.png",
            "microsoft outlook"        : "outlook_2003.png",
            "microsoft windows mail"   : "outlook_express.png",
            "microsoft-outlook-express": "outlook_express.png",
            "microsoft-entourage"      : "ms_entourage.png",
            "microsoft windows live mail" : "ms_patch.png",//people will hate me again, but what the heck :)
            "mime-tools"               : "perl.png",
            "mime::lite"               : "perl.png",
            "minerva mail"             : "minervamail.png",
            "mintersoft"               : "mintersoft.png",
            "mirapoint"                : "mirapoint.png",
            "miva"                     : "miva.png",
            "monit"                    : "monit.png",
            "mozilla thunderbird"      : "thunderbird.png",
            "mpop web-mail"            : "mail_ru.png",
            "mr/2 internet cruiser"    : "mr2.png",
            "msn"                      : "msn.png",
            "mt-newswatcher"           : "mt_newswatcher.png",
            "mulberry"                 : "mulberry.png",
            "mutt"                     : "mutt.png",
            "mynet webmailer"          : "mynet.png",
            "my own hands[tm]"         : "cweiske.png"
        },
        //User agents beginning with N
        "n" : {
            "nail"                     : "nail.png",
            "naja/"                    : "naja.png",
            "nateon web mailer"        : "nateon.png",
            "nbb-rekenprogramma"       : "nbb-rekenprogramma.png",
            "netflix"                  : "netflix.png",
            "neomail"                  : "neomail.png",
            "neotonic trakken"         : "neotonic.png",
            "netikus.net eventsentry"  : "eventsentry.png",
            "netmail modweb module"    : "netmail.png",
            "netscape"                 : "netscape.png",
            "nettalk email"            : "nettalk.png",
            "news rover"               : "newsrover.png",
            "news xpress"              : "news_xpress.png",
            "newshound"                : "newshound.png",
            "newsleecher"              : "newsleecher.png",
            "newsletter genius"        : "newsletter_genius.png",
            "newsman pro"              : "newsman_pro.png",
            "newsportal"               : "newsportal.png",
            "newspro"                  : "newspro.png",
            "newzbin"                  : "newzbin.png",
            "nms - 1:1 mail!"          : "nms.png",
            "nms formmail"             : "nms-formmail.png",
            "nn "                      : "nn.png",
            "nn/"                      : "nn.png",
            "nocc"                     : "nocc.png",
            "noworyta"                 : "noworyta.png",
            "npop"                     : "npop.png",
            "nuralstorm webmail"       : "nuralstorm.png",
            "nyt-post"                 : "new_york_times.png"
        },
        //User agents beginning with O
        "o" : {
            "o2.pl webmail"            : "o2_pl.png",
            "o3sis o2 communication center mail"    : "o2.png",
            "oempro"                   : "oempro.png",
            "oikal-web.de"             : "oikal.png",
            "omnixmail"                : "omnixmail.png",
            "onet.poczta"              : "onet_poczta.png",
            "open webmail"             : "openwebmail.png",
            "open-xchange"             : "open_xchange.png",
            "opengroupware.org"        : "opengroupware.png",
            "openwave webengine"       : "openwave.png",
            "openwebmail"              : "openwebmail.png",
            "opera"                    : "opera.png",
            "oracle"                   : "oracle.png",
            "oscommerce"               : "oscommerce.png",
            "ostrosoft smtp"           : "ostrosoft_smtp.png",
            "otrs mail service"        : "otrs.png",
            "outlook connector for mdaemon"     : "outlook_connector_mdaemon.png",
            "outlook express"          : "outlook_express.png",
            "ovh"                      : "ovh.png"
        },
        //User agents beginning with P
        "p" : {
            "pan/"                     : "pan.png",
            "paypal"                   : "paypal.png",
            "pear"                     : "pear.png",
            "pegasus"                  : "pegasus.png",
            "penn state webmail"       : "penn_state.png",
            "pepr"                     : "pear.png",
            "perl"                     : "perl.png",
            "phlymail"                 : "phlymail.png",
            "phoenix mail"             : "phoenix_mail.png",
            "phpmailer"                : "phpmailer.png",
            "php"                      : "php.png",
            "pineapple news"           : "pineapplenews.png",
            "pine"                     : "pine.png",
            "plaasoft technologies php-mailer"      : "plaasoft.png",
            "platypus"                 : "platypus.png",
            "pluriform"                : "pluriform.png",
            "pmmail"                   : "pmmail.png",
            "pmw-tb"                   : "thunderbird.png",
            "po4ta.com"                : "po4ta_com.png",
            "pobox"                    : "pobox.png",
            "poco"                     : "pocomail.png",
            "pol-system"               : "pol_system.png",
            "pop peeper"               : "poppeeper.png",
            "poplist mailinglist"      : "poplist.png",
            "popmail/"                 : "popmail.png",
            "postaci"                  : "postaci.png",
            "postie"                   : "postie.png",
            "postme"                   : "postme.png",
            "powermail -"              : "powermail.png",
            "prayer v"                 : "hermes_cambridge.png",
            "produced by confixx webmail"   : "confixx.png",
            "produced by msn mimeole"       : "ms_patch.png",
            "produced by microsoft mimeole" : "ms_patch.png",
            "produced by microsoft exchange": "ms_exchange.png",
            "produced by phpbb2"       : "phpbb.png",
            "pronews/"                 : "pronews.png",
            "protagona email services" : "protagona.png",
            "pse3"                     : "interia.png",
            "ptmail webmail"           : "ptmail.png",
            "python mail daemon"       : "python.png",
            "python syncmail"          : "syncmail.png"
        },
        //User agents beginning with Q
        "q" : {
            "quality web email"        : "quality_web_email.png",
            "qualcomm macintosh classic eudora" : "eudora2.png",
            "qualcomm macos x eudora"           : "eudora2.png",
            "qualcomm macos classic eudora"     : "eudora2.png",
            "qualcomm windows eudora"           : "eudora2.png",
            "quickmail pro"            : "quickmail.png"
        },
        //User agents beginning with R
        "r" : {
            "reo sv"                   : "t-online.png",
            "reportbug"                : "debian.png",
            "rmrnews"                  : "rmr.png",
            "roundcube"                : "roundcube.png",
            "roving constant contact"  : "roving_constant_contact.png",
            "roxen's email"            : "roxen.png",
            "ru.board"                 : "ru_board.png"
        },
        //User agents beginning with S
        "s" : {
            "sa-smtpmail"              : "softartisans.png",
            "sap "                     : "sap.png",
            "sav materiel.net"         : "materiel.png",
            "scalix"                   : "scalix.png",
            "sendemail-"               : "sendemail.png",
            "sendmail"                 : "sendmail.png",
            "sendsmtp"                 : "sendsmtp.png",
            "serendipity/"             : "serendipity.png",
            "setia"                    : "setia.png",
            "shopexpress"              : "shopexpress.png",
            "siebel"                   : "siebel.png",
            "simorganizer"             : "simorganizer.png",
            "sitemail"                 : "sitemail.png",
            "slrn "                    : "slrn.png",
            "slrn/"                    : "slrn.png",
            "smartagent/"              : "smartagent.png",
            "smarttools publishing serienmailer"    : "smarttools.png",
            "smtpc"                    : "smtpc.png",
            "smtpit - filemaker pro"   : "filemaker.png",
            "snappermail"              : "snappermail.png",
            "speedmail"                : "speedmail.png",
            "squirrelmail"             : "squirrelmail.png",
            "staroffice21"             : "staroffice21.png",
            "staroffice"               : "staroffice.png",
            "steganos"                 : "steganos.png",
            "steinbrecher software"    : "steinbrecher.png",
            "store-mailer/"            : "apple_store.png",
            "strato"                   : "strato.png",
            "streamserve"              : "streamserve.png",
            "stringdata"               : "post_sk.png",
            "strongmail"               : "strongmail.png",
            "stumbleupon"              : "stumbleupon.png",
            "su-webmail"               : "stockholm_uni.png",
            "sumo/"                    : "sumo.png",
            "sun java"                 : "java.png",
            "suse linux openexchange server" : "suse.png",
            "svn::notify"              : "subversion.png",
            "svnmailer"                : "subversion.png",
            "swwwing"                  : "swwwing.png",
            "sylpheed"                 : "sylpheed.png",
            "symbian os"               : "symbian_os.png",
            "sysaid server"            : "sysaid.png"
        },
        //User agents beginning with T
        "t" : {
            "t-online"                 : "t-online.png",
            "talisma mail"             : "talisma.png",
            "tapirmail"                : "tapir_mail.png",
            "tdc online webmail"       : "tdc.png",
            "teamware"                 : "teamware.png",
            "telix"                    : "telix.png",
            "testtrack smtp"           : "testtrack.png",
            "the bat!"                 : "the_bat.png",
            "tin/"                     : "tin.png",
            "toi"                      : "t-online.png",
            "toshiba network camera mailer" : "toshiba_network_camera2.png",
            "touchpoint campaign manager"   : "touchpoint.png",
            "trac"                     : "trac.png",
            "tricorn autobahn"         : "autobahn.png",
            "trn "                     : "trn.png",
            "tsworks e-mail"           : "tsworks.png",
            "turnpike"                 : "turnpike.png",
            "turukame"                 : "turukame.png",
            "tva e-mail transfer client"    : "tva.png",
            "tvnmail"                  : "tvnmail.png",
            "twig"                     : "twig.png",
            "typo3"                    : "typo3.png"
        },
        //User agents beginning with U
        "u" : {
            "ubb.threads"              : "ubbthreads.png",
            "uebimiau"                 : "uebimail.png",
            "ultrafunk popcorn"        : "popcorn.png",
            "umailng"                  : "umailng.png",
            "umo mail system"          : "mozilla_update.png",
            "unison/"                  : "unison.png",
            "unisys"                   : "unisys.png",
            "unitymail"                : "unity_mail.png",
            "upcdata.info"             : "upcdata.png",
            "usanet web-mailer"        : "usanet.png",
            "userland frontier"        : "userland_frontier.png",
            "usermin"                  : "usermin.png",
            "utstarcom"                : "vonage.png"
        },
        //User agents beginning with V
        "v" : {
            "v-webmail"                : "v-webmail.png",
            "vacation/"                : "vacation.png",
            "vbulletin"                : "vbulletin.png",
            "veritas smtp"             : "veritas.png",
            "versamail"                : "versamail.png",
            "virtual access"           : "virtual_access.png",
            "visualsoft mailer"        : "visualsoft.png",
            "volny.cz webmail2"        : "volny_cz.png",
            "voltimum newsletter-mailer"    : "voltimum.png"
        },
        //User agents beginning with W
        "w" : {
            "wamailer"                 : "wamailer.png",
            "wanderlust"               : "wanderlust.png",
            "waste.uk.com mailsystem"  : "waste.uk.com.png",
            "web-based email"          : "godaddy.png",
            "web based pronto"         : "web_based_pronto.png",
            "web de mail"              : "web_de_mail.png",
            "webmail version"          : "webmail.png",
            "webmail di libero"        : "libero.png",
            "webmail ulb"              : "ulb.png",
            "webmasterworld"           : "webmasterworld.png",
            "webmin"                   : "webmin.png",
            "wemail32"                 : "wemail.png",
            "west wind"                : "westwind.png",
            "whatcounts"               : "whatcounts.png",
            "winbiff"                  : "winbiff.png",
            "windmail"                 : "windmail.png",
            "windows eudora"           : "eudora2.png",
            "winnt's msm"              : "msm.png",
            "winvn"                    : "winvn.png",
            "wordperfect mail"         : "wordperfect.png",
            "worldclient"              : "worldclient.png",
            "writely"                  : "writely.png",
            "www.schworak.com"         : "schworak.png"
        },
        //User agents beginning with X
        "x" : {
            "xananews"                 : "xananews.png",
            "xanga.com"                : "xanga.png",
            "xe.com"                   : "xe_com.png",
            "xfmail"                   : "xfmail.png",
            "ximian evolution"         : "evolution.png",
            "xnews"                    : "xnews.png",
            "xp framework"             : "xpframework.png",
            "xpertmailer"              : "xpertmailer.png",
            "xpn/"                     : "xpn.png",
            "xtreeme"                  : "xtreeme_mailer.png"
        },
        //User agents beginning with Y
        "y" : {
            "y.a.m.c."                 : "yamc.png",
            "ya-newswatcher"           : "ya-newswatcher.png",
            "yam "                     : "yam.png",
            "yamail"                   : "yandex.png",
            "yahoo groups message poster" : "yahoo.png",
            "yahoomailrc"              : "yahoo.png"
        },
        //User agents beginning with Z
        "z" : {
            "zoznam mailer"            : "zoznam.png",
            "zuckmail"                 : "zuckmail.png"
        },
        //User agents not beginning with one of the above letters
        //This here have to be anywhere in the identification line
        "postsearch" : {
            "(c)agora mailer"          : "agora.png",
            "30gigs.com webmail"       : "30gigs.png",
            "40tude_dialog"            : "40tude_dialog.png",
            "602lan suite"             : "lansuite.png",
            "724mailer"                : "724mailer.png",
            ".0 for windows sub "      : "aol.png",
            ".0 for windows"           : "aol.png",
            ".0 se for windows sub "   : "aol.png",
            ".0 security edition for windows" : "aol.png",
            ".apache.org)"             : "apache.png",
            ".comfire-mail-system"     : "suse.png",
            ".islandnet.com"           : "islandnet.png",
            "6.0 sub "                 : "aol.png",
            "blat"                     : "blat.png",
            "by gecip sas"             : "gecip.png",
            "cisco"                    : "cisco.png",
            "communigate"              : "communigate.png",
            "egroupware"               : "egroupware.png",
            "emacs"                    : "emacs.png",
            "entourage"                : "ms_entourage.png",
            "freebit.com"              : "freebit.png",
            "global message exchange"  : "gmx.png",
            "gmane.org"                : "gmane.png",
            "gossamer"                 : "gossamer.png",
            "groupwise"                : "groupwise.png",
            "imail"                    : "imail.png",
            "libero.it"                : "libero.png",
            "m5mailer.com"             : "m5_mailer.png",
            "nifty webmail"            : "nifty.png",
            "pochta.ru"                : "pocta_ru.png",
            "poczty wirtualnej polski" : "poczta_wp_pl.png",
            "polarbar"                 : "polarbar.png",
            "registered ak-mail"       : "ak_mail.png",
            "rosecitysoftware"         : "courier_rose.png",
            "techtarget.com"           : "targetmail_techtarget.png",
            "webmail for roxen"        : "roxen.png",
            "www.xp2.de"               : "xp2.png",
            "<smtp32"                  : "imail.png"
        }
    },
    // }}}



    /**
    *	Does nothing.
    *	Only for Thunderbird < 1.1, it sets a timeout because
    *	we don't get notified about the endHeaders event.
    */
    noop: function()
    {
        if (dispMuaGlobals.bTimeoutWorkaround) {
            //try to display the icon half a second later
            window.setTimeout(dispMuaGlobals.dispMUA, 500);
        }
    },//noop: function()



    /**
    *	Tries to find the user agent in the header and displays it.
    */
    dispMUA: function()
    {
        var srcMsgURI = GetLoadedMessage();
        if (srcMsgURI == null) {
            //no message loaded
            return;
        }

        dispMuaGlobals.bLastUnknown = false;

        var strUserAgent	= "";
        var strOrganization	= "";
        var strMimeOLE		= "";
        var strMessageId	= "";
        var strExtra        = "";
        //loop through the headers
        var nHeader = 0;

        for (headerName in currentHeaderData) {
            nHeader++;
//			dump(headerName + "\r\n");
            if (headerName == "user-agent" || headerName == "x-mailer" || headerName == "x-newsreader") {
                strUserAgent = currentHeaderData[headerName].headerValue;
            } else if (headerName == "organization") {
                strOrganization = currentHeaderData[headerName].headerValue;
            } else if (headerName == "x-mimeole") {
                strMimeOLE = currentHeaderData[headerName].headerValue;
            } else if (headerName == "message-id") {
                strMessageId = currentHeaderData[headerName].headerValue;
            } else if (headerName == "x-bugzilla-reason") {
                strExtra = "bugzilla";
            } else if (headerName == "x-php-bug") {
                strExtra = "phpbug";
            }
        }
        //if it's below 6, the mail is a) very unusual or b) enigmail resp. mnenhy isn't installed
//        dump(nHeader + " header lines\n");
        var strUserAgentLower = strUserAgent.toLowerCase();
        var bOverlayIcon = false;

        //find an icon for the header
        var strDispMuaIcon = "empty.png";
        if (strUserAgentLower != "") {

            strDispMuaIcon = "unknown.png";
            dispMuaGlobals.bLastUnknown = true;
            var bFound = false;

            //user overlay array
            for (strDispMuaKey in dispMuaGlobals.arDispMuaOverlay) {
                if (strUserAgentLower.indexOf(strDispMuaKey) > -1) {
                    strDispMuaIcon = dispMuaGlobals.arDispMuaOverlay[strDispMuaKey];
                    bOverlayIcon = true;
                    bFound = true;
                    //that the user knows he made the crap
                    strUserAgent = strUserAgent + "\r\nUser override icon\r\n Key: " + strDispMuaKey + "\r\n Icon: " + strDispMuaIcon;
                    break;
                }
            }

            // {{{
            //in-built data
            if (!bFound) {
                //presearch
                for (strDispMuaKey in dispMuaGlobals.arDispMuaAllocation["presearch"]) {
                    //does it contain the key somewhere?
                    if (strUserAgentLower.indexOf(strDispMuaKey) > -1) {
                        bFound = true;
                        strDispMuaIcon = dispMuaGlobals.arDispMuaAllocation["presearch"][strDispMuaKey];
                        break;
                    }
                }
            }

            if (!bFound) {
                //Letter it begins with
                var chLetter = strUserAgentLower.substr(0, 1);
                if (dispMuaGlobals.arDispMuaAllocation[chLetter]) {
                    for (strDispMuaKey in dispMuaGlobals.arDispMuaAllocation[chLetter]) {
                        //check if the identification really begins with the key
                        if (strUserAgentLower.substr(0, strDispMuaKey.length) == strDispMuaKey) {
                            bFound = true;
                            strDispMuaIcon = dispMuaGlobals.arDispMuaAllocation[chLetter][strDispMuaKey];
                            break;
                        }
                    }
                }
            }

            if (!bFound) {
                //the rest
                for (strDispMuaKey in dispMuaGlobals.arDispMuaAllocation["postsearch"]) {
                    //does it contain the key somewhere?
                    if (strUserAgentLower.indexOf(strDispMuaKey) > -1) {
                        bFound = true;
                        strDispMuaIcon = dispMuaGlobals.arDispMuaAllocation["postsearch"][strDispMuaKey];
                        break;
                    }
                }
            }
            // }}}

            if (bFound) {
                dispMuaGlobals.bLastUnknown = false;
            }

        } else if (strExtra != "") {
            if (strExtra == "bugzilla") {
                strDispMuaIcon = "bugzilla.png";
                strUserAgent = "X-Bugzilla-Reason";
            } else if (strExtra == "phpbug") {
                strDispMuaIcon = "bug.png";
                strUserAgent = "X-PHP-Bug";
            }
        } else if (strOrganization != "") {
            var strOrganizationLower = strOrganization.toLowerCase();
            if (strOrganizationLower.indexOf("groups.google") > -1) {
                strDispMuaIcon = "google.png";
            } else if (strOrganizationLower.indexOf("freemail.web.de") > -1
                    || strOrganizationLower.indexOf("http://web.de") > -1) {
                strDispMuaIcon = "web_de.png";
            }
            strUserAgent = "Organization: " + strOrganization;

        } else if (strMimeOLE != "") {
            var strMimeOLELower = strMimeOLE.toLowerCase();
            if (strMimeOLELower.indexOf("exchange") > -1) {
                strDispMuaIcon = "ms_exchange.png";
            } else if (strMimeOLELower.indexOf("mimeole") > -1) {
                strDispMuaIcon = "outlook_2003.png";
            }
            strUserAgent = "X-MimeOLE: " + strMimeOLE;

        } else if (strMessageId != "") {
            if (strMessageId.indexOf("Pine.") > -1) {
                strDispMuaIcon = "pine.png";
                strUserAgent = "Message-ID: " + strMessageId;
            } else if (strMessageId.indexOf("mail.gmail.com") > -1
                    || strMessageId.indexOf("@gmail.com") > -1
            ) {
                strDispMuaIcon = "gmail.png";
                strUserAgent = document.getElementById('messenger-dispmua-strings').getString("dispMUA.Gmail") +"\r\nMessage-ID: " + strMessageId;
            } else if (strMessageId.indexOf("@persist.google.com>") > -1) {
                strDispMuaIcon = "gmail.png";
                strUserAgent = "Message-ID: " + strMessageId;
            } else if (strMessageId.indexOf(".yahoo.com") > -1) {
                strDispMuaIcon = "yahoo.png";
                strUserAgent = document.getElementById('messenger-dispmua-strings').getString("dispMUA.Yahoo") + "\r\nMessage-ID: " + strMessageId;
            } else if (strMessageId.indexOf("@phx.gbl") > -1) {
                strDispMuaIcon = "msn.png";
                strUserAgent = document.getElementById('messenger-dispmua-strings').getString("dispMUA.Hotmail") + "\r\nMessage-ID: " + strMessageId;
            } else if (strMessageId.indexOf("@cvsserver>") > -1) {
                strDispMuaIcon = "cvs.png";
                strUserAgent = "Message-ID: " + strMessageId;
            }
        }


        //icon name to full name
        if (bOverlayIcon) {
            //an overlay icon already has the full path in it, including the protocol
            //strDispMuaIcon = "file://" + strDispMuaIcon;
        } else {
            strDispMuaIcon = "chrome://messenger-dispmua/content/48x48/" + strDispMuaIcon;
        }

        //status bar
    //	document.getElementById('dispMUAstatus').label = strUserAgent;
    //	document.getElementById('dispMUAstatusIcon').src = strDispMuaIcon;

        //message header icon
        document.getElementById('dispMUAicon').src = strDispMuaIcon;
        var strTooltip = strUserAgent;
        var nReturnPos = strTooltip.indexOf("\r");
        if (nReturnPos != -1) {
            strTooltip = strUserAgent.substr(0,nReturnPos);
        }
        document.getElementById('dispMUAicon').setAttribute("tooltiptext",strTooltip);

        dispMuaGlobals.strDispMUAstring = strUserAgent;
    },//dispMUA: function()



    /**
    *   loads the user agent overlay file
    *   in which users can define their own icons
    *
    *   The overlay file has a semi-csv format.
    *   - On every line, there have to be two strings, split by a comma ","
    *   - The first string is the *lowercase* search string which shall match the user agent
    *   - The second is the absolute path to the icon
    *   If the search string shall include a comma itself, you can quote it.
    *    So >"money,inc",/data/icons/money.png< would match the user agent
    *    >Mail by Money,Inc. at Cayman Islands< but not >Moneymaker mailer<
    *   There is no check for a third (or higher) column, so everything
    *    behind the comma is used as the filename.
    *   The filename may be quoted as well.
    */
    loadMuaOverlayFile: function()
    {
//        dump("loadMuaOverlayFile\r\n");
        // open an input stream from file
        var istream;
        try {
            var file = Components.classes["@mozilla.org/file/directory_service;1"].
                getService(Components.interfaces.nsIProperties).
                get("ProfD", Components.interfaces.nsIFile);
            file.append(dispMuaGlobals.strOverlayFilename);
            istream = Components.classes["@mozilla.org/network/file-input-stream;1"]
                            .createInstance(Components.interfaces.nsIFileInputStream);
            istream.init(file, 0x01, 0444, 0);
            istream.QueryInterface(Components.interfaces.nsILineInputStream);
        } catch (e) {
//            dump("dispMUA: no overlay file " + strFilename + "\r\n");
            return;
        }

        // read lines into array
        var line = {}, hasmore;
        var strLine, nEndQuote, nCommaPos;
        var strKey, strValue;
        do {
            hasmore = istream.readLine(line);

            strLine = line.value;
            if (strLine.substr(0,1) == "#") {
                //comment
                continue;
            }

            if (strLine.substr(0,1) == "\"") {
                //with quotes
                //find end quote
                nEndQuote = strLine.indexOf("\"", 2);
                if (nEndQuote == -1) {
                    //no endquote? Bad line!
                    continue;
                }
                nCommaPos = strLine.indexOf(",",nEndQuote);
            } else {
                //no quote
                nCommaPos = strLine.indexOf(",");
            }

            if (nCommaPos == -1) {
                //no comma? Bad line!
                continue;
            }

            strKey   = dispMuaGlobals.stripSurroundingQuotes(strLine.substr(0,nCommaPos));
            strValue = dispMuaGlobals.stripSurroundingQuotes(strLine.substr(nCommaPos + 1));
            dispMuaGlobals.arDispMuaOverlay[strKey] = strValue;
//            dump("key: >" + strKey + "< icon: >" + strValue + "<\r\n");
        } while(hasmore);

        istream.close();

//        dump("loadMuaOverlayFile end\r\n");
    },//loadMuaOverlayFile: function()



    /**
    *   Removes quotes (one leading and one trailing) from the given string
    *   Used by loadMuaOverlayFile for quoted csv field values
    */
    stripSurroundingQuotes: function(string)
    {
        if (string.substr(0,1) == "\"" && string.substr(string.length - 1) == "\"") {
            string = string.substr(1);
            string = string.substr(0, string.length - 1);
        }

        return string;
    },//stripSurroundingQuotes: function(string)



    /**
    *	Displays a popup window with the current user agent string.
    */
    dispMUAwindow: function()
    {
        if (dispMuaGlobals.strDispMUAstring == "") {
            alert(
                document.getElementById('messenger-dispmua-strings').getString("dispMUA.NoUserAgent")
                + "\r\n\r\n" + document.getElementById('messenger-dispmua-strings').getString("dispMUA.sendpostcard")
            );
        } else {
            if (dispMuaGlobals.bLastUnknown == true) {
                alert(
                    "If you want this agent supported, make an icon for it\r\n"
                    + "(PNG, 48x48, transparent background, good quality)\r\n"
                    + "and sent it together with the following string to me:\r\n\r\n"
                    + dispMuaGlobals.strDispMUAstring
                );
                //unknown. Open a new mail window to cweiske@cweiske.de
//                var messenger = Components.classes["@mozilla.org/messenger;1"].createInstance(Components.interfaces.nsIMessenger);
//                messenger.SetWindow(window,null);
//                messenger.OpenURL("mailto:cweiske@cweiske.de?subject=dispMUA:%20new%20header&body=" + dispMuaGlobals.strDispMUAstring);
            } else {
                alert(
                    dispMuaGlobals.strDispMUAstring
                    + "\r\n\r\n" + document.getElementById('messenger-dispmua-strings').getString("dispMUA.sendpostcard")
                );
            }
        }
    },//dispMUAwindow: function()



    /**
    *	Sets up the event listener for dispMUA.
    *	We do that to get notified when a message
    *	has been selected
    */
    setupEventListener: function()
    {
        var listener = {};
        listener.onStartHeaders	= dispMuaGlobals.noop;
        listener.onEndHeaders	= dispMuaGlobals.dispMUA;
        gMessageListeners.push(listener);

//        dump("dispMUA: setupEventListener\r\n");

        //version check to check if we need the timeout workaround
        try {
            var strVersion = Components.classes['@mozilla.org/xre/app-info;1'].getService(Components.interfaces.nsIXULAppInfo).version;
            dump("dispMUA: version " + strVersion + "\r\n");
            if (strVersion == "1.0+" || Number(strVersion) >= 1.1) {
                dispMuaGlobals.bTimeoutWorkaround = false;
            } else {
                //1.0 or older -> workaround needed
                dispMuaGlobals.bTimeoutWorkaround = true;
                dump("dispMUA: Activating timeout workaround\r\n");
            }
        } catch (e) {
            //probably old version
            dispMuaGlobals.bTimeoutWorkaround = true;
            dump("dispMUA: Activating timeout workaround (no version found)\r\n");
        }

        dispMuaGlobals.loadMuaOverlayFile();
	}//setupEventListener: function()

}//var dispMuaGlobals

addEventListener('messagepane-loaded', dispMuaGlobals.setupEventListener, true);
