{-# OPTIONS -fth -cpp -fglasgow-exts -fallow-overlapping-instances -fallow-undecidable-instances #-}

{--
The comments in this file should provide a reasonable understanding of how to use HAppS for
HTTP applications.  If you just want an eample or a simple server, search for the word "Example Server"
in this file.  THere is an example provided at the bottom.
--}

module HAppS.Protocols.SimpleHTTP2 
    (-- * Server parts
     ServerPart(Handle, ModResp),request,respond,runServerParts,pass
     -- * Running the server
    ,stdHTTP,simpleHTTP,mainHTTP
    -- * FIXME
    ,multi,hOut,fromReqURI,h,moved,FromReqURI,val,hs  -- ,basicFileServe
    ,basicAuth,multiIf,re,Prefix(..)
     -- * Server parts tied to Ev
    ,serverXSLT,debugFilter
     -- * Output FIXME
    ,respIO
    ,notFound,notImplemented,seeOther,ok,forbidden,unauthorized
     -- * Default values
    ,clientPath,staticPath,stylePath
     -- * Ev Utilities [FIXME move these]
    ,getHost, getScheme,getLocalURI,getPath,getQuery
     -- * Styles
     ,style_xml,plain_xml,xml
     -- * Misc
    ,Index(..), noState
    ,pathRewrite,dirIndex
) where

import Data.List
import Data.Map (Map)
import qualified Data.Map as M (lookup)
import qualified Data.ByteString.Char8 as P (unpack, drop)
import Data.Maybe
import Control.Monad.Error
import Text.Regex
import HAppS.Protocols.HTTP.Types
import HAppS.Protocols.HTTP.Listen(listen) 
import HAppS.MACID -- (SyncH,getEvent)
import qualified HAppS.Protocols.Base64 as Base64 (decode)
import HAppS.Protocols.MessageWrap
import HAppS.Protocols.MinHaXML -- clean up 
import HAppS.Protocols.HTTP.FileServe
import HAppS.Protocols.SURI
import HAppS.Protocols.XSLT
--for example code 
import Control.Monad.State
-- if it was another file then just import HAppS and you get all the below
import HAppS.Util.StdMain
import HAppS.Util.StdMain.Config
import HAppS.Util.StdMain.StartState
import HAppS.Util.Common
import System.Log.Logger

-- | We model the server as a set of filters that either transform the request or create a response.
-- and then a set of filters that may transform the response.  Notice that the type of rsp allows
-- use to create complex server part trees.
data ServerPart m req m' res
    = Handle (req -> m (Either req (m' res)))
      -- ^ Handle a request when there is no response and return either a new request or a response.
    | ModResp (m (res -> m' res))
      -- ^ Modify the response on the way out.
    | Multi [ServerPart m req m' res]
      -- ^ Handle a list of ServerParts.

-- Handle functions always produce with either a request or a response

-- | Turn value into a response and return it.
respond :: Monad m => response -> m (Either request response)
respond = return . Right
-- | Turn value into a new request and return it.
request :: Monad m => request -> m (Either request response)
request = return . Left

-- | Run the server.
--   Note that an error handler should be usually used with runServerParts.
runServerParts :: (Monad m, Monad m') => [ServerPart m request m' response] -> request -> m (Either request (m' response))
runServerParts (Handle h  : ss) req = either (runServerParts ss) respond =<< h req
runServerParts (ModResp h : ss) req = either request (\r -> h >>= \h' -> respond (h' =<< r)) =<< runServerParts ss req
runServerParts (Multi pp  : ss) req = runServerParts (pp ++ ss) req
runServerParts []               req = request req



--useful example handler function that allows you to watch all requests and responses on console
-- debugFilter::(Monad m,Show req,Show result) => ServerPart m req IO result
-- debugFilter = multi [Handle (\req -> return $ (logM "HAppS.Protocols.SimpleHTTP2" INFO $ show req) >> request req)
--                     ,ModResp (return (\res -> logM "HAppS.Protocols.SimpleHTTP2" INFO (show res) >> return res))]

-- debugFilter2::(Monad m,Show req) => String -> ServerPart m req IO Result
-- debugFilter2 s = multi [Handle (\req -> (logM "HAppS.Protocols.SimpleHTTP2" INFO $ s ++ (show req)) >> request req)
--                        ,ModResp (return (\res -> logM "HAppS.Protocols.SimpleHTTP2" INFO (s++show res) >> return res))]


debugFilter ::(Monad m,Show req,Show result) => ServerPart m req IO result
debugFilter = multi [ModResp (return (\res -> logM "HAppS.Protocols.SimpleHTTP2" INFO (show res) >> return res))]


-- FIXME - Ev is not MonadIO
--debugFilterMIOEx :: MonadIO m => (req -> Maybe res -> m ()) -> ServerPart m req res
--debugFilterMIOEx mf = Modify $ \rq mrs -> mf rq mrs >> return (rq,mrs)


-- !!! need to modify this so ModResp has access to the modified request!
instance Error Request   -- because we are using Either as a monad

multi :: [ServerPart m req m' res] -> ServerPart m req m' res
multi = Multi
--multi parts = Handle $ rsp parts -- notice that you can do recursive stuff or multihosting etc.



-- | Transform ServerParts into a component that can be used with the stdMain framework.
-- 
-- This code runs the server inside the MACID monad and connects it with the code that is actually
-- talking HTTP on the socket.  Skip to next block unless you are concerned about overall HAppS stuff.
-- Contains an exception handler.
--
-- This can be used like:
--
-- > main = stdMain (simpleHTTP impl :*: foo :*: bar :*: End)
-- >   where impl = [ debugFilter,
-- >                , hs clientPath         GET  $ basicFileServe staticPath
-- >                , noState
-- >                ]
simpleHTTP :: [ServerPart (Ev st Request) Request IO Result] -> Conf -> Handler st
simpleHTTP ss conf = SyncH $ return (work,listen conf)
    where
    work etf = etf $ liftM (either (error "Expected response") id) . runServerParts ss' =<< getEvent
    ss' = ss ++ [     --3. default behavior
                 h ()          [GET,POST] $ notFound $ val "URL Not Found or Bad Body" 
                ,h ()          ()  $ notImplemented $ val "Method Not Implemented"
                ]

instance ConfHandler Conf where
    confUsage _ = copt ho
    confHandler fun = liftM fun (getOptM ho >>= foldM (\x y -> y x) nullConf)

ho :: [OptDescr (Conf -> OptM Conf)]
ho = [Option [] ["http-port"] (ReqArg (\h c -> do x <- readM h; return c { port = x }) "port") "port to bind http server"]



{--
Wrappers around Handle and ModResp that take care of matching URI and Method and converting
the body to the type you actually want.
--}
h :: (Monad m, Monad m', FromMessage req_msg, FromReqURI uri_match uri_msg, MatchMethod req_match)
  => uri_match -> req_match
  -> (uri_msg -> req_msg -> m (Either Request (m' res)))
  -> ServerPart m Request m' res
h uri meth handler = Handle (\req -> maybe (request req) (uncurry handler) $ hImpl uri meth req)

-- | Trivial handler for returning values
val :: Monad m => response -> () -> () -> m (Either request response)
val v () () = respond v 

hOut :: (FromMessage msg, FromReqURI urimatcher urimsg, MatchMethod matcher, Monad m)
     => urimatcher -> matcher
     -> (urimsg -> msg -> Ev st Request (res -> m res))
     -> ServerPart (Ev st Request) req m res
hOut uri meth handler
     = ModResp $ maybe (return return) (uncurry handler) . hImpl uri meth =<< getEvent

hs :: path -> method -> (path -> method -> res) -> res
hs pathExp meth handler = handler pathExp meth 


hImpl :: (FromMessage msg, FromReqURI urimatcher urimsg, MatchMethod matcher)
      => urimatcher -> matcher -> Request -> Maybe (urimsg, msg)
hImpl pathExp meth req = do
  boolM . matchMethod meth . rqMethod $ req -- if we match the method GET, POST
  reqURI <- fromReqURI pathExp (rqURI req) -- if requri matches "/s/" for example
  msg <- fromMessageM req -- if we can parse the incoming message...
  return (reqURI,msg) -- then we pass it all to the handler

-- MatchMethod and FromReqURI provide a lot of simplicity with power
class MatchMethod m where matchMethod :: m -> Method -> Bool
instance MatchMethod Method where matchMethod method = (== method) 
instance MatchMethod [Method] where matchMethod methods = (`elem` methods)
instance MatchMethod (Method -> Bool) where matchMethod f = f 
instance MatchMethod () where matchMethod () _ = True

newtype Prefix a = Prefix a

class FromReqURI expr res where
    fromReqURI :: expr -> SURI -> Maybe res

instance FromReqURI [String] () where
    fromReqURI expr uri = guard . null =<< prefixMatch expr . splitPath . path =<< return uri

instance FromReqURI [Regex] [String] where
    fromReqURI expr uri = (prefixRegex (const mzero) expr . splitPath . path) uri

instance FromReqURI [Regex] () where
    fromReqURI expr uri = (prefixRegex (const mzero) expr . splitPath . path) uri >> return ()

instance FromReqURI (Prefix [String]) () where
    fromReqURI (Prefix expr) uri = (prefixMatch expr . splitPath . path) uri >> return ()

instance FromReqURI (Prefix [String]) [String] where
    fromReqURI (Prefix expr) uri = (prefixMatch expr . splitPath . path) uri

instance FromReqURI (Prefix [Regex]) () where
    fromReqURI (Prefix expr) uri = (prefixRegex return expr . splitPath . path) uri >> return ()

instance FromReqURI (Prefix [Regex]) [String] where
    fromReqURI (Prefix expr) uri = (prefixRegex return expr . splitPath . path) uri

instance FromReqURI (Prefix ()) [String] where
    fromReqURI x uri = fromReqURI (Prefix ([]::[String])) uri

instance FromReqURI () () where
    fromReqURI _ _ = return ()

instance FromReqURI (SURI -> Maybe m) m  where
    fromReqURI expr uri = expr uri


splitPath         :: String -> [String]
splitPath ('/':x) = a : splitPath b where (a,b) = break (=='/') x
splitPath []      = []
splitPath _       = error "splitPath: malformed path"

prefixRegex                 :: ([String] -> Maybe [String]) -> [Regex] -> [String] -> Maybe [String]
prefixRegex _ []     []     = return []
prefixRegex f []     ys     = f ys
prefixRegex f (x:xs) (y:ys) = liftM2 (++) (matchRegex x y) (prefixRegex f xs ys)
prefixRegex _ _      _      = fail "prefixRegex: no match"

prefixMatch       :: [String] -> [String] -> Maybe [String]
prefixMatch [] ys =  return ys
prefixMatch (x:xs) (y:ys)
 | x == y         = prefixMatch xs ys
 | otherwise      = fail "prefixMatch: no match"
prefixMatch _ _   = fail "prefixMatch: no match"

re :: [String] -> [Regex]
re = map (\ regex -> mkRegexWithOpts ("^" ++ regex ++ "$") True True)

-- Ev utility functions that probably belong elsewhere

-- | Get the Host header from the request or fail if it doesn't exist.
getHost :: Ev st Request String
getHost = getHeader "Host" =<< getEvent
-- | Get the request scheme defaulting to \"http:\/\" from the \"x-scheme\" header.
getScheme :: Ev st Request String
getScheme = liftM (fromMaybe "http:/" . getHeader "x-scheme") getEvent
-- | Get the path elements from the request URI.
getPath :: Ev st Request [String]
getPath = liftM (pathEls . path . rqURI) getEvent
-- | Get the query part of the request URI.
getQuery :: Ev st Request String
getQuery = liftM (query . rqURI) getEvent
-- | Construct an URI with the given path components and host+scheme from the current request.
getLocalURI :: [String] -> Ev st Request SURI
getLocalURI p = do
  s <- getScheme
  h <- getHost
  return $ toSURI $ concat $ intersperse "/" (s:h:p)

pass :: Ev st Request (Either Request res)
pass = request =<< getEvent

--responding with particular HTTP status codes

respIO f p r = respond $ f p r
--note:  !!! why doesn't (respond . f) work?

ok, notFound, forbidden, notImplemented, unauthorized
               :: (Monad im, Monad m, ToMessage res)
               => (a -> b -> m (Either Request res))
               -> a -> b -> m (Either Request (im Result))
ok             = resp 200
notFound       = resp 404
forbidden      = resp 403
notImplemented = resp 501
unauthorized   = resp 401
--serverError :: OutHttpResult m im a b res
--serverError s = resp 500 s

-- FIXME
seeOther, moved
         :: (Monad m, Monad im, ToMessage t, ToSURI s)
         => (a -> b -> m (Either Request (s,t)))
         -> (a -> b -> m (Either Request (im Result)))
seeOther = redirectResp 303
moved    = redirectResp 301

resp                  :: (Monad m', Monad (Either req), Monad m, ToMessage t)
                      => Int
                      -> (a -> b -> m (Either req t))
                      -> a -> b -> m (Either req (m' Result))
resp code handler x y = liftM (liftM (liftM setCode . toMessageM)) $ handler x y
    where setCode res = res {rsCode = code}

redirectResp code handler x y = liftM (liftM f) $ handler x y
  where f (uri,body) = liftM (setCode . addHeader "Location" (render (toSURI uri))) (toMessageM body)
        setCode res = res {rsCode = code}
-- response content type handling -- we should add JSON rendering as well!

-- STYLES

type Styler monad from = from -> monad Result

-- | Convert to XML and set the stylesheet.
style_xml :: (ToElement elt, Monad m) => StyleSheet -> Styler m elt
style_xml style obj = 
  liftM (addHeader "Content-Type" "text/xml") . toMessageM . simpleDoc style . toElement $ obj
      -- !! modify this to produce an absolute style URL so we can avoid static in serverXSLT

-- | Serve XML with the default stylesheet from 'styleURI'.
xml :: (Monad m, ToElement elt) => Styler m elt
xml o = style_xml (XSL styleURI) o
instance (ToElement x)=>ToMessage x where toMessageM = xml

-- | Serve XML without a stylesheet
plain_xml :: (Monad m, ToElement elt) => Styler m elt
plain_xml o = style_xml NoStyle o

--- don't know where to put these
dirIndex::String->String->String
dirIndex s path = 
    if (head $ reverse path) /= '/' then path else path ++ s

pathRewrite f () req = request req  {rqURI = u_path f $ rqURI req}

---file serving example handler  !!needs some cleanup
--basicFileServe :: (Monad m, MatchMethod matcher, FromReqURI urimatcher [String])
--               => String -> urimatcher -> matcher -> ServerPart m Request IO Result
{--basicFileServe staticPath path meth = multi
    [h path       meth   . forbidden $ blockDotFiles3 "Dot files not alloed" . head
    ,h path       meth   $ respond . fileServe2 mimeTypes staticPath]

    where blockDotFiles3 msg pathInfo req
            | isDot pathInfo = respond msg
            | otherwise      = request req
--}
basicFileServe path = respIO $ fileServe path


-- | example server xslt response filter  -- notice use of IO in producing the response!!!
serverXSLT :: XSLPath -> () -> Request -> Ev st Request (Result -> IO Result)
serverXSLT xslPath () req = return $ \res ->
    let regex     = mkRegexWithOpts ".*(opera|khtml|lynx).*" True False
        cxml      = Just "text/xml"
        doXslt    = do new <- xsltFPSIO xslPath $ rsBody res
                       return . setHeader "Content-Type" "text/html" $ res { rsBody = new }
    in if getHeader "Content-Type" res /= cxml
         then return res
         else case getHeader "x-happs-style" req of
                Just "false" -> return res
                Just "true"  -> doXslt
                _ -> maybe (return res) (const doXslt) $ matchRegex regex =<< getHeader "User-Agent" req

---Should really just standardize on these unless you have reason not to!
styleName, clientPath, styleURI, staticPath, stylePath :: String
styleName = "style.xsl"
clientPath = "/s/"
styleURI = (clientPath ++ styleName)
staticPath = "static/"
stylePath = staticPath ++ styleName

-- | Just a dummy as an example of ToElement - creates an empty index element.
data Index = Index
instance ToElement Index where toElement _ = emptyElem "index" []


{--
The MACID monad presumes state.  So if you have no state then you can use the noState
handler to use () as your State type.
--}
noState :: Monad m => ServerPart (Ev () req) req m res
noState = Handle request

-- | Wrap stdMain for a HTTP-only application. Can be used like:
-- 
-- > main = stdHTTP [ debugFilter,
-- >                , hs clientPath         GET  $ basicFileServe staticPath
-- >                , noState
-- >                ]
stdHTTP :: (MonadIO m, StartStateEx st st, Serialize st) => [ServerPart (Ev st Request) Request IO Result] -> m ()
stdHTTP impl = stdMain $ (simpleHTTP impl) :*: End
mainHTTP impl = simpleMain $ (simpleHTTP impl) :*: End

type HTTP_H = [ServerPart (Ev () Request) Request IO Result]
instance CanMain HTTP_H where
    asMain = mainHTTP 

basicAuth :: (Monad m, Monad m', FromReqURI uri_match (), MatchMethod req_match)
          => String -> Map String String
          -> uri_match -> req_match
          -> ServerPart m Request m' Result
basicAuth realmName authMap path meth = h path meth basicAuthImpl
  where
    basicAuthImpl () rq =
      case getHeader "authorization" rq of
        Nothing -> err
        Just x  -> case parseHeader x of 
                     (name, ':':pass) | validLogin name pass -> request rq
                     _                                       -> err
    validLogin name pass = M.lookup name authMap == Just pass
    parseHeader = break (':'==) . Base64.decode . P.unpack . P.drop 6
    headerName  = "WWW-Authenticate"
    headerValue = "Basic realm=\"" ++ realmName ++ "\""
    err = (liftM . liftM . liftM $ (addHeader headerName headerValue)) $
           unauthorized (val "") () ()

multiIf :: (MatchMethod req_match, FromReqURI uri_match (), Monad m, Monad m')
        => uri_match -> req_match
        -> [ServerPart m Request m' res]
        -> ServerPart m Request m' res
multiIf a b ss = h a b $ \() -> runServerParts ss

