module HAppS.Protocols.HTTP.FileServe
    (
     MimeMap,fileServe, mimeTypes,isDot, blockDotFiles
    ) where

import System.Log.Logger

import HAppS.Protocols.HTTP.ServerPart(ServerPart(..))
import HAppS.Protocols.HTTP.Types
import HAppS.Protocols.SURI(path)

import Control.Monad
import Control.Monad.Trans
import qualified Data.ByteString.Char8 as P
import qualified Data.Map as Map
import System.Directory
import System.IO
import System.IO.Unsafe(unsafeInterleaveIO)
import Data.List

type MimeMap = Map.Map String String

intercalate x ys = concat $ intersperse x ys

-- | Serve files with a mime type map under a directory.
--   Uses the function to transform URIs to FilePaths.
fileServe localpath pathInfo rq = fileServeImpl' mimeTypes filepath rq
    where 
    safepath = filter (\x->head x /= '.') pathInfo
    filepath = intercalate "/" (localpath:safepath)


fileServeImpl':: Map.Map String String -> String -> Request -> IO Result
fileServeImpl' mime fp rq = liftIO $ do
    fe <- doesFileExist fp
    de <- doesDirectoryExist fp
    let status | de   = "DIR"
               | fe   = "file"
               | True = "NOT FOUND"
    logM "HAppS.Protocols.HTTP.FileServe" INFO ("fileServe: "++show fp++" \t"++status)
    case () of
        _ | de        -> sresult 403 "No directory indexing!"
          | not fe    -> sresult 404 "File not found!"
          | otherwise -> do 
              t <- getModificationTime fp
              let notmodified = getHeader "if-modified-since" rq == Just (show t)
              case () of
                  _ | notmodified -> sresult 304 ""
                    | otherwise -> do
                         let bufSize = 64 * 1024
                         h <- openBinaryFile fp ReadMode
                         s <- hFileSize h
                         let loop = do s <- P.hGet h bufSize
                                       if P.length s /= bufSize
                                          then do hClose h
                                                  return [s]
                                          else do ss <- unsafeInterleaveIO loop
                                                  return (s:ss)
                         res <- sresult' 200 =<< unsafeInterleaveIO loop
                         let mt = Map.findWithDefault "text/plain" (getExt fp) mime
                         return $ setHeader "Last-modified" (show t) 
                                $ setHeader "Content-Length" (show s)
                                $ setHeader "Content-Type" mt res

getExt fPath = reverse $ takeWhile (/='.') $ reverse fPath

-- | Ready collection of common mime types.
mimeTypes :: MimeMap
mimeTypes = Map.fromList
	    [("xml","application/xml")
	    ,("xsl","application/xml")
	    ,("js","text/javascript")
	    ,("html","text/html")
	    ,("css","text/css")
	    ,("gif","image/gif")
	    ,("jpg","image/jpeg")
	    ,("png","image/png")
	    ,("txt","text/plain")
	    ,("doc","application/msword")
	    ,("exe","application/octet-stream")
	    ,("pdf","application/pdf")
	    ,("zip","application/zip")
	    ,("gz","application/x-gzip")
	    ,("ps","application/postscript")
	    ,("rtf","application/rtc")
	    ,("wav","application/x-wav")
	    ,("hs","text/plain")]



-- | Block dotfiles.
blockDotFiles :: Monad m => ServerPart m Request Result
blockDotFiles = MaybeHandle w
    where w req | isDot $ path $ rqURI req = return $ sresult 403 "Dot files not allowed."
                | otherwise                        = return Nothing

isDot = isD . reverse
    where  
    isD ('.':'/':_) = True
    isD ['.']       = True
    --isD ('/':_)     = False
    isD (_:cs)      = isD cs
    isD []          = False
