{-# OPTIONS -fglasgow-exts  -fallow-undecidable-instances #-}
--Copyright (C) 2005 HAppS.org. All Rights Reserved.
module HAppS.DBMS.Example where

import HAppS.DBMS.Table 

import qualified HAppS.DBMS.Table as Table -- hiding (Prop)
import qualified HAppS.DBMS.Index as Index

import HAppS.DBMS.IndexVal
import HAppS.Util.Common

import Maybe
import Time
import qualified Data.Set as Set



--------------------
--You define your Record types
data Item = Item {itemId::Id,stock::Int,description::String,price::Cents} 
            deriving (Ord,Eq,Read,Show)

data Sale = Sale {date::CalendarTime,soldItemId::Id,qty::Int,salePrice::Cents} 
            deriving (Ord,Eq,Read,Show)

type Cents=Int

--Define the properties on which you want to query these types
data Properties item = Id (Table.Prop Index.Ord item Id)
		     | Stock (Table.Prop Index.Ord item Int) 
                     -- you might want an IsZero index
		     | Description (Table.Prop Index.Text item String)
		     | Price (Table.Prop Index.Ord item Cents)
		     | Date (Table.Prop Index.Ord item CalendarTime)
		     | Qty (Table.Prop Index.Ord item Int)
		     | SubTotal (Table.Prop Index.Ord item Cents)

--Note we should be able to use TH to make this more concise e.g.
-- $MakeProps Properties [(Id Index.Ord Id),(Stock Index.Ord Int) ...]

--Specify how to index each record type
instance Record Item Properties where
    index = [Id=:itemId, Stock=:stock,Description=:description,Price=:price]

instance Record Sale Properties where
    index =[Id=:soldItemId,Date=:date,Qty=:qty,SubTotal=: \x->salePrice x*qty x]

--once you generate the table using template haskell you can create a "database"
data State = State {tItem::PropertiesTable Item
		   ,tSale::PropertiesTable Sale} deriving (Read,Show)

startState=State empty empty

--you should also derive u_ and a_ update functions for this record
--now you can do cool database operations

--example inserts
addItem item  = u_tItem (insert1 item) 
addItems items = u_tItem (insert items)

updatePrice itemId price state = 
    maybe state	(\item->u_tItem (replace1 item item{price=price}) state) $
	  state // tItem // getOne ?? Id .==. itemId

deleteLowPriceItems minPrice state = 
    u_tItem (delete $ state // tItem // (Price .==. minPrice)) state

itemExists itemId state = isNothing $ state // tItem // getOne ?? Id .==. itemId

getItems itemIds state = state // tItem // ( Id `Table.elem` itemIds)

getSalesSince itemIds time state = 
    state // tSale // ((Id `Table.elem` itemIds) &&& (Date .>. time))
    --how do I set the precedence to eliminate the ()?

findItems keyword state = state // tItem // Description .~. keyword

getItemSales itemIds state = 
    Table.innerGroupJoin (state // tItem) Id soldItemId agg grouped
    where 
    grouped = Table.groupByIx Id $ state // tSale // (Id `Table.elem` itemIds)
    agg = sum.map (\x->qty x*salePrice x)

-- Stuff that should be generated using TemplateHaskell or Generics,
-- but I don't know how.

data PropertiesTable item = PropertiesTable 
    {propertiesTableSet::Set.Set item -- tables are actually sets + indices
    ,propertiesId::Index.Ord item Id
    ,propertiesStock::Index.Ord item Int
    ,propertiesDescription::Index.Text item String
    ,propertiesPrice::Index.Ord item Cents
    ,propertiesDate::Index.Ord item CalendarTime
    ,propertiesQty::Index.Ord item Int
    ,propertiesSubTotal::Index.Ord item Cents
    } 

instance (Show item,Ord item) => Show (PropertiesTable item) where
    showsPrec d = showsPrec d . toList 

--instance (Show item,Ord item, Table t item p) => Show (t item) where
--	showsPrec d pt = showsPrec d $ Set.toList $ toSet pt


instance (Read item,Ord item,Record item Properties)=> 
    Read (PropertiesTable item) where
    readsPrec d s = map (\x->(flip Table.insertList empty $ fst x,snd x)) $ 
                    readsPrec d s
	            --readsPrec d s = map (\x->(Table.fromList $ fst x,snd x)) $ readsPrec d s

--instance (Ord x) => Listable PropertiesTable x where toList x = toList' x

instance (Ord item) =>Empty (PropertiesTable item) where
    empty = PropertiesTable empty empty empty empty empty empty empty empty 

instance (Ord item) => Table PropertiesTable item Properties where
    --empty = PropertiesTable Set.empty Index.empty Index.empty Index.empty 
	--Index.empty Index.empty Index.empty Index.empty 
    toSet = propertiesTableSet
    setSet set table = table {propertiesTableSet = set}
    --note there should be a way to have a insDelIndex, but what is its type?
    insertIndex item (Id (Val s)) table =
	table {propertiesId =Index.insert item s $! (propertiesId table)}
    insertIndex item (Stock (Val s)) table =
	table {propertiesStock =Index.insert item s $! (propertiesStock table)}
	-- ... fill in for the rest
    deleteIndex item (Id (Val s)) table = 
	table {propertiesId=Index.delete item s $! (propertiesId table) }
    deleteIndex item (Stock (Val s)) table = 
	table {propertiesStock=Index.delete item s $! (propertiesStock table) }
	-- ... etc.
    queryFn (Id (Fn fn)) = fn . propertiesId
    queryFn (Stock (Fn fn)) = fn . propertiesStock
	--  ... etc.
    groupByImpl (Id _) = Index.keys . propertiesId
    groupByImpl (Stock _) = Index.keys . propertiesStock
	-- ... etc.

instance (Ord x) => Listable PropertiesTable x where toList x= toList' x

--better record update syntax
u_tItem f s = s{tItem= f $ tItem s}
u_tSale f s = s{tSale= f $ tSale s}
a_tItem v s = s{tItem= v}
a_tSale v s = s{tSale= v}


--------------older example

---generated by template haskell
data DBTable item = DBTable {dbTableSet::Set.Set item
			    ,names::Index.Ord item String
			    ,ages::Index.Ord item Int
			    ,descript::Index.Text item String}

instance (Ord x)=>Listable DBTable x where toList x=toList' x

instance (Ord item) =>Empty (DBTable item) where
    empty = DBTable empty empty empty empty

instance (Ord item) =>Table DBTable item IVal where
    -- empty = DBTable Set.empty Index.empty Index.empty Index.empty
    toSet = dbTableSet
    setSet set table = table {dbTableSet = set}
    insertIndex item (Name (Val s)) table =
        table {names=Index.insert item s $! (names table) }
    insertIndex item (Age (Val i)) table =
	table {ages=(Index.insert item i $! (ages table)) }
    deleteIndex item (Name (Val s)) table =
	table {names=Index.delete item s $! (names table) }
    deleteIndex item (Age (Val i)) table =
	table {ages=(Index.delete item i $! (ages table)) }
    queryFn (Name (Fn fn)) table  =fn (names table)
    queryFn (Age (Fn fn)) table =fn (ages table)
    queryFn (Desc (Fn fn)) table = fn (descript table)
    groupByImpl (Name _ ) table = Index.keys (names table)
    groupByImpl (Age _ ) table = Index.keys (ages table)
    groupByImpl (Desc _ ) table = Index.keys (descript table)

---user created---
data IVal item = Name (Table.Prop Index.Ord item String)
	       | Age (Table.Prop Index.Ord item Int)
	       | Desc (Table.Prop Index.Text item String)

data DBItem = DBItem {name::String,age::Int} deriving (Ord,Eq,Read,Show)

instance Record  DBItem IVal where index = [Name=:name, Age=:age]

---user used

dbTable = insert [DBItem "john doe" 23] empty ::DBTable DBItem
dbunion = union dbTable dbTable

--dbQuery n= Set.toList $ from dbTable (Name.==."john doe"  &&& Age.==.n) 
dbQuery n= toList $ from dbTable (Name .==. "john doe"  &&& Age .==. n) 


items n = map (DBItem "") [1..n]
dbTable' n = head $ toList $  insert (items n) (empty::DBTable DBItem)  

main = print $ dbTable' 100000

