{-# OPTIONS -fglasgow-exts #-}
module HAppS.Agents.SessionKeeper
    (-- * Types
     SessionState, State, SessionKey,
     -- * Sessions
     newSession, getSession, setSession, delSession,
     setSessionTimeOut, pretty, setNewSession
    ) where

import qualified Control.Monad.State as S
import Data.Typeable
import HAppS.MACID
import HAppS.DBMS.RSMap as M
import HAppS.Util.Cron
import HAppS.Util.Common
import HAppS.Util.StdMain.StartState


type SessionKey   = Integer
type SMap a = M.Map SessionKey (a,EpochSeconds)
type EvSession a r = forall ev. Ev (State a) ev r

data Request = RunCleanup deriving(Read,Show,Typeable)
instance Serialize Request where
    typeString _    = "Happs.Agents.SessionKeeper.Request"
    encodeStringM _ = return ""
    decodeStringM s = return (RunCleanup, s)


instance StartState (State a) where
    startStateM = return startState
    runPart     = fmap (:[]) . sessionKeeper


data SessionState a = State { smap :: SMap a } deriving(Read,Show,Typeable)
type State a = SessionState a

-- | Initial empty state.
startState  = State M.empty

-- | Create a new session with the given lifetime in seconds relative to current time and return the new SessionKey.
newSession :: Seconds -> a -> EvSession a SessionKey
newSession ep v = do rnd <- getRandom
                     ep0 <- getTime
                     S.modify $ \st -> st { smap = M.insert (abs rnd) (v,fromIntegral ep+ep0) (smap st) }
                     return $ abs rnd

-- | Low level function to set  the absolute timeout for a SessionKey and the associated value.
setNewSession :: SessionKey -> EpochSeconds -> a -> EvSession a ()
setNewSession sk ep v = S.modify $ \st -> st {smap=M.insert sk (v,ep) $ smap st}

-- | Get session data corresponding to the SessionKey or fail.
getSession :: SessionKey -> EvSession a a
getSession sk = do st <- S.get
                   case M.lookup sk $ smap st of
                     Just (x,_) -> return x
                     Nothing    -> fail "Invalid session."

-- | Update session data corresponding to the SessionKey.
setSession :: SessionKey -> a -> EvSession a ()
setSession sk v = S.modify $ \st -> st { smap = M.update (\(_,ep) -> Just (v,ep)) sk $ smap st }

-- | Set a new absolute timeout associated with the session.
setSessionTimeOut :: SessionKey -> EpochSeconds -> EvSession a ()
setSessionTimeOut sk ep = S.modify $ \st -> st { smap = M.update (\(v,_) -> Just (v,ep)) sk $ smap st }

-- | Delete the session referred by the SessionKey.
delSession :: SessionKey -> EvSession a ()
delSession sk = S.modify $ \st -> st { smap = M.delete sk $ smap st }

sessionKeeper :: (st -> State a) -> (State a -> st -> st) -> Handler st
sessionKeeper ifun ufun = everyNthSecond 60 $ localState ifun ufun $ do
    ep <- getTime
    let f (_,time) = time > ep
    S.modify $ \st -> st { smap = M.filter f $ smap st }



-- | Show the whole session state.
pretty :: Show a => EvSession a String
pretty = fmap show S.get
