/*
     This file is part of GNUnet.
     (C) 2007 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/peers/peers.c
 * @brief code providing peer information
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include <GNUnet/gnunet_directories.h>
#include <GNUnet/gnunet_getoption_lib.h>
#include <GNUnet/gnunet_identity_lib.h>
#include <GNUnet/gnunet_util_config_impl.h>
#include <GNUnet/gnunet_util_cron.h>
#include <GNUnet/gnunet_util_network_client.h>
#include <gtk/gtk.h>

#define REFRESH_RATE (15 * cronSECONDS)

static struct CronManager *cron;

static struct GE_Context *ectx;

static struct GC_Configuration *cfg;

static GdkPixbuf *green;

static GdkPixbuf *yellow;

static GdkPixbuf *red;

static GdkPixbuf *black;

struct Flags {
  struct Flags * next;
  char * cc;
  GdkPixbuf * flag;
};

static struct Flags * flags;

static int
collector (void *data,
           const PeerIdentity * identity,
           const void *address,
           unsigned int addr_len,
           cron_t last_message, unsigned int trust, unsigned int bpmFromPeer)
{
  GtkListStore *model = data;
  GtkTreeIter iter;
  EncName enc;
  GdkPixbuf *ledBuf;
  GdkPixbuf *flagBuf;
  char *cc;
  char *dir;
  char *fn;
  char *prefix;
  char *have;
  char *haddress;
  char *hostname;
  cron_t now;
  int i;
  int found;
  struct Flags * pos;

  hash2enc (&identity->hashPubKey, &enc);
  /* check if same peer is already in model! */
  found = NO;
  if (TRUE == gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), &iter))
    {
      do
        {
          gtk_tree_model_get (GTK_TREE_MODEL (model),
                              &iter, 0, &haddress, 3, &have, -1);
          if (have != NULL)
            {
              if (0 == strcmp (have, (char *) &enc))
                {
                  if (strlen (haddress) > 0)
                    {
                      FREE (have);
                      FREE (haddress);
                      return OK;
                    }
                  found = YES;
                }
              FREENONNULL (haddress);
              FREE (have);
            }
        }
      while ((found == NO) &&
             (TRUE == gtk_tree_model_iter_next (GTK_TREE_MODEL (model),
                                                &iter)));
    }

  hostname = network_get_ip_as_string (address, addr_len, YES);
  if (hostname == NULL)
    hostname = STRDUP ("NAT");
  /* get flag */
  flagBuf = NULL;
  ledBuf = NULL;
  cc = NULL;
  prefix = STRDUP (hostname);
  if (strstr (prefix, " ") != NULL)
    *strstr (prefix, " ") = '\0';
  cc = prefix;
  while (strstr (cc, ".") != NULL)
    cc = strstr (cc, ".") + 1;
  if (strstr (hostname, ".") == NULL)
    cc = NULL;
  else if ((0 == strcasecmp (cc, "edu")) ||
           (0 == strcasecmp (cc, "com")) ||
           (0 == strcasecmp (cc, "net")) ||
           (0 == strcasecmp (cc, "org")) ||
           (0 == strcasecmp (cc, "gov")) || (0 == strcasecmp (cc, "mil")))
    cc = "us";
  else if (0 == strcasecmp (cc, "uk"))
    cc = "gb";
  if ((cc != NULL) && (strlen (cc) > 2))
    cc = NULL;
  if (cc != NULL)
    {
      pos = flags;
      while (pos != NULL) {
	if (0 == strcmp(pos->cc, cc)) {
	  flagBuf = pos->flag;
	  break;
	}
	pos = pos->next;
      }
      if (pos == NULL) {      
	cc = STRDUP (cc);
	for (i = 0; i < strlen (cc); i++)
	  cc[i] = tolower (cc[i]);
	dir = os_get_installation_path (IPK_DATADIR);
	fn = MALLOC (strlen (dir) + 32);
	strcpy (fn, dir);
	strcat (fn,
		DIR_SEPARATOR_STR ".."
		DIR_SEPARATOR_STR "gnunet-gtk"
		DIR_SEPARATOR_STR "flags" DIR_SEPARATOR_STR);
	strcat (fn, cc);
	strcat (fn, ".png");
	FREE (dir);
	flagBuf = gdk_pixbuf_new_from_file (fn, NULL);
	FREE (fn);
	pos = MALLOC(sizeof(struct Flags));
	pos->cc = cc;
	pos->flag = flagBuf;
	pos->next = flags;
	flags = pos;
      }
    }

  /* get LED */
  now = get_time ();
  if (last_message + 150 * cronSECONDS > now)
    ledBuf = green;
  else if (last_message + 5 * cronMINUTES > now)
    ledBuf = yellow;
  else if (bpmFromPeer > 0)
    ledBuf = red;
  else
    ledBuf = black;

  /* add to model */
  if (found == NO)
    gtk_list_store_append (model, &iter);
  gtk_list_store_set (model,
                      &iter,
                      0, hostname,
                      1, trust,
                      2, bpmFromPeer,
                      3, (const char *) &enc,
                      4, ledBuf, 5, flagBuf, 6, cc, 7, last_message, -1);
  FREE (prefix);
  FREE (hostname);
  return OK;
}



/**
 * Compute an updated peer model.
 */
static void *
getPeerModel ()
{
  struct ClientServerConnection *sock;
  GtkListStore *model;

  model = gtk_list_store_new (8, G_TYPE_STRING, /* address */
                              G_TYPE_UINT,      /* trust */
                              G_TYPE_UINT,      /* bpm */
                              G_TYPE_STRING,    /* identity */
                              GDK_TYPE_PIXBUF,  /* LED */
                              GDK_TYPE_PIXBUF,  /* flag */
                              G_TYPE_STRING,    /* country name */
                              G_TYPE_UINT64);   /* last time seen */
  sock = client_connection_create (ectx, cfg);
  if (sock != NULL)
    {
      gnunet_identity_request_peer_infos (sock, &collector, model);
      connection_destroy (sock);
    }
  return model;
}


/**
 * Update the model for the peers list.
 */
static void *
updatePeerInfoSafe (void *m)
{
  GtkListStore *model = m;
  GtkTreeModel *old_model;
  GtkWidget *w;
  GtkSortType order;
  gint sort_column;

  w = glade_xml_get_widget (getMainXML (), "peersTreeView");
  old_model = gtk_tree_view_get_model (GTK_TREE_VIEW (w));
  if (old_model != NULL)
    {
      if (TRUE ==
          gtk_tree_sortable_get_sort_column_id (GTK_TREE_SORTABLE (old_model),
                                                &sort_column, &order))
        gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (model),
                                              sort_column, order);
    }
  gtk_tree_view_set_model (GTK_TREE_VIEW (w), GTK_TREE_MODEL (model));
  gtk_tree_selection_set_mode (gtk_tree_view_get_selection
                               (GTK_TREE_VIEW (w)), GTK_SELECTION_NONE);
  return NULL;
}

static void
updatePeerInfo (void *dummy)
{
  GtkListStore *model;

  model = getPeerModel ();
  gtkSaveCall (&updatePeerInfoSafe, model);
}


void
init_peers (struct GE_Context *e, struct GC_Configuration *c)
{
  GtkWidget *tab;
  GtkWidget *peers;
  GtkCellRenderer *renderer;
  int col;
  GtkTreeViewColumn *column;
  char *dir;
  char *fn;

  ectx = e;
  cfg = c;
  peers = glade_xml_get_widget (getMainXML (), "peersTreeView");
  renderer = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (peers),
                                                     -1,
                                                     _("Address"),
                                                     renderer,
                                                     "text", 0, NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (peers), col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, 0);

  renderer = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (peers),
                                                     -1,
                                                     _("Trust"),
                                                     renderer,
                                                     "text", 1, NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (peers), col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, 1);

  renderer = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (peers),
                                                     -1,
                                                     _("Bandwidth"),
                                                     renderer,
                                                     "text", 2, NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (peers), col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, 2);


  renderer = gtk_cell_renderer_pixbuf_new ();
  col = gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (peers),
                                                     -1,
                                                     _("Country"),
                                                     renderer,
                                                     "pixbuf", 5, NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (peers), col - 1);
  gtk_tree_view_column_set_resizable (column, FALSE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, 6);

  renderer = gtk_cell_renderer_pixbuf_new ();
  col = gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (peers),
                                                     -1,
                                                     _("Status"),
                                                     renderer,
                                                     "pixbuf", 4, NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (peers), col - 1);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, 7);
  gtk_tree_view_column_set_resizable (column, FALSE);

  renderer = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (peers),
                                                     -1,
                                                     _("Identity"),
                                                     renderer,
                                                     "text", 3, NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (peers), col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, 3);


  dir = os_get_installation_path (IPK_DATADIR);
  fn = MALLOC (strlen (dir) + 32);
  strcpy (fn, dir);
  strcat (fn,
          DIR_SEPARATOR_STR ".."
          DIR_SEPARATOR_STR "gnunet-gtk" DIR_SEPARATOR_STR "red.png");
  red = gdk_pixbuf_new_from_file (fn, NULL);

  strcpy (fn, dir);
  strcat (fn,
          DIR_SEPARATOR_STR ".."
          DIR_SEPARATOR_STR "gnunet-gtk" DIR_SEPARATOR_STR "yellow.png");
  yellow = gdk_pixbuf_new_from_file (fn, NULL);

  strcpy (fn, dir);
  strcat (fn,
          DIR_SEPARATOR_STR ".."
          DIR_SEPARATOR_STR "gnunet-gtk" DIR_SEPARATOR_STR "green.png");
  green = gdk_pixbuf_new_from_file (fn, NULL);

  strcpy (fn, dir);
  strcat (fn,
          DIR_SEPARATOR_STR ".."
          DIR_SEPARATOR_STR "gnunet-gtk" DIR_SEPARATOR_STR "black.png");
  black = gdk_pixbuf_new_from_file (fn, NULL);
  FREE (fn);
  FREE (dir);

  tab = glade_xml_get_widget (getMainXML (), "peersScrolledWindow");
  gtk_widget_show (tab);
  cron = gnunet_gtk_get_cron ();
  cron_add_job (cron, &updatePeerInfo, REFRESH_RATE, REFRESH_RATE, NULL);
}

void
done_peers ()
{
  GtkWidget *w;
  struct Flags * pos;

  cron_del_job (cron, &updatePeerInfo, REFRESH_RATE, NULL);
  if (red != NULL)
    g_object_unref (red);
  if (green != NULL)
    g_object_unref (green);
  if (black != NULL)
    g_object_unref (black);
  if (yellow != NULL)
    g_object_unref (yellow);
  w = glade_xml_get_widget (getMainXML (), "peersTreeView");
  gtk_tree_view_set_model (GTK_TREE_VIEW (w), NULL);

  while (flags != NULL) {
    pos = flags->next;    
    if (flags->flag != NULL)
      g_object_unref(flags->flag);
    FREE (flags->cc);
    FREE(flags);
    flags = pos;
  }
}


/* end of peers.c */
