//  Gnomoradio - roboradio/song-list-radio.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version)
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "song-list-radio.h"
#include "init.h"

using namespace std;
using namespace Roboradio;

// todo: make it so the user can choose picking criteria

Roboradio::SongListRadio::SongListRadio (const ref_ptr<SongList> &sl,
					 Glib::ustring name,
					 unsigned short pr,
					 bool stock)
	: SongListMutable(name),
	  source(sl),
	  percent_recommend(pr)
{
	if (percent_recommend > 100)
		percent_recommend = 100;

	repeatable = false;
	shufflable = false;

	signal_removed.connect(mem_fun(*this, &SongListRadio::on_song_removed));
	signal_current_song_changed.connect(mem_fun(*this, &SongListRadio::on_active_song_change));
	if (stock)
		restock();
	Init::get_recommendation().signal_new_recommendations.connect(mem_fun(*this, &SongListRadio::restock));
}

Roboradio::SongListRadio::~SongListRadio ()
{
}

void Roboradio::SongListRadio::set_percent_recommend (unsigned short pr)
{
	if (pr > 100)
		pr = 100;

	if (pr != percent_recommend) {
		percent_recommend = pr;
		signal_percent_recommend_changed(pr);
	}
}

void Roboradio::SongListRadio::on_active_song_change ()
{
	if (&*current_song && current_song != begin()) {
		pop_front();
		// remove anything unavailable
		SongList::iterator p = begin();
		while (p != current_song) {
			if ((*p)->get_status().available)
				++p;
			else {
				SongList::iterator q = p;
				++q;
				remove(p);
				p = q;
			}
		}
		
		// move current song to front of list
		move(current_song, begin());
	}
	restock();
}

void Roboradio::SongListRadio::on_song_removed (iterator s)
{
	restock();
}

Time Roboradio::SongListRadio::calculate_remaining_time () const
{
	Time total = calculate_total_time();
	return begin() == end() ? total : total - (*begin())->get_length();
}

void Roboradio::SongListRadio::restock ()
{
	int rep = 1000;
	int accept_rating = Song::max_rating;
	while ((size() < 3
		|| (calculate_remaining_time() < Init::get_state().get_plan_ahead_time() * 60
		    && size() < 50))
	       && --rep) {
		if (rand() % 100 < percent_recommend) {
			SongRef song(Init::get_recommendation().get_next());
			if (&*song)
				push_back(song);
		} else {
			int songs = source->size();
			if (songs == 0)
				continue;
			int rnd = rand() % songs;
			
			SongRef song;
			
			if (rnd > songs / 2) {
				// go at it from the end
				SongList::reverse_iterator p = source->rbegin();
				int from_end = songs - rnd - 1;
				for (int i = 0; i < from_end; ++i)
					++p;
				song = *p;
			} else {
				// go at it from the beginning
				SongList::iterator p = source->begin();
				for (int i = 0; i < rnd; ++i)
					++p;
				song = *p;
			}
			
			if (song->get_rating() >= accept_rating--
			    && (song->get_status().available || song->importable())) {
				push_back(song);
				accept_rating = Song::max_rating;
			}
		}
	}

	// be sure to download the next few songs
	SongList::iterator p = begin();
	int c = 0;
	Time total_time = 0;
	while (p != end() && (c++ < 3 || total_time < 20) && c < 7) {
		(*p)->import();
		total_time += (*p)->get_length();
		++p;
	}
}
