// -*- c++ -*-

//  Gnomoradio - rainbow/rdf-resource.h
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __RAINBOW_RDF_RESOURCE_H
#define __RAINBOW_RDF_RESOURCE_H

#include <map>
#include <set>
#include <memory>
#include <sigc++/sigc++.h>
#include <libxml++/libxml++.h>
#include <glibmm.h>
#include "rainbow/http-client.h"
#include "rainbow/alarm.h"

namespace Rainbow
{
	template <class T>
	class ref_ptr
	{
	protected:
		T *s;
	public:
		explicit ref_ptr (T *p = 0) : s(p) { if (s) s->ref(); }
		ref_ptr (const ref_ptr<T> &ref) : s(ref.s) { if (s) s->ref(); }
		~ref_ptr () { if (s) s->unref(); }
		
		ref_ptr<T> & operator= (const ref_ptr<T> &p) {
			if (s != p.s) {
				if (s)
					s->unref();
				s = p.s;
				if (s)
					s->ref();
			}
			return *this;
		}
		T& operator* () const { return *s; }
		T* operator-> () const { return s; }
	};
	
	class RdfResource : public sigc::trackable
	{
	public:
		static void get_and_do (const Glib::ustring &uri,
					const sigc::slot<void,xmlpp::Element*,ref_ptr<RdfResource> > &slot);

		static void make_absolute_uri (Glib::ustring &uri,
					       const Glib::ustring &base);
		static void remove_tag_from_uri (Glib::ustring &uri,
						 Glib::ustring &id);
		static std::string absolute_uri (const Glib::ustring &uri,
						 const Glib::ustring &base)
			{ Glib::ustring u(uri); make_absolute_uri(u, base); return u; }

		void ref () { ++refcnt; }
		void unref () { if (--refcnt == 0) delete this; }

		xmlpp::Element *get_secondary_info (const Glib::ustring &uri);
		Glib::ustring get_base_uri () const { return base_uri; }

	protected:
		RdfResource (const Glib::ustring &uri);
		~RdfResource ();

		void get();
		void on_downloaded (bool success);
		void clear_signal_map ();

		std::map<Glib::ustring,sigc::signal<void,xmlpp::Element*,ref_ptr<RdfResource> >*> signal_map;
		std::auto_ptr<HttpClient> http;
		std::map<Glib::ustring,xmlpp::Element*> id_map, about_map;
		xmlpp::DomParser xml_tree;
		Glib::ustring base_uri, resource_uri;

		bool downloaded; // means that download was completed with no regard to success
		unsigned int refcnt;

		Alarm alarm;

		static std::map<Glib::ustring,RdfResource*> resource_map;
	};
}

#endif
