/* The following function for setting a window as transient for its parent
 * xterm was swiped from Zenity */

/*
 * util.c
 *
 * Copyright (C) 2002 Sun Microsystems, Inc.
 *           (C) 1999, 2000 Red Hat Inc.
 *           (C) 1998 James Henstridge
 *           (C) 1995-2002 Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors: Glynn Foster <glynn.foster@sun.com>
 *          Havoc Pennington <hp@redhat.com>
 *          James Henstridge <james@daa.com.au>
 *          Tom Tromey <tromey@redhat.com>
 */

#include <config.h>

#include <stdlib.h>
#include <gtk/gtk.h>
#ifdef GDK_WINDOWING_X11
#include <gdk/gdkx.h>
#endif
#include "gnome-gpg.h"

#ifdef GDK_WINDOWING_X11

static Window
transient_get_xterm (void)
{
  const char *wid_str = g_getenv ("WINDOWID");
  if (wid_str) {
    char *wid_str_end;
    Window wid = strtoul (wid_str, &wid_str_end, 10);
    if (*wid_str != '\0' && *wid_str_end == '\0' && wid != 0) {
      XWindowAttributes attrs;
      gdk_error_trap_push ();
      XGetWindowAttributes (GDK_DISPLAY(), wid, &attrs);
      gdk_flush();
      if (gdk_error_trap_pop () != 0) {
        return None;
      }
      return wid;
    }
  }
  return None;
}

static void
transient_x_free (void *ptr)
{
  if (ptr)
    XFree (ptr);
}

static gboolean
transient_is_toplevel (Window wid)
{
  XTextProperty prop;
  Display *dpy = GDK_DISPLAY ();
  if (!XGetWMName (dpy, wid, &prop))
    return FALSE;
  transient_x_free (prop.value);
  return !!prop.value;
}

/*
 * GNOME Terminal doesn't give us its toplevel window, but the WM needs a
 * toplevel XID for proper stacking.  Other terminals work fine without this
 * magic.  We can't use GDK here since "xterm" is a foreign window.
 */

static Window
transient_get_xterm_toplevel (void)
{
  Window xterm = transient_get_xterm ();
  Display *dpy = GDK_DISPLAY ();
  while (xterm != None && !transient_is_toplevel (xterm))
  {
    Window root, parent, *children;
    unsigned nchildren;
    XQueryTree (dpy, xterm,
                &root, &parent,
                &children, &nchildren);
    transient_x_free (children);
    if (parent == root)
      xterm = None;
    else
      xterm = parent;
  }
  return xterm;
}

static void
zenity_util_make_transient (GdkWindow *window)
{
  Window xterm = transient_get_xterm_toplevel ();
  if (xterm != None) {
    GdkWindow *gdkxterm = gdk_window_foreign_new (xterm);
    if (gdkxterm) {
      gdk_window_set_transient_for (window, gdkxterm);
      g_object_unref (G_OBJECT (gdkxterm));
    }
  }
}

#endif /* GDK_WINDOWING_X11 */

void
zenity_util_show_dialog (GtkWidget *dialog)
{
  gtk_widget_realize (dialog);
#ifdef GDK_WINDOWING_X11
  g_assert (dialog->window);
  zenity_util_make_transient (dialog->window);
#endif
  gtk_widget_show (dialog);
}

