/*--------------------------------------------------------------------
 *	$Id: psbasemap.c,v 1.39 2007/03/24 01:42:07 pwessel Exp $
 *
 *	Copyright (c) 1991-2007 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: gmt.soest.hawaii.edu
 *--------------------------------------------------------------------*/
/*
 * psbasemap plots out a basemap for the given area using the specified map
 * projection.
 *
 * Author:	Paul Wessel
 * Date:	08-JUL-2000
 * Version:	4
 *
 */
 
#include "gmt.h"
#include "pslib.h"

struct PSBASEMAP_CTRL {
	struct E {	/* -Eazim/elev */
		BOOLEAN active;
		double azimuth, elevation;
	} E;
	struct G {	/* -G<fill> */
		BOOLEAN active;
		struct GMT_FILL fill;
	} G;
	struct L {	/* -L */
		BOOLEAN active;
		struct GMT_MAP_SCALE item;
	} L;
	struct T {	/* -L */
		BOOLEAN active;
		struct GMT_MAP_ROSE item;
	} T;
	struct Z {	/* -Z<z_level> */
		BOOLEAN active;
		double level;
	} Z;
};

int main (int argc, char **argv)
{
	int i;

	BOOLEAN error = FALSE;

	double w = 0.0, e = 0.0, s = 0.0, n = 0.0;

	struct PSBASEMAP_CTRL *Ctrl;

	void *New_Psbasemap_Ctrl (), Free_Psbasemap_Ctrl (struct PSBASEMAP_CTRL *C);
	
	argc = GMT_begin (argc, argv);

	Ctrl = (struct PSBASEMAP_CTRL *)New_Psbasemap_Ctrl ();	/* Allocate and initialize a new control structure */

	for (i = 1; i < argc; i++) {
		if (argv[i][0] != '-') {
			error = TRUE;
			fprintf (stderr, "%s: GMT SYNTAX ERROR:  Options must start with leading -. Your argument \"%s\" is unrecognized\n", GMT_program, argv[i]);
			continue;
		}

		switch (argv[i][1]) {

			/* Common parameters */

			case 'B':
			case 'J':
			case 'K':
			case 'O':
			case 'P':
			case 'R':
			case 'U':
			case 'V':
			case 'X':
			case 'x':
			case 'Y':
			case 'y':
			case 'c':
			case '\0':
				error += GMT_parse_common_options (argv[i], &w, &e, &s, &n);
				break;

			/* Supplemental options */

			case 'E':
				Ctrl->E.active = TRUE;
				sscanf (&argv[i][2], "%lf/%lf", &Ctrl->E.azimuth, &Ctrl->E.elevation);
				break;

			case 'G':
				Ctrl->G.active = TRUE;
				if (GMT_getfill (&argv[i][2], &Ctrl->G.fill)) {
					GMT_fill_syntax ('G', " ");
					error++;
				}
				break;

			case 'L':
				Ctrl->L.active = TRUE;
				error += GMT_getscale (&argv[i][2], &Ctrl->L.item);
				break;

			case 'T':
				Ctrl->T.active = TRUE;
				error += GMT_getrose (&argv[i][2], &Ctrl->T.item);
				break;

			case 'Z':
				if (argv[i][2]) {
					Ctrl->Z.level = atof (&argv[i][2]);
					Ctrl->Z.active = TRUE;
				}
				else {
					error++;
					fprintf (stderr, "%s: GMT SYNTAX ERROR -Z:  Must append a new z-value [0]\n", GMT_program);
				}
				break;

			/* Illegal options */

			default:
				error = TRUE;
				GMT_default_error (argv[i][1]);
				break;
		}
	}

	if (GMT_give_synopsis_and_exit || argc == 1) {
		fprintf (stderr,"psbasemap %s - To plot PostScript basemaps\n\n", GMT_VERSION);
		fprintf (stderr, "usage: psbasemap %s %s %s [-Eaz/el] [-G<fill>]\n", GMT_B_OPT, GMT_J_OPT, GMT_Rgeoz_OPT);
		fprintf (stderr, "\t[-K] [%s] [%s]\n", GMT_Jz_OPT, GMT_SCALE);
		fprintf (stderr, "\t[-O] [-P] [%s] [%s] [-V]\n", GMT_TROSE, GMT_U_OPT);
		fprintf (stderr, "\t[%s] [%s] [-Z<zlevel>] [%s]\n\n", GMT_X_OPT, GMT_Y_OPT, GMT_c_OPT);

		if (GMT_give_synopsis_and_exit) exit (EXIT_FAILURE);

		GMT_explain_option ('B');
		GMT_explain_option ('J');
		GMT_explain_option ('Z');
		GMT_explain_option ('R');
		fprintf (stderr, "\n\tOPTIONS:\n");
		fprintf (stderr, "\t-E set azimuth and elevation of viewpoint for 3-D perspective [180/90]\n");
		GMT_fill_syntax ('G', "Select fill inside of basemap.");
		GMT_explain_option ('K');
		fprintf (stderr, "\t-L draws a simple map scaLe centered on <lon0>/<lat0>.  Use -Lx to specify Cartesian\n");
		fprintf (stderr, "\t   coordinates instead.  Scale is calculated at latitude <slat>; optionally give longitude\n");
		fprintf (stderr, "\t   <slon> [Default is central longitude].  <length> is in km, or [nautical] miles if [n] m\n");
		fprintf (stderr, "\t   is appended.  -Lf draws a \"fancy\" scale [Default is plain]. By default, the label is set\n");
		fprintf (stderr, "\t   to the distance unit and placed on top [t].  Use the :label:<just> mechanism to specify another\n");
		fprintf (stderr, "\t   label (or - to keep the default) and placement (t,b,l,r, and u - to use the label as a unit.\n");
		fprintf (stderr, "\t   Append +p<pen> and/or +f<fill> to draw/paint a rectangle beneath the scale [no rectangle]\n");
		GMT_explain_option ('O');
		GMT_explain_option ('P');
		fprintf (stderr, "\t-T draws a north-poinTing rose centered on <lon0>/<lat0>.  Use -Tx to specify Cartesian\n");
		fprintf (stderr, "\t   coordinates instead.  -Tf draws a \"fancy\" rose [Default is plain].  Give rose diameter\n");
		fprintf (stderr, "\t   <size> and optionally the west, east, south, north labels desired [W,E,S,N].\n");
		fprintf (stderr, "\t   For fancy rose, specify as <info> the kind you want: 1 draws E-W, N-S directions [Default],\n");
		fprintf (stderr, "\t   2 adds NW-SE and NE-SW, while 3 adds WNW-ESE, NNW-SSE, NNE-SSW, and ENE-WSW.\n");
		fprintf (stderr, "\t   For Magnetic compass rose, specify -Tm.  Use the optional <info> = <dec>/<dlabel> (where <dec> is\n");
		fprintf (stderr, "\t   the magnetic declination and <dlabel> is a label for the magnetic compass needle) to plot\n");
		fprintf (stderr, "\t   directions to both magnetic and geographic north [Default is just geographic].\n");
		fprintf (stderr, "\t   If the North label = \'*\' then a north star is plotted instead of the label.\n");
		fprintf (stderr, "\t   Append +<gints>/<mints> to override default annotation/tick interval(s) [10/5/1/30/5/1].\n");
		GMT_explain_option ('U');
		GMT_explain_option ('V');
		GMT_explain_option ('X');
		fprintf (stderr, "\t-Z For 3-D plots: Set the z-level of map [0]\n");
		GMT_explain_option ('c');
		GMT_explain_option ('.');
		exit (EXIT_FAILURE);
	}

	if (!project_info.region_supplied) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify -R option\n", GMT_program);
		error++;
	}
	if (!(frame_info.plot || Ctrl->L.active || Ctrl->T.active || Ctrl->G.active)) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify at least one of -B, -G, -L, -T\n", GMT_program);
		error++;
	}
	if (Ctrl->E.elevation <= 0.0 || Ctrl->E.elevation > 90.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -E option:  Elevation must be in 0-90 range\n", GMT_program);
		error++;
	}

	if (error) exit (EXIT_FAILURE);

	z_project.view_azimuth = Ctrl->E.azimuth;
	z_project.view_elevation = Ctrl->E.elevation;

	if (gmtdefs.verbose) fprintf (stderr, "psbasemap: Constructing basemap\n");

	GMT_err_fail (GMT_map_setup (w, e, s, n), "");

	GMT_plotinit (argc, argv);

	if (project_info.three_D) ps_transrotate (-z_project.xmin, -z_project.ymin, 0.0);

	if (Ctrl->G.active) {
		double *x, *y;
		int np, donut;
		np = GMT_map_clip_path (&x, &y, &donut);
		GMT_fill (x, y, (1 + donut) * np, &Ctrl->G.fill, FALSE);
		GMT_free ((void *)x);
		GMT_free ((void *)y);
	}

	if (Ctrl->Z.active) project_info.z_level = Ctrl->Z.level;

	GMT_map_basemap ();

	if (Ctrl->L.active) GMT_draw_map_scale (&Ctrl->L.item);

	if (Ctrl->T.active) GMT_draw_map_rose (&Ctrl->T.item);

	if (project_info.three_D) ps_rotatetrans (z_project.xmin, z_project.ymin, 0.0);
	
	GMT_plotend ();

	Free_Psbasemap_Ctrl (Ctrl);	/* Deallocate control structure */

	GMT_end (argc, argv);

	exit (EXIT_SUCCESS);
}

void *New_Psbasemap_Ctrl () {	/* Allocate and initialize a new control structure */
	struct PSBASEMAP_CTRL *C;
	
	C = (struct PSBASEMAP_CTRL *) GMT_memory (VNULL, 1, sizeof (struct PSBASEMAP_CTRL), "New_Psbasemap_Ctrl");
	
	/* Initialize values whose defaults are not 0/FALSE/NULL */
	C->E.azimuth = 180.0;
	C->E.elevation = 90.0;
	GMT_init_fill (&C->G.fill, -1, -1, -1);
	memset ((void *)&C->L.item, 0, sizeof (struct GMT_MAP_SCALE));
	memset ((void *)&C->T.item, 0, sizeof (struct GMT_MAP_ROSE));
		
	return ((void *)C);
}

void Free_Psbasemap_Ctrl (struct PSBASEMAP_CTRL *C) {	/* Deallocate control structure */
	GMT_free ((void *)C);	
}
