/*  GFC-Core: GTK+ Foundation Classes (Core Library)
 *  Copyright (C) 2003-2004 The GFC Development Team.
 *
 *  asyncqueue.cc - A GAsyncQueue C++ wrapper interface.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "main.hh"
#include "timeval.hh"

using namespace GFC;

/*  G::get_current_time
 */

void
G::get_current_time(TimeVal& timeval)
{
	g_get_current_time(timeval);
}

/*  G::MainContext
 */

G::MainContext::MainContext()
: context_(g_main_context_new())
{
}

G::MainContext::MainContext(GMainContext *context, bool owns_reference)
: context_(context_)
{
	set_owns_reference(owns_reference);
}

G::MainContext::~MainContext()
{
	if (context_)
	{
		unref();
	}
}

Pointer<G::MainContext>
G::MainContext::get_default()
{
	GMainContext *context = g_main_context_default();
	return context ? new G::MainContext(context, false) : 0;
}

void
G::MainContext::ref()
{
	g_main_context_ref(context_);
	GFC::Object::ref();
}

void
G::MainContext::unref()
{
	g_main_context_unref(context_);
	context_ = 0;
	GFC::Object::unref();
}

/*  G::MainLoop
 */

G::MainLoop::MainLoop(bool is_running)
: loop_(g_main_loop_new(0, is_running))
{
}

G::MainLoop::MainLoop(MainContext& context, bool is_running)
: loop_(g_main_loop_new(context.g_main_context(), is_running))
{
}

G::MainLoop::~MainLoop()
{
	if (loop_)
	{
		unref();
	}
}

void
G::MainLoop::ref()
{
	g_main_loop_ref(loop_);
	GFC::Object::ref();
}

void
G::MainLoop::unref()
{
	g_main_loop_unref(loop_);
	loop_ = 0;
	GFC::Object::unref();
}

Pointer<G::MainContext>
G::MainLoop::get_context() const
{
	GMainContext *context = g_main_loop_get_context(loop_);
	return context ? new MainContext(context, false) : 0;
}

/*  G::Source
 */

G::Source::Source(GSource *source, bool owns_reference)
: source_(source)
{
	set_owns_reference(owns_reference);
}

G::Source::~Source()
{
	if (source_)
	{
		g_source_unref(source_);
		source_ = 0;
	}
}

Pointer<G::MainContext>
G::Source::get_context() const
{
	GMainContext *context = g_source_get_context(source_);
	return context ? new G::MainContext(context, false) : 0;
}

void
G::Source::get_current_time(TimeVal& timeval)
{
	g_source_get_current_time(source_, timeval.g_time_val());
}

bool
G::Source::is_attached(G::MainContext& context) const
{
	return g_source_get_context(source_) == context.g_main_context();
}

unsigned int
G::Source::attach(MainContext *context)
{
	return g_source_attach(source_, *context);
}

void
G::Source::ref()
{
	g_source_ref(source_);
}

void
G::Source::unref()
{
	bool last_unref = source_->ref_count == 1;
	g_source_unref(source_);
	source_ = 0;
	if (last_unref && is_dynamic())
	{
		delete this;
	}
}

namespace { // ChildWatchCallback

struct ChildWatchCallback
{
	typedef G::ChildWatchSource::WatchSlot WatchSlot;
	WatchSlot slot_;

	ChildWatchCallback(const WatchSlot& slot)
	: slot_(slot)
	{
	}

	static void notify(GPid pid, int status, void *data)
	{
		ChildWatchCallback *cb = static_cast<ChildWatchCallback*>(data);
		cb->slot_(pid, status);
	}

	static void destroy(void *data)
	{
		ChildWatchCallback *cb = static_cast<ChildWatchCallback*>(data);
		delete cb;
	}
};

} // namespace

/*  G::ChildWatchSource
 */

G::ChildWatchSource::ChildWatchSource(GPid pid)
: G::Source(g_child_watch_source_new(pid))
{
	set_priority(PRIORITY_DEFAULT);
}

G::ChildWatchSource::ChildWatchSource(GPid pid, const WatchSlot& slot)
: G::Source(g_child_watch_source_new(pid))
{
	ChildWatchCallback *cb = new ChildWatchCallback(slot);
	g_source_set_callback(g_source(), (GSourceFunc)&ChildWatchCallback::notify, cb, &ChildWatchCallback::destroy);
	set_priority(PRIORITY_DEFAULT);
}

void
G::ChildWatchSource::set_callback(const WatchSlot& slot)
{
	ChildWatchCallback *cb = new ChildWatchCallback(slot);
	g_source_set_callback(g_source(), (GSourceFunc)&ChildWatchCallback::notify, cb, &ChildWatchCallback::destroy);
}

namespace { // SourceCallback

struct SourceCallback
{
	typedef sigc::slot<bool> SourceSlot;
	SourceSlot slot_;

	SourceCallback(const SourceSlot& slot)
	: slot_(slot)
	{
	}

	static gboolean notify(void *data)
	{
		SourceCallback *cb = static_cast<SourceCallback*>(data);
		return cb->slot_();
	}

	static void destroy(void *data)
	{
		SourceCallback *cb = static_cast<SourceCallback*>(data);
		delete cb;
	}
};

} // namespace

/*  G::TimeoutSource
 */

G::TimeoutSource::TimeoutSource(unsigned int interval)
: G::Source(g_timeout_source_new(interval))
{
	set_priority(PRIORITY_HIGH);
}

G::TimeoutSource::TimeoutSource(const TimeoutSlot& slot, unsigned int interval)
: G::Source(g_timeout_source_new(interval))
{
	SourceCallback *cb = new SourceCallback(slot);
	g_source_set_callback(g_source(), &SourceCallback::notify, cb, &SourceCallback::destroy);
	set_priority(PRIORITY_HIGH);
}

void
G::TimeoutSource::set_callback(const TimeoutSlot& slot)
{
	SourceCallback *cb = new SourceCallback(slot);
	g_source_set_callback(g_source(), &SourceCallback::notify, cb, &SourceCallback::destroy);
}

/*  G::IdleSource
 */

G::IdleSource::IdleSource()
: G::Source(g_idle_source_new())
{
	set_priority(PRIORITY_DEFAULT_IDLE);
}

G::IdleSource::IdleSource(const IdleSlot& slot)
: G::Source(g_idle_source_new())
{
	SourceCallback *cb = new SourceCallback(slot);
	g_source_set_callback(g_source(), &SourceCallback::notify, cb, &SourceCallback::destroy);
	set_priority(PRIORITY_DEFAULT_IDLE);
}

void
G::IdleSource::set_callback(const IdleSlot& slot)
{
	SourceCallback *cb = new SourceCallback(slot);
	g_source_set_callback(g_source(), &SourceCallback::notify, cb, &SourceCallback::destroy);
}

namespace { // IOWatchCallback

struct IOWatchCallback
{
	typedef G::IOSource::IOSlot IOSlot;
	IOSlot slot_;

	IOWatchCallback(const IOSlot& slot)
	: slot_(slot)
	{
	}

	static gboolean notify(GIOChannel *source, GIOCondition condition, void *data)
	{
		IOWatchCallback *cb = static_cast<IOWatchCallback*>(data);
		Pointer<G::IOChannel> tmp_channel = new G::IOChannel(source, false);
		return cb->slot_(*tmp_channel, (G::IOConditionField)condition);
	}

	static void destroy(void *data)
	{
		IOWatchCallback *cb = static_cast<IOWatchCallback*>(data);
		delete cb;
	}
};

} // namespace

/*  G::IOSource
 */

G::IOSource::IOSource(G::IOChannel& channel, G::IOConditionField condition)
: G::Source(g_io_create_watch(channel.g_io_channel(), (GIOCondition)condition))
{
	set_priority(PRIORITY_DEFAULT);
}

G::IOSource::IOSource(G::IOChannel& channel, G::IOConditionField condition, const IOSlot& slot)
: G::Source(g_io_create_watch(channel.g_io_channel(), (GIOCondition)condition))
{
	IOWatchCallback *cb = new IOWatchCallback(slot);
	g_source_set_callback(g_source(), (GSourceFunc)&IOWatchCallback::notify, cb, &IOWatchCallback::destroy);
	set_priority(PRIORITY_DEFAULT);
}

void
G::IOSource::set_callback(const IOSlot& slot)
{
	IOWatchCallback *cb = new IOWatchCallback(slot);
	g_source_set_callback(g_source(), (GSourceFunc)&IOWatchCallback::notify, cb, &IOWatchCallback::destroy);
}

/*  G::ChildWatchSignal
 */

G::ChildWatchSignal G::child_watch_signal;

sigc::connection
G::ChildWatchSignal::connect(const SlotType& slot, GPid pid, int priority)
{
	ChildWatchCallback *cb = new ChildWatchCallback(slot);
	g_child_watch_add_full(priority, pid, &ChildWatchCallback::notify, cb, &ChildWatchCallback::destroy);
	return sigc::connection(const_cast<SlotType&>(slot));
}

/*  G::TimeoutSignal
 */

G::TimeoutSignal G::timeout_signal;

sigc::connection
G::TimeoutSignal::connect(const SlotType& slot, unsigned int interval, int priority)
{
	SourceCallback *cb = new SourceCallback(slot);
	g_timeout_add_full(priority, interval, &SourceCallback::notify, cb, &SourceCallback::destroy);
	return sigc::connection(const_cast<SlotType&>(slot));
}

/*  G::IdleSignal
 */

G::IdleSignal G::idle_signal;

sigc::connection
G::IdleSignal::connect(const SlotType& slot, int priority)
{
	SourceCallback *cb = new SourceCallback(slot);
	g_idle_add_full(priority, &SourceCallback::notify, cb, &SourceCallback::destroy);
	return sigc::connection(const_cast<SlotType&>(slot));
}

/*  G::IOSignal
 */

G::IOSignal G::io_signal;

sigc::connection
G::IOSignal::connect(G::IOChannel& channel, G::IOConditionField condition, const SlotType& slot, int priority)
{
	IOWatchCallback *cb = new IOWatchCallback(slot);
	g_io_add_watch_full(channel.g_io_channel(), priority, (GIOCondition)condition, &IOWatchCallback::notify, cb, &IOWatchCallback::destroy);
	return sigc::connection(const_cast<SlotType&>(slot));
}

