/* Copyright (C) 2004-2006 Damir Zucic */

/*=============================================================================

			    hyphob_function4.c

Purpose:
	Draw the hydrophobicity function F4.  The sequence stored to the
	main sequence buffer  is used  to calculate  the function value.

Input:
	(1) Pointer to the storage where the minimal function value will
	    be stored.
	(2) Pointer to the storage where the maximal function value will
	    be stored.
	(3) Pointer to RuntimeS structure.

Output:
	(1) Function F4 calculated and stored.
	(2) Return value.

Return value:
	(1) Positive on success.
	(2) Negative on failure.

Notes:
	(1) The function  F4 may be modified and used for many purposes.
	    Originally, it was introduced while searching for the method
	    which will be suitable for prediction of the porin secondary
	    structure.

========includes:============================================================*/

#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======calculate the hydrophobicity function F4:============================*/

int HyphobFunction4_ (double *min_functionP, double *max_functionP,
		      RuntimeS *runtimeSP)
{
int             residuesN, residueI;   /* Do not use size_t instead of int ! */
int		max_length;
int		window_width = 7;
int		half_window_width;
int		windowI, combinedI;
int		used_residuesN;
char		*residue_nameP;
char		residue_nameA[RESNAMESIZE];
int		residue_from_setF;
double		function_value;

/*------prepare some parameters:---------------------------------------------*/

/* The number of residues in sequence buffer: */
residuesN = (int) runtimeSP->residuesN;
if (residuesN == 0) return -1;

/* The maximal residue name length: */
max_length = RESNAMESIZE - 1;

/* Half of the window width: */
half_window_width = window_width / 2;

/*------calculate the function F4:-------------------------------------------*/

/* Initialize the extreme values: */
*min_functionP = +999999.0;
*max_functionP = -999999.0;

/* 20040927.1536: */
/* A window of window_width residues is scanned. The total number of very */
/* polar residues  (ASP, GLU, ASN, GLN, ARG, LYS, but no HIS) is counted. */

/* Initialize (reset) the function value: */
for (residueI = 0; residueI < residuesN; residueI++)
	{
	/* Reset the function value, it might be initialized before: */
	*(runtimeSP->function4P + residueI) = 0.0;
	}

/* Scan the sequence: */
for (residueI = 0; residueI < residuesN; residueI++)
	{
	/* Reset the counts of residues used to prepare the raw values: */
	used_residuesN = 0;

	/* Reset the function value, it might be initialized before: */
	*(runtimeSP->function4P + residueI) = 0.0;

	/* Reset the local variable reserved for the function value: */
	function_value = 0.0;

	/* Scan the sliding window: */
	for (windowI = 0; windowI < window_width; windowI++)
		{
		/* Prepare and check the combined index: */
		combinedI = residueI + windowI - half_window_width;
		if (combinedI < 0) continue;
		if (combinedI >= residuesN) continue;

		/* Pointer  to  the residue  name of */
		/* the residue defined by combinedI: */
		residue_nameP = runtimeSP->sequenceP + combinedI * max_length;

		/* Copy the residue name: */
		strncpy (residue_nameA, residue_nameP, max_length);
		residue_nameA[max_length] = '\0';

		/* Now check the residue name. */

		/* Initial assumption: residue belongs to the specified set: */
		residue_from_setF = 1;

		/* Formal, single pass loop: */
		do
			{
			/* Check the residue name: */
			if      (strcmp (residue_nameA, "ASP") == 0) break;
			else if (strcmp (residue_nameA, "GLU") == 0) break;
			else if (strcmp (residue_nameA, "ASN") == 0) break;
			else if (strcmp (residue_nameA, "GLN") == 0) break;
			else if (strcmp (residue_nameA, "ARG") == 0) break;
			else if (strcmp (residue_nameA, "LYS") == 0) break;

			/* If this point is reached, this residue */
			/* does not belong  to the specified set. */
			residue_from_setF = 0;

			/* End of single pass loop: */
			} while (0);

		/* If  this residue belongs to  the specified */
		/* set, increment the function value for one: */
		if (residue_from_setF) function_value += 1.0;

		/* Update the number of used residues: */
		used_residuesN++;
		}

	/* Reduce the function value (arbitrary scaling): */
	if (function_value < 4) function_value = 0.0; /*@@*/
	function_value /= 4.0; /*@@*/

	/* Store the function value: */
	*(runtimeSP->function4P + residueI) = function_value;

        /* Dummy values should not be used to determine extreme values: */
        if (used_residuesN < window_width) continue;

	/* Find the extreme values: */
	if (function_value < *min_functionP) *min_functionP = function_value;
	if (function_value > *max_functionP) *max_functionP = function_value;

	/* End of residueI loop: */
	}

/*---------------------------------------------------------------------------*/

return 1;
}

/*===========================================================================*/


