/*
 *
 * Copyright (c) 2003 The Regents of the University of California.  All 
 * rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Neither the name of the University nor the names of its
 *   contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
 

/*
 * /proc/zero test program
 *
 * $Id: zero.c,v 1.7 2003/07/11 22:29:40 cerpa Exp $
 */


#include <linux/config.h>
#include <linux/stddef.h>
#include <linux/tqueue.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#ifdef MODULE
#include <linux/module.h>
#endif
#include <linux/timer.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/devfs_fs_kernel.h>

#ifndef MODULE
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif

/************************************************************************/

static ssize_t zero_read(struct file *file,
			 char *user_buffer,    /* The buffer to fill with data */
			 size_t user_length,   /* The length of the buffer */
			 loff_t *offset)  /* Our offset in the file */
{
  int retval = 0;
  char *buffer;
  buffer = kmalloc(user_length, GFP_KERNEL);
  if (buffer == NULL) {
    return -ENOMEM;
  }

  memset(buffer, 0, user_length);

  if (copy_to_user(user_buffer, buffer, user_length)) {
    retval = -EFAULT;
    goto out;
  }

  retval = user_length;
  
 out:
  kfree(buffer);
  return retval;
}


static unsigned int zero_poll(struct file *file, poll_table *wait)
{
  return POLLIN | POLLRDNORM;
}


static int zero_open(struct inode *inode, struct file *file)
{
  MOD_INC_USE_COUNT;
  return 0;
}


static int zero_release(struct inode *inode, struct file *file)
{
  MOD_DEC_USE_COUNT;
  return 0;
}


static struct file_operations zero_fops = {
  owner: THIS_MODULE,
  read: zero_read,
  poll: zero_poll,
  open: zero_open,
  release: zero_release
};


static devfs_handle_t zero_handle = 0;

int init_module(void)
{
  zero_handle = devfs_register(NULL, "zero2",
			       DEVFS_FL_AUTO_DEVNUM,
			       0, 0,
			       S_IFCHR | S_IRUSR | S_IWUSR, //| 0666,
			       &zero_fops, NULL);

  if (zero_handle == 0) {
    printk("zero2: unable to register character device\n");
    return -EIO;
  }

  return 0;
}



void cleanup_module(void)
{
  /* unregister the character device */
  devfs_unregister(zero_handle);
  printk("done..\n");
}



