<?
	/**
	 * Filesystem functions for use with the Human Theme for Firefox
	 * creation script.
	 */
	
	// define the base directories for icons
	DEFINE(ICONDIR, '/usr/share/icons/');
	DEFINE(HUMANDIR, ICONDIR.'Human/');
	DEFINE(TANGERINEDIR, ICONDIR.'Tangerine/');
	DEFINE(GNOMEDIR, ICONDIR.'gnome/');
	DEFINE(CUSTOMDIR, 'custom/');
	DEFINE(CUSTOMICONDIR, CUSTOMDIR.'icons/');
	 
	// define icon categories
	DEFINE(ACTIONS, 'actions');
	DEFINE(APPS, 'apps');
	DEFINE(CATEGORIES, 'categories');
	DEFINE(DEVICES, 'devices');
	DEFINE(EMBLEMS, 'emblems');
	DEFINE(EMOTES, 'emotes');
	DEFINE(MIMETYPES, 'mimetypes');
	DEFINE(PLACES, 'places');
	DEFINE(STATUS, 'status');
	DEFINE(STOCKGENERIC, 'stock/generic');
	
	// size constant for scalable versions
	DEFINE(SCALABLE, '-1');
	DEFINE(SCALABLEDIR, 'scalable/');
	
	// extension constants
	DEFINE(PNG, 'png');
	DEFINE(SVG, 'svg');
	
	// temporary files cache
	global $tempfiles;
	$tempfiles = array();

	/**
	 * Get the preferred filename for an icon.
	 *
	 * This method will search for the desired icon file
	 * by falling back through sets. The current orders
	 * are:
	 * 
	 * Human:		Human -> Tangerine -> gnome
	 * Tangerine:	Tangerine -> gnome
	 * gnome:		gnome -> Tangerine
	 *
	 * If the icon can not be found, the custom icon set
	 * is searched. If it is not available there, a fatal
	 * error is thrown and execution is stopped.
	 * 
	 * This method uses the global $theme variable to
	 * determine the fallback icon order.
	 */
	function getIconFilenameFor($category, $icon, $size) {
		// global theme selection
		global $theme;

		// determine the filenames for each scenario
		$human_png =
			formatIconFilename(
				HUMANDIR, $category, $icon, $size);
		$human_svg =
			formatIconFilename(
				HUMANDIR, $category, $icon, SCALABLE);
		$tangerine_png =
			formatIconFilename(
				TANGERINEDIR, $category, $icon, $size);
		$tangerine_svg =
			formatIconFilename(
				TANGERINEDIR, $category, $icon, SCALABLE);
		$gnome_png =
			formatIconFilename(
				GNOMEDIR, $category, $icon, $size);
		$gnome_svg =
			formatIconFilename(
				GNOMEDIR, $category, $icon, SCALABLE);
		
		// add custom path for supplied fallback icons
		$custom_png =
			formatIconFilename(
				CUSTOMICONDIR, $category, $icon, $size);
		$custom_svg = 
			formatIconFilename(
				CUSTOMICONDIR, $category, $icon, SCALABLE);

		// build the list of desired filenames
		$filenames = array();
		switch($theme) {
			case HUMAN:
				// fall through to Tangerine
				$filenames[] = $human_png;
				$filenames[] = $human_svg;
			case TANGERINE:
				// fall back to gnome
				$filenames[] = $tangerine_png;
				$filenames[] = $tangerine_svg;
				$filenames[] = $gnome_png;
				$filenames[] = $gnome_svg;
				break;
			case GNOME:
				// gnome falls back to Tangerine for SVGs
				$filenames[] = $gnome_png;
				$filenames[] = $gnome_svg;
				$filenames[] = $tangerine_png;
				$filenames[] = $tangerine_svg;
				break;
		}
		
		// always add custom fallback paths
		$filenames[] = $custom_png;
		$filenames[] = $custom_svg;
		
		// check for the first matching filename
		foreach($filenames as $filename) {
			if(file_exists($filename)) {
				echo "$theme $category $icon $size ==> $filename\n";
				debugIcon($category, $icon, $size, $filename);
				return $filename;
			}
		}
		
		// no matching file found, bail out
		echo "Fatal error: no icon for $theme $category $icon $size\n";
		echo "None of these files exist:\n";
		print_r($filenames);
		die(2);
	}
	
	/**
	 * Get the filename for an icon in a given size.
	 *
	 * Use the SCALABLE constant to get the SVG filename.
	 */
	function formatIconFilename($icondir, $category, $icon, $size) {
		if(SCALABLE == $size) {
			return $icondir.SCALABLEDIR.$category.'/'.$icon.'.svg';
		} else {
			$sizedir = $size.'x'.$size.'/';
			return $icondir.$sizedir.$category.'/'.$icon.'.png';
		}
	}
	
	/**
	 * Get a custom image included with the script.
	 */
	function getCustomImage($image) {
		// check if the file exists
		$filename = CUSTOMDIR.$image.'.png';
		
		if(file_exists($filename)) {
			return $filename; 
		} else {
			echo "Fatal error: no custom icon $filename\n";
			die(2);
		}
	}
	
	/**
	 * Create and return a new temporary file name.
	 */
	function getTempFile($extension) {
		$tempnam = tempnam('/tmp', 'htf');

		// add temporary file to the tempfiles list
		global $tempfiles;
		$tempfiles[] = array($tempnam, $extension);
		
		return $tempnam.'.'.$extension;	
	}
	
	/**
	 * Check if the given filename is an SVG image.
	 */
	function isSvg($filename) {
		$pathparts = pathinfo($filename);
		if(SVG == $pathparts['extension']) {
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * Check if the given filename is a custom icon.
	 */
	function isCustom($filename) {
		if(strpos($filename, CUSTOMDIR) === FALSE) {
			return false;
		} else {
			return true;
		}		
	}
	
	/**
	 * Check if the given filename is a correctly themed icon.
	 */
	function isThemed($filename) {
		global $theme;
		
		$themedir = 'none';		
		switch($theme) {
			case HUMAN:
				$themedir = HUMANDIR;
				break;
			case TANGERINE:
				$themedir = TANGERINEDIR;
				break;
			case GNOME:
				$themedir = GNOMEDIR;
				break;
		}
		
		if(strpos($filename, $themedir) === FALSE) {
			return false;
		} else {
			return true;
		}
	}

	/**
	 * Clean up all the created temporary files.
	 */
	function cleanupTempFiles() {
		global $tempfiles;
		$count = count($tempfiles) * 2;
		
		echo "Cleaning up $count temporary file(s)\n";
		foreach($tempfiles as $tempfile) {
			$base = $tempfile[0];
			$ext = $tempfile[1];			
			
			unlink($base);
			unlink($base.'.'.$ext);
		}
	}

?>