# FIAIF is an Intelligent firewall, version: $Revision: 1.140 $
#
# description: Automates a packet filtering firewall with iptables.
#
# Script Author:	Anders Fugmann <afu@fugmann.net>
# 
# FIAIF is an Intelligent firewall
# Copyright (C) 2002-2003 Anders Peter Fugmann
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

###############################################################################
# Read a zone
# RETURN=0 if reading was successful
###############################################################################
function read_zone ()
{
    local ZONE=$1

    # Clear all variables
    for VAR in INPUT OUTPUT FORWARD SNAT IP_EXTRA NET_EXTRA MARK ${!REPLY_*} \
	       ${!REDIRECT_*} ${!LIMIT_*} ${!TC_*} IP_DROP MAC_DROP WATCH_IP \
               GLOBAL DHCP_SERVER DYNAMIC ECN_REMOVE;  do
                 unset $VAR
    done

    local ZONE_FILE=CONF_${ZONE}
    ZONE_FILE=${!ZONE_FILE}
    local SOURCE_FILE=${CONF_DIR}/${ZONE_FILE}

    # First check that the syntax is ok.
    
    if [[ -f ${SOURCE_FILE} ]]; then       
	source ${SOURCE_FILE}
    else
	if [[ -z "${ZONE_FILE}" ]]; then
	    echo "CONF_${ZONE} variable not set or empty."
	    debug_out "CONF_${ZONE} variable not set or empty."
	else
	    echo "Zonefile not found: ${SOURCE_FILE}"
	    debug_out "Zonefile not found: ${SOURCE_FILE}"
	fi
	let FILE_ERRORS++
	return 1
    fi

    # Accept the old PUBLIC variable.
    if [[ -z "${GLOBAL}" && -n "${PUBLIC}" ]]; then
	GLOBAL=${PUBLIC}
	unset PUBLIC
    fi

    # Accept the old STATIC 
    if [[ -z "${DYNAMIC}" && -n "${STATIC}" ]]; then 
	if (( STATIC == 1 )); then
	    DYNAMIC=0
	else
	    DYNAMIC=1
	fi
	unset STATIC
    fi

    # Default values.
    declare -i GLOBAL=${GLOBAL:=1}
    declare -i DHCP_SERVER=${DHCP_SERVER:=0}
    declare -i DYNAMIC=${DYNAMIC:=1}
    declare -i TC_ENABLE=${TC_ENABLE:=0}
    declare    TC_TYPE=${TC_TYPE:=CBQ}

    # Got all nessesary information about the interface?
    if (( DYNAMIC == 0 )); then	
        # Test that important settings are available.
	if [[ -z "${IP}" ]]; then
	    echo -e "\n${ZONE_FILE}: IP not defined.\n"
	    debug_out "${ZONE_FILE}: IP not defined."
	    return 1
	fi
	if [[ -z "${MASK}" ]]; then
	    echo -e "\n${ZONE_FILE}: MASK not defined.\n"
	    debug_out "${ZONE_FILE}: MASK not defined."
	    return 1
	fi
	if [[ -z "${NET}" ]]; then
	    echo -e "\n${ZONE_FILE}: NET not defined.\n"
	    debug_out "${ZONE_FILE}: NET not defined."
	    return 1
	fi
	if [[ -z "${BCAST}" ]]; then
	    echo -e "\n${ZONE_FILE}: BCAST not defined.\n"
	    debug_out "${ZONE_FILE}: BCAST not defined."
	    return 1
	fi
    fi
}
###############################################################################
# add_ip_drop
###############################################################################
function add_ip_drop ()
{
    local ZONE=$1
    local IP=$2
    # Resolve aliases.
    get_alias ${IP}	
    local IPS=${RESULT}
    if [ -z "${IPS}" ]; then
	IPS=$IP
    fi
    for IP in ${IPS}; do
	IPTABLES -A INPUT_${ZONE}   -s ${IP} -j LOG_DROP
	IPTABLES -A OUTPUT_${ZONE}  -d ${IP} -j LOG_DROP
	IPTABLES -A FORWARD_${ZONE} -d ${IP} -j LOG_DROP   
	IPTABLES -A SEND_${ZONE}    -s ${IP} -j LOG_DROP
    done
}

###############################################################################
# add_mac_drop
###############################################################################
function add_mac_drop ()
{
    local ZONE=$1
    local MAC=$2
    IPTABLES -A INPUT_${ZONE} -m mac --mac-source ${MAC} -j LOG_DROP
    IPTABLES -A SEND_${ZONE}  -m mac --mac-source ${MAC} -j LOG_DROP   
}

###############################################################################
# add_watch_ip
###############################################################################
function add_watch_ip ()
{
    local ZONE=$1
    local IP=$2
     # Resolve aliases.
    get_alias ${IP}
    local IPS=${RESULT}
    if [ -z "${IPS}" ]; then
        IPS=$IP
    fi  
    for IP in ${IPS}; do
	IPTABLES -A INPUT_${ZONE}   -s ${IP} -j LOG_WATCH
	IPTABLES -A OUTPUT_${ZONE}  -d ${IP} -j LOG_WATCH
	IPTABLES -A FORWARD_${ZONE} -d ${IP} -j LOG_WATCH   
	IPTABLES -A SEND_${ZONE}    -s ${IP} -j LOG_WATCH
    done
}

###############################################################################
# remove_ecn (remove the ECN-bit)
###############################################################################
function remove_ecn ()
{
    local CHAIN=$1
    local IP=$2
    get_alias ${IP}
    local IPS=${RESULT}
    if [ -z "${IPS}" ]; then
        IPS=$IP
    fi  
    for IP in ${IPS}; do
	IPTABLES -t mangle -A ${CHAIN} -p tcp -d ${IP} -j ECN --ecn-tcp-remove
    done
}

###############################################################################
# Add reply with
###############################################################################
function add_reply_with ()
{
    local RULEID=$1
    local ZONE=$2
    local TYPE=$3
    local PROTOCOL=$4
    shift 4
    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp|ICMP|icmp) ]]; then
	PORTS=$1
	shift
    fi
    local IP=$1
    shift
    
    if ! check_zone ${ZONE} "Zone ${ZONE} is not defined in REPLY rule."; then 
	let ZONE_ERRORS++
	return
    fi
        
    local PARAM="-j REJECT --reject-with ${TYPE}"
    
    case ${ZONE} in
	all|ALL)
	    add_rule_protocol_ip_port "filter" INPUT_${NAME} \
		${PROTOCOL} ${PORTS} ${IP} ${PARAM}
	    add_rule_protocol_ip_port "filter" FORWARD_${NAME} \
		${PROTOCOL} ${PORTS} ${IP} ${PARAM}
	    ;;
	${NAME})
	    add_rule_protocol_ip_port "filter" INPUT_${NAME} \
		${PROTOCOL} ${PORTS} ${IP} ${PARAM}
	    ;;
	*)
	    IPTABLES -N ${RULE_ID}_${NAME}
	    jump_zone ${ZONE} "SRC" "filter" FORWARD_${NAME} ${RULE_ID}_${NAME}
	    add_rule_protocol_ip_port "filter" ${FORWARD_CHAIN} \
		${PROTOCOL} ${PORTS} ${IP} ${PARAM} 

    esac
}

###############################################################################
# Limit packet types.
###############################################################################
function add_limit ()
{
    # LIMIT_XXX = <chain> <new_chain> <zone> <policy> <limit> <burst> [protocol [port[,port]*]] 
    local CHAIN=$1
    local NEW_CHAIN=$2
    local ZONE=$3
    local POLICY=$4
    local LIMIT=$5
    local BURST=$6
    local PROTOCOL=$7
    shift 7
    local PORTS=""
    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp|ICMP|icmp) ]]; then
	PORTS=$1
	shift
    fi
    local IP=$1
    shift

    if [[ "${IP}" == @(ALL|all) ]]; then 
	print_err ""
	print_err "Error in rules."
	print_err "It seems that a port was specified with protocol 'ALL'."
	print_err "Please recheck INPUT, OUTPUT, FORWARD, MARK and LIMIT rules".
	let RULE_ERRORS++
	IP=$1
	shift
    fi

    if ! check_zone ${ZONE} "Zone ${ZONE} is not defined in LIMIT rule."; then 
	let ZONE_ERRORS++
	return
    fi

    local LIMIT="-m limit --limit ${LIMIT} --limit-burst ${BURST}" 
    IPTABLES -N ${NEW_CHAIN} 
    IPTABLES -A ${NEW_CHAIN} ${LIMIT} -j RETURN
    IPTABLES -A ${NEW_CHAIN} -j LOG_LIMIT_${POLICY}
    add_rule_protocol_ip_port "filter" ${CHAIN} ${PROTOCOL} ${PORTS} ${IP} \
	-j ${NEW_CHAIN}
}

###############################################################################
# Mark packets.
# <chain_id> <chain_pre> <chain_out> <zone|ALL> <mark> <protocol|all> <ports|all> <ipmask=>ipmask>
###############################################################################
function add_mark ()
{
    local RULE_ID=$1
    local CHAIN_PRE=$2
    local CHAIN_OUT=$3
    local SRC_ZONE=$4
    local MARK=$5
    local PROTOCOL=$6
    shift 6
    local PORTS=""
    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp|ICMP|icmp) ]]; then
	PORTS=$1
	shift
    fi
    local IP=$1
    shift

    if ! check_zone ${ZONE} "Zone ${ZONE} is not defined in MARK rule."; then 
	let ZONE_ERRORS++
	return
    fi

    local PARAM 
    PARAM="-j MARK --set-mark ${MARK}" 
        
    # Get the device
    case ${SRC_ZONE} in
	ALL|all)
	    add_rule_protocol_ip_port "mangle" ${CHAIN_PRE} \
		${PROTOCOL} ${PORTS} ${IP} ${PARAM} 
	    add_rule_protocol_ip_port "mangle" ${CHAIN_OUT} \
		${PROTOCOL} ${PORTS} ${IP} ${PARAM} 

	    ;;
	${NAME})
	    add_rule_protocol_ip_port "mangle" ${CHAIN_OUT} \
		${PROTOCOL} ${PORTS} ${IP} ${PARAM} 
	    ;;
	*)
	    IPTABLES -t mangle -N MARK_${RULE_ID}
	    jump_zone ${ZONE} "SRC" "mangle" ${CHAIN_PRE} MARK_${RULE_ID}
	    add_rule_protocol_ip_port "mangle" MARK_${RULE_ID} \
		${PROTOCOL} ${PORTS} ${IP} ${PARAM} 
	    ;;
    esac
	    
}


###############################################################################
# Add redirect rules.
###############################################################################
function add_redirect ()
{
    local CHAIN=$1
    local SNAT_CHAIN=$2
    local FORWARD_CHAIN=$3
    local RULE_ID=$4
    local PROTOCOL=$5
    shift 5
    local PORTS=""
    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp|ICMP|icmp) ]]; then
	PORTS=$1
	shift
    fi
    local SRC_IP=$1
    local DNAT_IPS=$2
    shift 2
    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp|ICMP|icmp) ]]; then
	local DNAT_PORT=$1
	shift
    fi
    
    # If DNAT_PORT is an alias, then translate it into a port number.
    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp) && 
	  ${DNAT_PORT} != *([0-9]) ]]; then
	get_service_port ${PROTOCOL} ${DNAT_PORT}
	if [[ -n ${RESULT} ]]; then
	    DNAT_PORT=${RESULT}
	fi
    fi
    
    # Allow use of aliases in DNAT_IPS
    get_alias ${DNAT_IPS}
    if [[ -n ${RESULT} ]]; then
	DNAT_IPS=${RESULT/ /,}
    fi

    local PARAM=""

    local REDIRECT
    # Test if destination is localhost.
    if [[ ${DNAT_IPS} == "127.0.0.1" || ${DNAT_IPS} == "localhost" ]]; then
	REDIRECT="-j REDIRECT"
	# Tests if there are ports involved
	if [[ -n ${DNAT_PORT}  ]]; then 
	    REDIRECT="${REDIRECT} --to-ports ${DNAT_PORT}"
	fi
    else
	local DNAT_IP
	REDIRECT="-j DNAT"
	for DNAT_IP in ${DNAT_IPS//,/ }; do
	    REDIRECT="${REDIRECT} --to-destination ${DNAT_IP}"
	    if [[ -n ${DNAT_PORT} ]]; then
		REDIRECT="${REDIRECT}:${DNAT_PORT}"
	    fi
	    # Test if a SNAT is needed to route back to the same network.
	    if (( DYNAMIC == 0 && GLOBAL == 0 )); then
	        # Need to examine all networks - A zone can have NET_EXTRA
		local NETWORKS=${ZONE}_NETS
		NETWORKS=${!NETWORKS}
		local NET 
		local RET
		for NET in ${NETWORKS}; do
		    if ip_in_network ${NET%/*} ${NET#*/} ${DNAT_IP}; then
			RET=1
			break;
		    fi
		done
		if (( RET == 1 )); then
		    # Add SNAT for packets going back to the zone.	    
		    local SNAT_ADDR=${NAME}_IP
		    SNAT_ADDR=${!SNAT_ADDR}
		    
		    local NEW_SNAT_CHAIN=${RULE_ID}_${NAME}
		    local NEW_FORWARD_CHAIN=${RULE_ID}_${NAME}
		    IPTABLES -t nat -N ${NEW_SNAT_CHAIN}
		    add_rule_protocol_ip_port "nat" ${NEW_SNAT_CHAIN} \
			${PROTOCOL} ${DNAT_PORT} "0.0.0.0/0=>${DNAT_IP}" \
			-j SNAT --to-source ${SNAT_ADDR}
		    
		    IPTABLES -N ${NEW_FORWARD_CHAIN}
		    add_rule_protocol_ip_port "filter" ${NEW_FORWARD_CHAIN} \
			${PROTOCOL} ${DNAT_PORT} "0.0.0.0/0=>${DNAT_IP}" \
			-i ${DEV} -j ACCEPT
			
		    
		    for NET in ${NETWORKS}; do
			IPTABLES -t nat -A ${SNAT_CHAIN} -s ${NET} \
			    -j ${NEW_SNAT_CHAIN}
			IPTABLES        -A ${FORWARD_CHAIN} -s ${NET} \
			    -j ${NEW_FORWARD_CHAIN}
		    done
		fi
	    fi
	done
    fi
    add_rule_protocol_ip_port "nat" ${CHAIN} \
	${PROTOCOL} ${PORTS} ${SRC_IP} ${REDIRECT}

}

###############################################################################
# add_snat: <zone> [ports [ip[/mask]]]
###############################################################################

function add_snat ()
{
    local RULE_ID=$1
    local SNAT_ZONE=$2

    local PROTOCOL=$3
    shift 3
    local PORTS=""
    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp|ICMP|icmp) ]]; then
	PORTS=$1
	shift
    fi
    local IP=$1
  
    local NAT_TYPE=""
    local PARAM=""
    local PARAM_PORTS=""
    local SOURCE=""
    local SNAT_IPS=""
   
    if check_ip ${SNAT_ZONE}; then
    	SNAT_IPS=${SNAT_ZONE}
    	SNAT_ZONE=""
     	# Find the destination zone
    	local AZONE 
    	local ZONE_IPS
    	local ZONE_IP
    	for AZONE in ${ZONES}; do
    	    ZONE_IPS=${AZONE}_IPS
    	    ZONE_IPS=${!ZONE_IPS}
    	    for ZONE_IP in ${ZONE_IPS}; do
    		if [[ "${ZONE_IP}" == "${SNAT_IPS}" ]]; then
    		    SNAT_ZONE=${AZONE}
    		    break
    		fi
    	    done
    	    if [[ -n "${SNAT_ZONE}" ]]; then
    		break
    	    fi
    	done
    	if [[ -z "${SNAT_ZONE}" ]]; then
    	    print_err "SNAT rule error: Unable to locate destnation zone for ip ${SNAT_IPS}"
    	    print_err "The specified IP must be specified in another zone under IP or IP_EXTRA"
    	    let ZONE_ERRORS++
    	    return 1
    	fi
    else
	if check_zone ${SNAT_ZONE} "Zone ${ZONE} is not defined in SNAT rule."; then 
	    local ZONE_IPS=${SNAT_ZONE}_IPS
	    SNAT_IPS=${!ZONE_IPS}
	else
	    let ZONE_ERRORS++
	    return 1
	fi
    fi

    if [[ "${SNAT_ZONE}" == ${NAME} ]]; then
	print_err "SNAT rule error: Cannot SNAT back into the same zone as packets originates"
	let ZONE_ERRORS++
        return 1
    fi

    local SNAT_CHAIN=POSTROUTING_NAT_${SNAT_ZONE}
    local ZONE_GLOBAL=${ZONE}_GLOBAL
    local ZONE_DYNAMIC=${ZONE}_DYNAMIC
    local SNAT_DYNAMIC=${SNAT_ZONE}_DYNAMIC

    ZONE_GLOBAL=${!ZONE_GLOBAL}
    ZONE_DYNAMIC=${!ZONE_DYNAMIC}
    SNAT_DYNAMIC=${!SNAT_DYNAMIC}

    local NEW_SNAT_CHAIN=SNAT_${NAME}_${RULE_ID}
    IPTABLES -t nat -N ${NEW_SNAT_CHAIN}

    local NETWORKS=${NAME}_NETS
    NETWORKS=${!NETWORKS}
    local NETWORK
    if (( ZONE_GLOBAL == 0 && ZONE_DYNAMIC == 0 )); then
	for NETWORK in ${NETWORKS}; do
	    IPTABLES -t nat -A ${SNAT_CHAIN} -s ${NETWORK} -j ${NEW_SNAT_CHAIN}
	done
    else
	# No way to test. SNAT all
	IPTABLES -t nat -A ${SNAT_CHAIN} -j ${NEW_SNAT_CHAIN}
    fi
	
    
    local SNAT_PARAM=""
    if (( SNAT_DYNAMIC == 0 )); then
	NAT_TYPE="SNAT"
	local SNAT_IP
	for SNAT_IP in ${SNAT_IPS}; do
	    SNAT_PARAM="${SNAT_PARAM} --to-source ${SNAT_IP}"
	done
    else
	NAT_TYPE="MASQUERADE"
    fi

    add_rule_protocol_ip_port "nat" ${NEW_SNAT_CHAIN} \
	${PROTOCOL} ${PORTS} ${IP} -j ${NAT_TYPE} ${SNAT_PARAM}
}

###############################################################################
# Add deafult rules for in/out.
###############################################################################
function add_zone_rule()
{	
    local CHAIN=$1
    local POLICY=$2
    local PROTOCOL=$3
    shift 3
    local PORTS=""
    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp|ICMP|icmp) ]]; then
	PORTS=$1
	shift
    fi
    local IP=$1
    shift

    if [[ "${IP}" == @(ALL|all) ]]; then 
	print_err ""
	print_err "Error in rules."
	print_err "It seems that a port was specified with protocol 'ALL'."
	print_err "Please recheck INPUT, OUTPUT, FORWARD, MARK and LIMIT rules".
	let RULE_ERRORS++
	IP=$1
	shift
    fi

    # Actual function
    add_rule_protocol_ip_port "filter" ${CHAIN} \
	${PROTOCOL} ${PORTS} ${IP} -j LOG_${POLICY}
}    

###############################################################################
# add_zone_forward: chain zone protocol ports policy
###############################################################################
function add_zone_forward ()
{
    local RULE_ID=$1
    local CHAIN=$2
    local ZONE=$3
    local POLICY=$4
    local PROTOCOL=$5
    shift 5
    local PORTS=""
    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp|ICMP|icmp) ]]; then
	PORTS=$1
	shift
    fi
    local IP=$1
    shift

    if ! check_zone ${ZONE} "Zone ${ZONE} is not defined in FORWARD rule."; then 
	let ZONE_ERRORS++
	return
    fi
  
    case ${ZONE} in 
	ALL|all)
	    add_rule_protocol_ip_port "filter" ${CHAIN} \
		${PROTOCOL} ${PORTS} ${IP} -j LOG_${POLICY}
	    ;;
	*)
	    IPTABLES -N FORWARD_${RULE_ID}
	    jump_zone ${ZONE} "SRC" "filter" ${CHAIN} FORWARD_${RULE_ID}
	    add_rule_protocol_ip_port "filter" FORWARD_${RULE_ID} \
		${PROTOCOL} ${PORTS} ${IP} -j LOG_${POLICY}
	    ;;
    esac
}

###############################################################################
# Setup Zone chains.
# Give zone name as argument
###############################################################################
function zone_chains ()
{
    local ZONE=$1
    debug_out "Creating global chains for zone: ${ZONE}"
 
    # FILTER chains
    for CHAIN in ${BUILT_IN_CHAINS_filter} SEND; do
	IPTABLES -N ${CHAIN}_${ZONE}
    done
    for CHAIN in ${BUILT_IN_CHAINS_mangle}; do
	IPTABLES -t mangle -N ${CHAIN}_MANGLE_${ZONE}
    done

    for CHAIN in ${BUILT_IN_CHAINS_nat}; do
	IPTABLES -t nat -N ${CHAIN}_NAT_${ZONE}
    done
    
}

###############################################################################
# Setup zone specific rules.
###############################################################################
function configure_zone ()
{
    local ZONE=$1
    read_zone ${ZONE}

    # Stop processing if the device is not available due 
    # to a configuration error.
    local ZONE_AVAILABLE=${ZONE}_AVAILABLE
    ZONE_AVAILABLE=${!ZONE_AVAILABLE}
    if (( ZONE_AVAILABLE == 0 )); then
	return
    fi
    debug_out ""
    debug_out ""
    debug_out "Configuring zone: ${NAME}"
    debug_out "DEV=${DEV}, DYNAMIC=${DYNAMIC}, GLOBAL=${GLOBAL}"
    if (( DYNAMIC == 0 )); then
	debug_out "IP=${IP}, MASK=${MASK}, NET=${NET}, BCAST=${BCAST}"
    fi
    debug_out ""
    debug_out "Creating zone chains"

    local PRE_NAT_CHAIN=PREROUTING_NAT_${NAME}
    local POST_NAT_CHAIN=POSTROUTING_NAT_${NAME}

    local PRE_MANGLE_CHAIN=PREROUTING_MANGLE_${NAME}
    local POST_MANGLE_CHAIN=POSTROUTING_MANGLE_${NAME}
    local OUTPUT_MANGLE_CHAIN=OUTPUT_MANGLE_${NAME}

    # Allow server to answer DHCP queries.
    debug_out "DHCP_SERVER=${DHCP_SERVER}"
    if (( DHCP_SERVER == 1 )); then 
	IPTABLES -A DEV_${DEV}_SRC -p UDP -d 255.255.255.255 \
	    --dport bootps -j LOG_ACCEPT
	IPTABLES -A DEV_${DEV}_DST -p UDP -d 255.255.255.255 \
	    --dport bootpc -j LOG_ACCEPT 
    fi
    
    # Allow server to send DHCP queries.
    debug_out "DYNAMIC=${DYNAMIC}"
    if (( DYNAMIC == 1 )); then 
	IPTABLES -A DEV_${DEV}_SRC -p UDP -d 255.255.255.255 \
	    --dport bootpc -j LOG_ACCEPT
	IPTABLES -A DEV_${DEV}_DST -p UDP -d 255.255.255.255 \
	    --dport bootps -j LOG_ACCEPT 
    fi
 
    # Forward packets to queues.    
    # Only allow packets that can originate from the device. 
    # This way we can have more than one zone on one interface
    # - Unless its global, then there can be only one zone per interface.

    debug_out "GLOBAL=${GLOBAL}"

    local CHAIN_INPUT=INPUT_${NAME}
    local CHAIN_OUTPUT=OUTPUT_${NAME}
    local CHAIN_FORWARD=FORWARD_${NAME}
    local CHAIN_SEND=SEND_${NAME}

    # Both GLOBAL and NONGLOBAL have net and net_extra specified. 
    if (( DYNAMIC == 0 )); then
	local NETWORK
	for NETWORK in ${NET} ${NET_EXTRA}; do
	    IPTABLES -A DEV_${DEV}_DST -d ${NETWORK} -j RETURN
	    IPTABLES -A DEV_${DEV}_SRC -s ${NETWORK} -j RETURN
	done
    fi


    if (( GLOBAL == 1 || DYNAMIC == 1 )); then	
	
	IPTABLES -t nat -A PREROUTING -i ${DEV} -j ${PRE_NAT_CHAIN}
	IPTABLES -t nat -A POSTROUTING -o ${DEV} -j ${POST_NAT_CHAIN}
	
	IPTABLES -t mangle -A PREROUTING -i ${DEV} -j ${PRE_MANGLE_CHAIN}
	IPTABLES -t mangle -A POSTROUTING -o ${DEV} -j ${POST_MANGLE_CHAIN}
	IPTABLES -t mangle -A OUTPUT -o ${DEV} -j ${OUTPUT_MANGLE_CHAIN}

	if (( DYNAMIC == 0 )); then
	    IPTABLES -A DEV_${DEV}_DST -j PRIVATE_DST
	    IPTABLES -A DEV_${DEV}_SRC -j PRIVATE_SRC
	fi

	IPTABLES -A DEV_${DEV}_DST -j RESERVED_DST
	IPTABLES -A DEV_${DEV}_DST -j RETURN	

	IPTABLES -A DEV_${DEV}_SRC -j RESERVED_SRC
	IPTABLES -A DEV_${DEV}_SRC -j RETURN	

	# Jump to corect chains.
	IPTABLES -A INPUT_NEW_${DEV} -j ${CHAIN_INPUT}
	IPTABLES -A OUTPUT_NEW_${DEV} -j ${CHAIN_OUTPUT}
	IPTABLES -A FORWARD_NEW_${DEV} -j ${CHAIN_FORWARD}
	IPTABLES -A SEND_NEW_${DEV} -j ${CHAIN_SEND}

    else
	for NETWORK in ${NET} ${NET_EXTRA}; do

	    IPTABLES -t nat -A PREROUTING -i ${DEV} -s ${NETWORK} \
		-j ${PRE_NAT_CHAIN}
	    IPTABLES -t nat -A POSTROUTING -o ${DEV} -d ${NETWORK} \
		-j ${POST_NAT_CHAIN}

	    IPTABLES -t mangle -A PREROUTING -i ${DEV} -s ${NETWORK} \
		-j ${PRE_MANGLE_CHAIN}
	    IPTABLES -t mangle -A POSTROUTING -o ${DEV} -d ${NETWORK} \
		-j ${POST_MANGLE_CHAIN}
	    IPTABLES -t mangle -A OUTPUT -o ${DEV} -d ${NETWORK} \
		-j ${OUTPUT_MANGLE_CHAIN}

	    IPTABLES -A INPUT_NEW_${DEV} -s ${NETWORK} -j ${CHAIN_INPUT}
	    IPTABLES -A OUTPUT_NEW_${DEV} -d ${NETWORK} -j ${CHAIN_OUTPUT}
	    IPTABLES -A FORWARD_NEW_${DEV} -d ${NETWORK} -j ${CHAIN_FORWARD}
	    IPTABLES -A SEND_NEW_${DEV} -s ${NETWORK} -j ${CHAIN_SEND}
	done
    fi


    # Setup redirects. These are added to the NAT chain.
    local REDIRECT
    for REDIRECT in ${!REDIRECT_*}; do
	debug_out "${REDIRECT}=${!REDIRECT}"
	add_redirect ${PRE_NAT_CHAIN} ${POST_NAT_CHAIN} ${CHAIN_FORWARD} \
	    ${REDIRECT} ${!REDIRECT} 
    done   
    
    local I
    for ((I=0;I<${#MARK[*]};I++)); do
 	debug_out "MARK[${I}]=${MARK[I]}"   
	add_mark ${ZONE}${I} ${PRE_MANGLE_CHAIN} ${OUTPUT_MANGLE_CHAIN} \
	    ${MARK[I]}
    done   

    local VAR

    #Setup Watch rules:
    debug_out "WATCH_IP: ${WATCH_IP}"
    if [[ -f "${CONF_DIR}/${WATCH_IP}" ]]; then
	cat ${CONF_DIR}/${WATCH_IP} | cut -d"#" -f1 | while read VAR; do
	    if [[ -z ${VAR} ]]; then
		continue
	    fi
	    add_watch_ip ${NAME} ${VAR}
	done
    else
	for VAR in ${WATCH_IP}; do
	    add_watch_ip ${NAME} ${VAR}
	done
    fi

    #Setup ECN_REMOVE rules:
    debug_out "ECN_REMOVE: ${ECN_REMOVE}"
    if [[ -f "${CONF_DIR}/${ECN_REMOVE}" ]]; then
	cat ${CONF_DIR}/${ECN_REMOVE} | cut -d"#" -f1 | while read VAR; do
	    if [[ -z ${VAR} ]]; then
		continue
	    fi
	    remove_ecn ${POST_MANGLE_CHAIN} ${VAR}
	    remove_ecn ${OUTPUT_MANGLE_CHAIN} ${VAR}
	done
    else
	for VAR in ${ECN_REMOVE}; do
	    remove_ecn ${POST_MANGLE_CHAIN} ${VAR}
	    remove_ecn ${OUTPUT_MANGLE_CHAIN} ${VAR}
	done
    fi

    # Setup drop rules
    debug_out "MAC_DROP: ${MAC_DROP}"
    if [[ -f "${CONF_DIR}/${MAC_DROP}" ]]; then
	cat ${CONF_DIR}/${MAC_DROP} | cut -d"#" -f1 | while read VAR; do
	    if [[ -z ${VAR} ]]; then
		continue
	    fi
	    add_mac_drop ${NAME} ${VAR}
	done
    else
	for VAR in ${MAC_DROP}; do
	    add_mac_drop ${NAME} ${VAR}
	done
    fi

    debug_out "IP_DROP: ${IP_DROP}"
    if [[ -f "${CONF_DIR}/${IP_DROP}" ]]; then
	cat ${CONF_DIR}/${IP_DROP} | cut -d"#" -f1 | while read VAR; do
	    if [[ -z ${VAR} ]]; then
		continue
	    fi
	    add_ip_drop ${NAME} ${VAR}
	done
    else
	for VAR in ${IP_DROP}; do
	    add_ip_drop ${NAME} ${VAR}
	done
    fi
    
    # Source NAT are done in the NAT table - postrouting.
    for (( I=0;I<${#SNAT[*]};I++ )); do
	debug_out "SNAT[${I}]=${SNAT[I]}"
	add_snat ${I} ${SNAT[I]}
    done
       
    # Setup reply_with
    local REPLY
    for REPLY in ${!REPLY_*}; do
	debug_out "${REPLY}=${!REPLY}"
	add_reply_with ${REPLY} ${!REPLY}
    done

    # Limit number of packets.
    local LIMIT
    for LIMIT in ${!LIMIT_*}; do
	debug_out "${LIMIT}=${!LIMIT}"
	if [[ "${!LIMIT%% *}" = "${NAME}" ]]; then
	    add_limit ${CHAIN_INPUT} ${LIMIT}_${ZONE} ${!LIMIT}
	else
	    add_limit ${CHAIN_FORWARD} ${LIMIT}_${ZONE} ${!LIMIT}
	fi
    done

    # Create rules for INPUT.
    if (( DHCP_SERVER == 1 )); then
	# Allow renewal of DHCP leases.
        IPTABLES -A ${CHAIN_INPUT} -p udp \
	    --sport bootpc --dport bootps -j ACCEPT
    fi

    # Add user chains
    local CHAIN
    local FIAIF_CHAIN
    for CHAIN in INPUT OUTPUT FORWARD; do
	IPTABLES -N USER_${CHAIN}_${NAME}
	FIAIF_CHAIN=CHAIN_${CHAIN}
	IPTABLES -I ${!FIAIF_CHAIN} -j USER_${CHAIN}_${NAME}
    done

    for (( I=0;I<${#INPUT[*]};I++ )); do
 	debug_out "INPUT[${I}]=${INPUT[I]}"   
	add_zone_rule ${CHAIN_INPUT} ${INPUT[I]}
    done

    # Create rules for OUTPUT.
    for (( I=0;I<${#OUTPUT[*]};I++ )); do
 	debug_out "OUTPUT[${I}]=${OUTPUT[I]}"   
	add_zone_rule ${CHAIN_OUTPUT} ${OUTPUT[I]}
    done
    
    # Create rules for FORWARD.
    for (( I=0;I<${#FORWARD[*]};I++ )); do
 	debug_out "FORWARD[${I}]=${FORWARD[I]}"   
	add_zone_forward ${ZONE}${I} ${CHAIN_FORWARD} ${FORWARD[I]}
    done

    debug_out "Log all unmatched packets in this zone"
    IPTABLES -A ${CHAIN_INPUT} -j LOG_MISS_${NAME}
    IPTABLES -A ${CHAIN_OUTPUT} -j LOG_MISS_${NAME}
    IPTABLES -A ${CHAIN_FORWARD} -j LOG_MISS_${NAME}

    debug_out "Done configuring zone"
}

