/***************************************************************************
                          FLSqlQuery.h  -  description
                             -------------------
    begin                : sb jun 22 2002
    copyright            : (C) 2002 by Federico Albujer Zornoza
    email                : mail@infosial.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef FLSQLQUERY_H
#define FLSQLQUERY_H

#include <qsqlquery.h>
#include <qvariant.h>
#include <qdict.h>
#include <qregexp.h>
#include <qstringlist.h>
#include <qinputdialog.h>

class FLParameterQuery;
class FLGroupByQuery;

/** Maneja consultas con caractersticas especficas para FacturaLUX, hereda de QSqlQuery.

    Ofrece la funcionalidad para manejar consultas de QSqlQuery y adems ofrece mtodos
    para trabajar con consultas parametrizadas y niveles de agrupamiento.

    @author Federico Albujer Zornoza
    @version 0.4  */
class FLSqlQuery : public QObject, public QSqlQuery
{
    Q_OBJECT

    public:

    /** constructor */	
    FLSqlQuery( );

    /** destructor */
    ~FLSqlQuery();

    /** Ejecuta la consulta */
    bool exec( );

    /** Aade la descripcin parmetro al diccionario de parmetros.

        @param  p   Objeto FLParameterQuery con la descripcin del parmetro a aadir */
    void addParameter(const FLParameterQuery * p);

    /** Aade la descripcin de un grupo al diccionario de grupos.

        @param  g   Objeto FLGroupByQuery con la descripcin del grupo a aadir */
    void addGroup (const FLGroupByQuery * g);

    /** Tipo de datos diccionario de parametros */
    typedef QDict<FLParameterQuery> FLParameterQueryDict;

    /** Tipo de datos diccionaro de grupos */
    typedef QDict<FLGroupByQuery> FLGroupByQueryDict;

    /** Para establecer el nombre de la consulta.

        @param n Nombre de la consulta */
    void setName( const QString & n)
    {
        name_ = n;
    }

    /** Para obtener el nombre de la consulta */
    const QString & name() const
    {
        return name_;
    }

    /** Para obtener la parte SELECT de la sentencia SQL de la consulta */
    const QString & select() const
    {
        return select_;
    }

    /** Para obtener la parte FROM de la sentencia SQL de la consulta */
    const QString & from() const
    {
        return from_;
    }

    /** Para obtener la parte WHERE de la sentencia SQL de la consulta */
    const QString & where() const
    {
        return where_;
    }

    /** Para establecer la parte SELECT de la sentencia SQL de la consulta.

        @param s Cadena de texto con la parte SELECT de la sentencia SQL que
                         genera la consulta. Esta cadena NO debe incluir la palabra reservada
                         SELECT, ni tampoco el caracter '*' como comodn. Solo admite la lista
                         de campos que deben aparecer en la consulta separados por comas */
    void setSelect( const QString & s );

    /** Para establecer la parte FROM de la sentencia SQL de la consulta.

        @param f Cadena de texto con la parte FROM de la sentencia SQL que
                         genera la consulta. */
    void setFrom( const QString & f )
    {
        from_ = f.stripWhiteSpace ();
        from_ = from_.simplifyWhiteSpace ();
    }

    /** Para establecer la parte WHERE de la sentencia SQL de la consulta.

        @param s Cadena de texto con la parte WHERE de la sentencia SQL que
                         genera la consulta. */
    void setWhere( const QString & w )
    {
        where_ = w.stripWhiteSpace ();
        where_ = where_.simplifyWhiteSpace ();
    }

    /** Para obtener la sentencia completa SQL de la consulta.

        Este mtodo une las tres partes de la consulta (SELECT, FROM Y WHERE),
        sustituye los parmetros por el valor que tienen en el diccionario y devuelve
        todo en una cadena de texto.

        @return Cadena de texto con la sentencia completa SQL que genera la consulta */
    const QString sql() const;

    /** Para obtener los parametros de la consulta.

        @return Diccionario de parmetros */
    FLParameterQueryDict  * parameterDict() const
    {
        return parameterDict_;
    }

    /** Para obtener los niveles de agrupamiento de la consulta.

        @return Diccionario de niveles de agrupamiento */
    FLGroupByQueryDict * groupDict() const
    {
        return groupDict_;
    }

    /** Para obtener la lista de nombres de los campos.

        @return Lista de cadenas de texto con los nombres de los cmapos de la
                        consulta. */
    QStringList * fieldList() const
    {
        return fieldList_;
    }

    /** Asigna un diccionario de parmetros, al diccionario de parmetros de la consulta.

        El diccionario de parmetros del tipo FLGroupByQueryDict , ya construido,
        es asignado como el nuevo diccionario de grupos de la consulta, en el caso de que
        ya exista un diccionario de grupos, este es destruido y sobreescrito por el nuevo.
        El diccionario pasado a este mtodo pasa a ser propiedad de la consulta, y ella es la
        encargada de borrarlo. Si el diccionario que se pretende asignar es nulo o vaco este
        mtodo no hace nada.

         @param gd  Diccionario de parmetros */
    void setGroupDict (FLGroupByQueryDict * gd);

    /** Asigna un diccionario de grupos, al diccionario de grupos de la consulta.

        El diccionario de grupos del tipo FLParameterQueryDict , ya construido,
        es asignado como el nuevo diccionario de parmetros de la consulta, en el caso de que
        ya exista un diccionario de parmetros, este es destruido y sobreescrito por el nuevo.
        El diccionario pasado a este mtodo pasa a ser propiedad de la consulta, y ella es la
        encargada de borrarlo. Si el diccionario que se pretende asignar es nulo o vaco este
        mtodo no hace nada.

         @param pd  Diccionario de parmetros */
    void setParameterDict (FLParameterQueryDict * pd);

    /** Este mtodo muestra el contenido de la consulta, por la slida estndar.

        Est pensado slo para tareas de depuracin */
    void showDebug();

    /** Obtiene el valor de un campo de la consulta.

        Dado un nombre de un campo de la consulta, este mtodo devuelve un objeto QVariant
        con el valor de dicho campo. El nombre debe corresponder
        con el que se coloco en la parte SELECT de la sentenica SQL de la consulta.

        @param  n Nombre del campo de la consulta. */
    QVariant value(const QString & n);

    /** Reimplementacion de QSqlQuery::value(int) */
    QVariant value(int i) const;

    /** Devuelve el nombre de campo, dada su posicion en la consulta.

        @param  p   Posicion del campo en la consulta, empieza en cero y de izquierda
                            a derecha
        @return Nombre del campo correspondiente. Si no existe el campo devuelve
                      QString::null */
    QString posToFieldName(const int p);

    /** Devuelve la posicin de una campo en la consulta, dado su nombre.

        @param  n  Nombre del campo
        @return Posicion del campo en la consulta. Si no existe el campo devuelve -1*/
    int fieldNameToPos(const QString & n);

    private:

    /** Nombre de la consulta */
    QString name_;

    /** Parte SELECT de la consulta */
    QString select_;

    /** Parte FROM de la consulta */
    QString from_;

    /** Parte WHERE de la consulta */
    QString where_;

    /** Lista de nombres de los campos */
    QStringList * fieldList_;

    /** Lista de parmetros */
    FLParameterQueryDict * parameterDict_;

    /** Lista de grupos */
    FLGroupByQueryDict * groupDict_;
};

#endif
