/*
 *
 *   (C) Copyright IBM Corp. 2001, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: activation.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <glob.h>
#include <pthread.h>
#include <unistd.h>
#include <sys/stat.h>

#include "fullengine.h"
#include "engine.h"
#include "lists.h"
#include "handlemgr.h"
#include "internalAPI.h"
#include "memman.h"
#include "message.h"
#include "volume.h"
#include "config.h"
#include "remote.h"


/*
 * A programmatic way of doing "rm -rf [dir_name]".
 */
static int destroy_tree(char * dir_name) {

	int rc = 0;
	int tmp_rc = 0;
	struct stat stat_buf;
	DIR * dir;
	struct dirent * dir_ent;
	char name_buf[PATH_MAX];
	int dir_path_len;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Request to destroy directory %s\n", dir_name);

	dir = opendir(dir_name);

	if (dir == NULL) {
		LOG_DEBUG("Failed to open directory %s.  Error code is %d: %s\n",
			  dir_name, errno, strerror(errno));
		LOG_PROC_EXIT_INT(errno);
		return errno;
	}

	strcpy(name_buf, dir_name);
	dir_path_len = strlen(name_buf);
	name_buf[dir_path_len] = '/';
	dir_path_len++;

	for (dir_ent = readdir(dir);
	     dir_ent != NULL;
	     dir_ent = readdir(dir)) {
		
		/* Skip the "." and ".." entries. */
		if ((strcmp(dir_ent->d_name,".") == 0) ||
		    (strcmp(dir_ent->d_name,"..") == 0)) {
			continue;
		}

		/*
		 * Build the full file name by appending the
		 * dir_ent->d_name to the directory name.
		 */
		strcpy(&name_buf[dir_path_len], dir_ent->d_name);

		if (stat(name_buf, &stat_buf) == 0) {

			/*
			 * If this entry is a directory, call this function
			 * recursively to process the subdirectory.
			 */
			if (S_ISDIR(stat_buf.st_mode)) {
				
				destroy_tree(name_buf);

				LOG_DEBUG("Remove directory %s\n", name_buf);
				tmp_rc = rmdir(name_buf);
				if (tmp_rc != 0) {
					LOG_DEBUG("Failed to remove directory %s.  Error code is %d: %s\n",
						  name_buf, rc, strerror(rc));

					/* Use the first error code. */
					if (rc == 0) {
						rc = tmp_rc;
					}
				}

			} else {
				rc = unlink(name_buf);
				if (rc != 0) {
					LOG_DEBUG("Failed to unlink %s.  Error code is %d: %s\n",
						  name_buf, rc, strerror(rc));

					/* Use the first error code. */
					if (rc == 0) {
						rc = tmp_rc;
					}
				}
			}
		}
	}

	closedir(dir);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

#define EVMS_NAMES_DIR		EVMS_DEV_NODE_PATH ".names"
#define EVMS_NAMES_PATH		EVMS_NAMES_DIR "/"
#define EVMS_NAMES_PATH_LEN	(sizeof(EVMS_NAMES_PATH) - 1)

char name_buf[EVMS_NAMES_PATH_LEN + EVMS_NAME_SIZE + 1] = EVMS_NAMES_PATH;

void build_names(list_anchor_t objects) {

	list_element_t iter;
	storage_object_t * obj;
	char * name_ptr = name_buf + EVMS_NAMES_PATH_LEN;
	char * last_slash;
	int status;
	struct stat stat_buf;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(objects, iter, obj) {
		if (obj->data_type != DATA_TYPE) {
			continue;
		}

		strcpy(name_ptr, obj->name);

		last_slash = strrchr(name_buf, '/');
		*last_slash = '\0';

		make_directory(name_buf,(S_IFDIR | S_IRWXU |
					 S_IRGRP | S_IXGRP |
					 S_IROTH | S_IXOTH));

		*last_slash = '/';

		LOG_DEBUG("Make dev node for \"%s\".\n", name_buf);
		status = stat(name_buf, &stat_buf);
		if (status == 0) {
			LOG_DEBUG("\"%s\" already exists.\n", name_buf);

		} else{
			status = mknod(name_buf, (S_IFCHR |
						  S_IRUSR | S_IWUSR |
						  S_IRGRP | S_IWGRP),
				       makedev(1,3));

			if (status != 0) {
				LOG_WARNING("Error creating node %s.  Error code was %d: %s\n",
					    name_buf, errno, strerror(errno));
			}
		}

		if (obj->consuming_container != NULL) {
			build_names(obj->consuming_container->objects_produced);

		} else if (!list_empty(obj->parent_objects)) {
			build_names(obj->parent_objects);

		} else if (obj->volume != NULL) {
			strcpy(name_ptr, obj->volume->name);

			last_slash = strrchr(name_buf, '/');
			*last_slash = '\0';

			make_directory(name_buf,(S_IFDIR | S_IRWXU |
						 S_IRGRP | S_IXGRP |
						 S_IROTH | S_IXOTH));

			*last_slash = '/';

			LOG_DEBUG("Make dev node for \"%s\".\n", name_buf);
			status = stat(name_buf, &stat_buf);
			if (status == 0) {
				LOG_DEBUG("\"%s\" already exists.\n", name_buf);

			} else{
				status = mknod(name_buf, (S_IFCHR |
							  S_IRUSR | S_IWUSR |
							  S_IRGRP | S_IWGRP),
					       makedev(1,3));

				if (status != 0) {
					LOG_WARNING("Error creating node %s.  Error code was %d: %s\n",
						    name_buf, errno, strerror(errno));
				}
			}
		}
	}

	LOG_PROC_EXIT_VOID();
	return;
}


/*
 * Build a tree with file names that correspond to the names of the volumes
 * and objects that were discovered.  glob() will be used on the tree to
 * find the names of objects that match the activate.include[] names and
 * don't match the activate.exclude[] names.
 */

static void build_names_tree(list_anchor_t discover_objects) {

	list_element_t iter;
	storage_object_t * obj;
	char * name_ptr = name_buf + EVMS_NAMES_PATH_LEN;
	char * last_slash;
	int status;
	struct stat stat_buf;

	LOG_PROC_ENTRY();

	destroy_tree(EVMS_NAMES_DIR);

	LIST_FOR_EACH(discover_objects, iter, obj) {
		if (obj->consuming_container != NULL) {
			build_names(obj->consuming_container->objects_produced);

		} else if (!list_empty(obj->parent_objects)) {
			build_names(obj->parent_objects);

		} else if (obj->volume != NULL) {
			strcpy(name_ptr, obj->volume->name);

			last_slash = strrchr(name_buf, '/');
			*last_slash = '\0';

			make_directory(name_buf,(S_IFDIR | S_IRWXU |
						 S_IRGRP | S_IXGRP |
						 S_IROTH | S_IXOTH));

			*last_slash = '/';

			LOG_DEBUG("Make dev node for \"%s\".\n", name_buf);
			status = stat(name_buf, &stat_buf);
			if (status == 0) {
				LOG_DEBUG("\"%s\" already exists.\n", name_buf);

			} else{
				status = mknod(name_buf, (S_IFCHR |
							  S_IRUSR | S_IWUSR |
							  S_IRGRP | S_IWGRP),
					       makedev(1,3));

				if (status != 0) {
					LOG_WARNING("Error creating node %s.  Error code was %d: %s\n",
						    name_buf, errno, strerror(errno));
				}
			}
		}
	}

	build_names(discover_objects);

	LOG_PROC_EXIT_VOID();
	return;
}


static void add_subdir_contents_to_glob(char * dir, glob_t * names_glob) {

	int rc;
	int first_name_index;
	int last_name_index;
        char pattern[(EVMS_NAME_SIZE + 1) * 2];
	int i;
	struct stat statbuf;
	int status;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Processing directory %s\n", dir);

	strcpy(pattern, dir);

	if (pattern[strlen(pattern) - 1] != '/') {
		strcat(pattern, "/");
	}
	strcat(pattern, "*");

	first_name_index = names_glob->gl_pathc;
	
	rc = glob(pattern, GLOB_APPEND, NULL, names_glob);

	last_name_index = names_glob->gl_pathc - 1;
	if (rc == 0) {

		for (i = first_name_index; i <= last_name_index; i++) {
			status = stat(names_glob->gl_pathv[i], &statbuf);
			if (status == 0) {
				if (S_ISDIR(statbuf.st_mode)) {
					add_subdir_contents_to_glob(names_glob->gl_pathv[i], names_glob);
				}
			}
		}

	} else {
		if (rc != GLOB_NOMATCH) {
			LOG_WARNING("glob() of pattern %s failed with error %s\n", pattern,
				    (rc == GLOB_NOSPACE) ? "GLOB_NOSPACE" :
				    (rc == GLOB_ABORTED) ? "GLOB_ABORTED" :
				    "(unknown)");
		}
	}


	LOG_PROC_EXIT_VOID();
}


/*
 * Given a directory on which to work, find the files that match the
 * names.  The resulting file names are returned in the names_glob.
 */
static void get_names_glob(int names_count,
			   const char * const * names,
			   glob_t * names_glob) {

	int rc;
        char pattern[(EVMS_NAME_SIZE + 1) * 2];
	int last_name_index;
	int i;
	int glob_flags = 0;

	LOG_PROC_ENTRY();

	strcpy(pattern, EVMS_NAMES_PATH);

	for (i = 0; i < names_count; i++) {
			
		/* Don't include the leading '/' of an include name. */
		if (*names[i] == '/') {
			strcpy(pattern + EVMS_NAMES_PATH_LEN, names[i] + 1);
		} else {
			strcpy(pattern + EVMS_NAMES_PATH_LEN, names[i]);
		}

		rc = glob(pattern, glob_flags, NULL, names_glob);

		if (rc == 0) {
			glob_flags |= GLOB_APPEND;

		} else {
			if (rc != GLOB_NOMATCH) {
				LOG_WARNING("glob() of pattern %s failed with error %s\n", pattern,
					    (rc == GLOB_NOSPACE) ? "GLOB_NOSPACE" :
					    (rc == GLOB_ABORTED) ? "GLOB_ABORTED" :
					    "(unknown)");
			}
		}
	}

	/* Include all subdirectory contents in the glob. */
	last_name_index = names_glob->gl_pathc - 1;

	for (i = 0; i <= last_name_index; i++) {
		int status;
		struct stat statbuf;

		status = stat(names_glob->gl_pathv[i], &statbuf);
		if (status == 0) {
			if (S_ISDIR(statbuf.st_mode)) {
				add_subdir_contents_to_glob(names_glob->gl_pathv[i], names_glob);
			}
		}
	}

	LOG_PROC_EXIT_VOID();
}


typedef struct hash_entry_s {
	char *                name;
	object_type_t         type;
	void *                thing;
	struct hash_entry_s * next;
} hash_entry_t;

/*
 * The hash table size should be prime number.  A prime number promotes
 * a more even distribution of entries in the hash table.
 * Other "prime" candidates that are near a power of 10 or power of 2:
 * 257, 499, 503, 509, 521, 997, 1009, 1021, 1031
 */
#define HASH_TABLE_SIZE 127

/*
 * Hash value computer.  Modeled after ElfHash().
 */
static u_int32_t hash_value_for_name(char * name) {

	u_int32_t h = 0;
	u_int32_t g;
	int i;
	u_char * pByte = (u_char *) name;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Get hash value for name \"%s\".\n", name);

        for (i = 0; pByte[i] != '\0'; i++) {
		h = (h << 4) + pByte[i];
		g = h & 0xF0000000;

		if (g != 0) {
			h ^= g >> 24;
		}

		h &= ~g;
	}

	LOG_PROC_EXIT_INT(h);
	return h;
}

#define HASH_INDEX(name) (hash_value_for_name(name) % HASH_TABLE_SIZE)


static void free_hash(hash_entry_t * * hash) {

	LOG_PROC_ENTRY();

	if (hash != NULL) {
		int i;

		for (i = 0; i < HASH_TABLE_SIZE; i++) {
			while (hash[i] != NULL) {
				hash_entry_t * next = hash[i]->next;
				engine_free(hash[i]);
				hash[i] = next;
			}
		}

		engine_free(hash);
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Build a hash for logical_volume_t and storage_object_t structures based on
 * the name of the volume or object.
 */
static hash_entry_t * * build_names_hash(void) {

	hash_entry_t * * names_hash;
	list_element_t iter;
	storage_object_t * obj;
	logical_volume_t * vol;
	u_int32_t hash_index;
	hash_entry_t * * slot;
	hash_entry_t * hash_entry;

	LOG_PROC_ENTRY();

	names_hash = engine_alloc(sizeof(hash_entry_t *) * HASH_TABLE_SIZE);

	if (names_hash != NULL) {

		LIST_FOR_EACH(&volumes_list, iter, vol) {
			/* Volume names don't use the leading '/'. */
			hash_index = HASH_INDEX(vol->name + 1);

			LOG_DEBUG("%s goes in index %u.\n", vol->name + 1, hash_index);
			slot = &names_hash[hash_index];
			while (*slot != NULL) {
				slot = &(*slot)->next;
			}
			hash_entry = engine_alloc(sizeof(hash_entry_t));
			if (hash_entry != NULL) {
				hash_entry->name = vol->name + 1;
				hash_entry->type = VOLUME;
				hash_entry->thing = vol;
				hash_entry->next = NULL;

				*slot = hash_entry;
			}
		}

		LIST_FOR_EACH(&disks_list, iter, obj) {
			hash_index = HASH_INDEX(obj->name);

			LOG_DEBUG("%s goes in index %u.\n", obj->name, hash_index);
			slot = &names_hash[hash_index];
			while (*slot != NULL) {
				slot = &(*slot)->next;
			}
			hash_entry = engine_alloc(sizeof(hash_entry_t));
			if (hash_entry != NULL) {
				hash_entry->name = obj->name;
				hash_entry->type = obj->object_type;
				hash_entry->thing = obj;
				hash_entry->next = NULL;

				*slot = hash_entry;
			}
		}

		LIST_FOR_EACH(&segments_list, iter, obj) {
			hash_index = HASH_INDEX(obj->name);

			LOG_DEBUG("%s goes in index %u.\n", obj->name, hash_index);
			slot = &names_hash[hash_index];
			while (*slot != NULL) {
				slot = &(*slot)->next;
			}
			hash_entry = engine_alloc(sizeof(hash_entry_t));
			if (hash_entry != NULL) {
				hash_entry->name = obj->name;
				hash_entry->type = obj->object_type;
				hash_entry->thing = obj;
				hash_entry->next = NULL;

				*slot = hash_entry;
			}
		}

		LIST_FOR_EACH(&regions_list, iter, obj) {
			hash_index = HASH_INDEX(obj->name);

			LOG_DEBUG("%s goes in index %u.\n", obj->name, hash_index);
			slot = &names_hash[hash_index];
			while (*slot != NULL) {
				slot = &(*slot)->next;
			}
			hash_entry = engine_alloc(sizeof(hash_entry_t));
			if (hash_entry != NULL) {
				hash_entry->name = obj->name;
				hash_entry->type = obj->object_type;
				hash_entry->thing = obj;
				hash_entry->next = NULL;

				*slot = hash_entry;
			}
		}

		LIST_FOR_EACH(&EVMS_objects_list, iter, obj) {
			hash_index = HASH_INDEX(obj->name);

			LOG_DEBUG("%s goes in index %u.\n", obj->name, hash_index);
			slot = &names_hash[hash_index];
			while (*slot != NULL) {
				slot = &(*slot)->next;
			}
			hash_entry = engine_alloc(sizeof(hash_entry_t));
			if (hash_entry != NULL) {
				hash_entry->name = obj->name;
				hash_entry->type = obj->object_type;
				hash_entry->thing = obj;
				hash_entry->next = NULL;

				*slot = hash_entry;
			}
		}
	}

	LOG_PROC_EXIT_PTR(names_hash);
	return names_hash;
}


/*
 * Scan the lists to find a volume or object with the name given.
 * Return a pointer to the volume or object and its type.
 */
static void * find_thing(char * name, object_type_t * p_type) {

	list_element_t iter;
	logical_volume_t * vol;
	storage_object_t * obj;
	
	LOG_PROC_ENTRY();

	LOG_DEBUG("Looking for \"%s\".\n", name);

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		/* Volume names don't have the leading '/'. */
		if (strcmp(name,vol->name + 1) == 0) {
			LOG_DEBUG("Found volume %s.\n", vol->name);
			*p_type = VOLUME;
			LOG_PROC_EXIT_PTR(vol);
			return vol;
		}
	}

	LIST_FOR_EACH(&disks_list, iter, obj) {
		if (strcmp(name,obj->name) == 0) {
			LOG_DEBUG("Found disk %s.\n", obj->name);
			*p_type = obj->object_type;
			LOG_PROC_EXIT_PTR(obj);
			return obj;
		}
	}

	LIST_FOR_EACH(&segments_list, iter, obj) {
		if (strcmp(name,obj->name) == 0) {
			LOG_DEBUG("Found segment %s.\n", obj->name);
			*p_type = obj->object_type;
			LOG_PROC_EXIT_PTR(obj);
			return obj;
		}
	}

	LIST_FOR_EACH(&regions_list, iter, obj) {
		if (strcmp(name,obj->name) == 0) {
			LOG_DEBUG("Found region %s.\n", obj->name);
			*p_type = obj->object_type;
			LOG_PROC_EXIT_PTR(obj);
			return obj;
		}
	}

	LIST_FOR_EACH(&EVMS_objects_list, iter, obj) {
		if (strcmp(name,obj->name) == 0) {
			LOG_DEBUG("Found EVMS object %s.\n", obj->name);
			*p_type = obj->object_type;
			LOG_PROC_EXIT_PTR(obj);
			return obj;
		}
	}

	*p_type = 0;
	LOG_PROC_EXIT_PTR(NULL);
	return NULL;
}


/*
 * Find an entry in the hash with the given name.  Return a pointer to the
 * thing and its type.
 */
static void * lookup_name(char * name, hash_entry_t * * names_hash, object_type_t * p_type) {
	
	void * thing = NULL;
	object_type_t type = 0;
	
	LOG_PROC_ENTRY();

	LOG_DEBUG("Lookup name \"%s\".\n", name);

	if (names_hash != NULL) {
		int hash_index;
		hash_entry_t * hash_entry;

		hash_index = HASH_INDEX(name);
		hash_entry = names_hash[hash_index];

		while ((hash_entry != NULL) &&
		       strcmp(hash_entry->name, name) != 0) {
			hash_entry = hash_entry->next;
		}

		if (hash_entry != NULL) {
			LOG_DEBUG("Found \"%s\".\n", name);
			type  = hash_entry->type;
			thing = hash_entry->thing;

		} else {
			LOG_DEBUG("\"%s\" not found.\n", name);
		}

	} else {
		/* There is no hash.  Lookup the name the O(n^2) way. */
		thing = find_thing(name, &type);
	}

	*p_type = type;
	LOG_PROC_EXIT_PTR(thing);
	return thing;
}


/*
 * Set the object's SOFLAG_NEEDS_ACTIVATE flag, then call myself
 * recursively to set the flags for all child objects this object needs.
 */
static void set_needs_activate(storage_object_t * obj) {

	list_element_t iter;
	storage_object_t * child;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Request to mark object %s as needing activation.\n", obj->name);

	if (!(obj->flags & SOFLAG_ACTIVE)) {
		LOG_DEBUG("Set SOFLAG_NEEDS_ACTIVATE for object %s.\n", obj->name);
		obj->flags |= SOFLAG_NEEDS_ACTIVATE;

	} else {
		LOG_DEBUG("Object %s is already marked for activation.\n", obj->name);
	}

	/* Cancel any pending deactivate. */
	LOG_DEBUG("Cancel pending deactivate on object %s.\n", obj->name);
	obj->flags &= ~SOFLAG_NEEDS_DEACTIVATE;

	LIST_FOR_EACH(obj->associated_children, iter, child) {
		set_needs_activate(child);
	}

	if (obj->producing_container != NULL) {
		LIST_FOR_EACH(obj->producing_container->objects_consumed, iter, child) {
			set_needs_activate(child);
		}

	} else {
		LIST_FOR_EACH(obj->child_objects, iter, child) {
			set_needs_activate(child);
		}
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * For each entry in names_glob, use the names_hash to find the volume or object
 * for the name.  If the volume or object is not active, set its NEEDS_ACTIVATE
 * flag.
 */
static void set_needs_activate_flags(glob_t * names_glob, hash_entry_t * * names_hash) {

	int i;
	char * name;
	object_type_t type;
	void * thing;
	storage_object_t * obj;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	for (i = 0; i < names_glob->gl_pathc; i++) {

		name = names_glob->gl_pathv[i] + EVMS_NAMES_PATH_LEN;

		thing = lookup_name(name, names_hash, &type);

		if (thing != NULL) {

			switch (type) {
				case VOLUME:
					vol = (logical_volume_t *) thing;
					if (!(vol->flags & VOLFLAG_ACTIVE)) {
						LOG_DEBUG("Set VOLFLAG_NEEDS_ACTIVATE on volume %s.\n", vol->name);
						vol->flags |= VOLFLAG_NEEDS_ACTIVATE;

						/* Mark the volume's object to be activated. */
						set_needs_activate(vol->object);
					}
					break;

				default:
					/* Anything else is a storage object. */
					obj = (storage_object_t *) thing;

					if (!(obj->flags & SOFLAG_ACTIVE)) {
						set_needs_activate(obj);
					}
					break;
			}

		} else {
			LOG_DEBUG("Name %s was not found.\n", name);
		}
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Clear the object's SOFLAG_NEEDS_ACTIVATE flag, then call myself
 * recursively to clear the flags for all parent objects and volumes
 * using this object.
 */
static void clear_needs_activate(storage_object_t * obj) {

	list_element_t iter;
	storage_object_t * parent;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Clear SOFLAG_NEEDS_ACTIVATE on object %s.\n", obj->name);
	obj->flags &= ~SOFLAG_NEEDS_ACTIVATE;

	LIST_FOR_EACH(obj->associated_parents, iter, parent) {
		clear_needs_activate(parent);
	}

	if (obj->consuming_container != NULL) {
		LIST_FOR_EACH(obj->consuming_container->objects_produced, iter, parent) {
			clear_needs_activate(parent);
		}

	} else if (!list_empty(obj->parent_objects)) {
		LIST_FOR_EACH(obj->parent_objects, iter, parent) {
			clear_needs_activate(parent);
		}

	} else if (obj->volume != NULL) {
		LOG_DEBUG("Clear VOLFLAG_NEEDS_ACTIVATE on volume %s.\n", obj->volume->name);
		obj->volume->flags &= ~VOLFLAG_NEEDS_ACTIVATE;
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Set the object's SOFLAG_NEEDS_DEACTIVATE flag, then call myself
 * recursively to set the flags for all parent objects an volumes
 * using this object.
 */
static void set_needs_deactivate(storage_object_t * obj) {

	list_element_t iter;
	storage_object_t * parent;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Request to mark object %s as needing to be deactivated.\n", obj->name);

	if (obj->flags & SOFLAG_ACTIVE) {
		LOG_DEBUG("Set SOFLAG_NEEDS_DEACTIVATE for object %s.\n", obj->name);
		obj->flags |= SOFLAG_NEEDS_DEACTIVATE;
	} else {
		LOG_DEBUG("Object %s is already marked for deactivateion.\n", obj->name);
	}

        /* Cancel any pending activate. */
	LOG_DEBUG("Cancel any pending activate on object %s.\n", obj->name);
	obj->flags &= ~SOFLAG_NEEDS_ACTIVATE;

	LIST_FOR_EACH(obj->associated_parents, iter, parent) {
		set_needs_deactivate(parent);
	}

	if (obj->consuming_container != NULL) {
		LIST_FOR_EACH(obj->consuming_container->objects_produced, iter, parent) {
			set_needs_deactivate(parent);
		}

	} else if (!list_empty(obj->parent_objects)) {
		LIST_FOR_EACH(obj->parent_objects, iter, parent) {
			set_needs_deactivate(parent);
		}

	} else if (obj->volume != NULL) {
		logical_volume_t * vol = obj->volume;

		if (vol->flags & VOLFLAG_ACTIVE) {
			LOG_DEBUG("Set  VOLFLAG_NEEDS_DEACTIVATE on volume %s.\n", obj->volume->name);
			vol->flags |= VOLFLAG_NEEDS_DEACTIVATE;
		}

		/* Cancel any pending activate. */
		LOG_DEBUG("Cancel any pending activate on volume %s.\n", obj->volume->name);
		vol->flags &= ~VOLFLAG_NEEDS_ACTIVATE;
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * For each entry in names_glob, use the names_hash to find the volume or object
 * for the name.  Make sure the NEEDS_ACTIVATE is cleared for the object and all
 * parent objects and volumes that use the object.
 */
static void clear_needs_activate_flags(glob_t * names_glob, hash_entry_t * * names_hash) {

	int i;
	char * name;
	object_type_t type;
	void * thing;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	for (i = 0; i < names_glob->gl_pathc; i++) {

		name = names_glob->gl_pathv[i] + EVMS_NAMES_PATH_LEN;

		thing = lookup_name(name, names_hash, &type);

		if (thing != NULL) {

			switch (type) {
				case VOLUME:
					vol = (logical_volume_t *) thing;

					LOG_DEBUG("Clear VOLFLAG_NEEDS_ACTIVATE on volume %s.\n", vol->name);
					vol->flags &= ~VOLFLAG_NEEDS_ACTIVATE;
					break;

				default:
					/* Anything else is a storage object. */
					clear_needs_activate((storage_object_t *) thing);
					break;
			}

		} else {
			LOG_DEBUG("Name %s was not found.\n", name);
		}
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Use the activate{} section in evms.conf to determine which volumes and
 * objects should be activated.  If the activate.include is not found, the
 * default is to activate everything (include = [*]).  If activate.exclude is
 * not found, the default is to exclude nothing (exclude = []).
 */
void mark_for_activation(list_anchor_t discover_objects) {

	int rc;
	int include_count;
        const char * const * include_names;
        const char * default_include_names[] = {"*"};
	int exclude_count;
        const char * const * exclude_names;
	glob_t include_names_glob = {0};
	glob_t exclude_names_glob = {0};
	hash_entry_t * * names_hash;

	LOG_PROC_ENTRY();

	build_names_tree(discover_objects);

	names_hash = build_names_hash();

	rc = evms_get_config_string_array("activate.include", &include_count, &include_names);
	if (rc != 0) {
		include_count = 1;
		include_names = default_include_names;
	}

	get_names_glob(include_count, include_names, &include_names_glob);

	set_needs_activate_flags(&include_names_glob, names_hash);

	rc = evms_get_config_string_array("activate.exclude", &exclude_count, &exclude_names);
	if (rc != 0) {
		exclude_count = 0;
		exclude_names = NULL;
	}

	if (exclude_count != 0) {
		get_names_glob(exclude_count, exclude_names, &exclude_names_glob);
	}

	if (exclude_names_glob.gl_pathc != 0) {
		clear_needs_activate_flags(&exclude_names_glob, names_hash);
	}

	globfree(&include_names_glob);
	
	if (exclude_count != 0) {
		globfree(&exclude_names_glob);
	}

	destroy_tree(EVMS_NAMES_DIR);

	free_hash(names_hash);

	LOG_PROC_EXIT_VOID();
}


/*
 * Check if this object and all its underlying objects can be activated.
 */
static int can_activate_object(storage_object_t * obj,
			       debug_level_t      log_level) {

	int rc = 0;
	list_element_t iter;
	storage_object_t * child;

	LOG_PROC_ENTRY();

	if (!(obj->flags & SOFLAG_ACTIVE)) {
		rc = obj->plugin->functions.plugin->can_activate(obj);

		if (rc == 0) {
			if (obj->producing_container != NULL) {
				LIST_FOR_EACH(obj->producing_container->objects_consumed, iter, child) {
					rc = can_activate_object(child, log_level);
				}

			} else {
				LIST_FOR_EACH(obj->child_objects, iter, child) {
					rc = can_activate_object(child, log_level);
				}
			}

		} else {
			LOG(log_level, "The %s plug-in cannot activate object %s.  "
			    "Error code is %d: %s\n",
			    obj->plugin->short_name, obj->name, rc, evms_strerror(rc));
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_activate(engine_handle_t handle,
			debug_level_t   log_level) {

	int rc = 0;
	void * thing;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = translate_handle(handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	switch (type) {
		case VOLUME:
			{
				logical_volume_t * vol = (logical_volume_t *) thing;


				if (is_kernel_volume_mounted(vol, log_level)) {
					LOG_PROC_EXIT_INT(EBUSY);
					rc = EBUSY;
					break;
				}

				if ((vol->flags & VOLFLAG_ACTIVE) &&
				    !(vol->flags & VOLFLAG_NEEDS_DEACTIVATE)) {
					LOG_DETAILS("Volume %s is already active.\n", vol->name);
					rc = EEXIST;
				}

				if (vol->flags & VOLFLAG_NEEDS_ACTIVATE) {
					LOG_DETAILS("Volume %s is already scheduled to be activated.\n", vol->name);
					rc = EEXIST;
				}
			}
                        break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			{
				storage_object_t * obj = (storage_object_t *) thing;

				/* Non-DATA_TYPE objects cannot be activated. */
                                if (obj->data_type != DATA_TYPE) {
					LOG(log_level, "Object %s is not a data object.\n", obj->name);
					rc = EINVAL;
				}

				if ((obj->flags & SOFLAG_ACTIVE) &&
				    !(obj->flags & SOFLAG_NEEDS_DEACTIVATE)) {
					LOG_DETAILS("Object %s is already active.\n", obj->name);
					rc = EEXIST;
				}
				
				if (obj->flags & SOFLAG_NEEDS_ACTIVATE) {
					LOG_DETAILS("Object %s is already scheduled to be activated.\n", obj->name);
					rc = EEXIST;
				}

                                if (rc == 0) {
					rc = can_activate_object(obj, log_level);
				}
			}
			break;

		case CONTAINER:
			{
				storage_container_t * con = (storage_container_t *) thing;

				LOG(log_level, "Handle %u is for container %s.  Containers cannot be activated.\n",
				    handle, con->name);
				rc = EINVAL;
			}
			break;

		case PLUGIN:
			{
				plugin_record_t * plugrec = (plugin_record_t *) thing;

				LOG(log_level, "Handle %u is for plug-in %s.  Plug-ins cannot be activated.\n",
				    handle, plugrec->short_name);
				rc = EINVAL;
				break;
			}

		case TASK:
			LOG(log_level, "Handle %u is for a task.  Tasks cannot be activated.\n",
			    handle);
			rc = EINVAL;
			break;

		default:
			LOG(log_level, "Handle %u is for a a thing of unknown type %#x.\n",
			    handle, type);
			rc = EINVAL;
			break;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_can_activate(object_handle_t handle) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_activate(handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_activate(handle, DETAILS);

	/*
	 * If the thing is already activated or is scheduled to be activated
	 * (indicated by EEXIST), say the thing is not valid.  Leaving the
	 * return code at EEXIST will give the user a confusing "File exists"
	 * error message.  Don't return 0 or the UIs will precent "Activate" as
	 * a valid operation on this thing when it is not needed.
	 */
	if (rc == EEXIST) {
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_activate(object_handle_t handle) {

	int rc;
	void * thing;
	object_type_t type;
	storage_object_t * obj;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_activate(handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_activate(handle, ERROR);
	if (rc != 0) {
		if (rc == EEXIST) {
			/*
			 * The thing is already active or scheduled to be
			 * activated.  That's OK.
			 */
			LOG_PROC_EXIT_INT(0);
			return 0;

		} else {
			LOG_PROC_EXIT_INT(rc);
			return rc;
		}
	}

        /*
	 * can_activate() verified that the handle is valid, so we don't need
	 * to check the return code from translate_handle().
	 */
	translate_handle(handle,
			 &thing,
			 &type);

	switch (type) {
		case VOLUME:
			vol = (logical_volume_t *) thing;

			if (!(vol->flags & VOLFLAG_ACTIVE)) {
				vol->flags |= VOLFLAG_NEEDS_ACTIVATE;
			}

			/* Cancel any pending deactivation. */
			vol->flags &= ~VOLFLAG_NEEDS_DEACTIVATE;

			set_needs_activate(vol->object);
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			obj = (storage_object_t *) thing;

			set_needs_activate(obj);
			break;

		default:
			/*
			 * We should not get here since can_activate()
			 * verified that the type was a VOLUME, EVMS_OBJECT,
			 * REGION, SEGMENT, or DISK.
			 */
			LOG_ERROR("Cannot activate handle %u of type %#x.\n",
				  handle, type);
			rc = EINVAL;
			break;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_deactivate_object(storage_object_t * obj,
				 debug_level_t      log_level) {
	int rc = 0;
	list_element_t iter;
	storage_object_t * parent;

	LOG_PROC_ENTRY();

	if (obj->flags & SOFLAG_ACTIVE) {
		rc = obj->plugin->functions.plugin->can_deactivate(obj);
		if (rc == 0) {
			if (obj->consuming_container != NULL) {
				LIST_FOR_EACH(obj->consuming_container->objects_produced, iter, parent) {
					rc = can_deactivate_object(parent, log_level);
				}

			} else if (!list_empty(obj->parent_objects)) {
				LIST_FOR_EACH(obj->parent_objects, iter, parent) {
					rc = can_deactivate_object(parent, log_level);
				}

			} else if (obj->volume != NULL) {
				if (is_volume_mounted(obj->volume)) {
					LOG(log_level, "Object %s is part of volume %s which is mounted on %s.\n",
					    obj->name, obj->volume->name, obj->volume->mount_point);
					rc = EBUSY;
				}
			}

		} else {
			LOG(log_level, "The %s plug-in cannot deactivate object %s.  "
			    "Error code is %d: %s\n",
			    obj->plugin->short_name, obj->name, rc, evms_strerror(rc));
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_deactivate(engine_handle_t handle,
			  debug_level_t   log_level) {

	int rc = 0;
	void * thing;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = translate_handle(handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	switch (type) {
		case VOLUME:
			{
				logical_volume_t * vol = (logical_volume_t *) thing;

				if (is_volume_mounted(vol)) {
					LOG(log_level, "Volume %s is mounted on %s.\n",
					    vol->name, vol->mount_point);
					rc = EBUSY;
					break;
				}

				if (!(vol->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEEDS_ACTIVATE))) {
					LOG_DETAILS("Volume %s is not active.\n", vol->name);
					rc = EEXIST;
				}

				if (vol->flags & VOLFLAG_NEEDS_DEACTIVATE) {
					LOG_DETAILS("Volume %s is already marked to be deactivated.\n", vol->name);
					rc = EEXIST;
				}

				if (!(vol->flags & VOLFLAG_HAS_OWN_DEVICE)) {
					LOG(log_level, "Volume %s does not have its own device.  "
					    "The volume is deactivated when its object, %s, is deactivated.\n",
					    vol->name, vol->object->name);
					rc = EINVAL;
				}
			}
                        break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			{
				storage_object_t * obj = (storage_object_t *) thing;

				if (!(obj->flags & (SOFLAG_ACTIVE | SOFLAG_NEEDS_ACTIVATE))) {
					LOG_DETAILS("Object %s is not active.\n", obj->name);
					rc = EEXIST;
				}
				
				if (obj->flags & SOFLAG_NEEDS_DEACTIVATE) {
					LOG_DETAILS("Object %s is already marked to be deactivated.\n", obj->name);
					rc = EEXIST;
				}

				if (rc == 0) {
					rc = can_deactivate_object(obj, log_level);
				}
			}
			break;

		case CONTAINER:
			{
				storage_container_t * con = (storage_container_t *) thing;

				LOG(log_level, "Handle %u is for container %s.  Containers cannot be deactivated.\n",
				    handle, con->name);
				rc = EINVAL;
			}
			break;

		case PLUGIN:
			{
				plugin_record_t * plugrec = (plugin_record_t *) thing;

				LOG(log_level, "Handle %u is for plug-in %s.  Plug-ins cannot be deactivated.\n",
				    handle, plugrec->short_name);
				rc = EINVAL;
			}
			break;

		case TASK:
			LOG(log_level, "Handle %u is for a task.  Tasks cannot be deactivated.\n",
			    handle);
			rc = EINVAL;
			break;

		default:
			LOG(log_level, "Handle %u is for a thing of unknown type %#x.\n",
			    handle, type);
			rc = EINVAL;
			break;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_can_deactivate(object_handle_t handle) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_deactivate(handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_deactivate(handle, DETAILS);

	/*
	 * If the thing is already deactivated or is scheduled to be deactivated
	 * (indicated by EEXIST), say the thing is not valid.  Leaving the
	 * return code at EEXIST will give the user a confusing "File exists"
	 * error message.  Don't return 0 or the UIs will precent "Deactivate"
	 * as a valid operation on this thing when it is not needed.
	 */
	if (rc == EEXIST) {
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_deactivate(object_handle_t handle) {

	int rc;
	void * thing;
	object_type_t type;
	storage_object_t * obj;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_deactivate(handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_deactivate(handle, ERROR);
	if (rc != 0) {
		if (rc == EEXIST) {
			/*
			 * The thing is already deactivated or scheduled to be
			 * deactivated.  That's OK.
			 */
			LOG_PROC_EXIT_INT(0);
			return 0;

		} else {
			LOG_PROC_EXIT_INT(rc);
			return rc;
		}
	}

        /*
	 * can_deactivate() verified that the handle is valid, so we don't need
	 * to check the return code from translate_handle().
	 */
	translate_handle(handle,
			 &thing,
			 &type);

	switch (type) {
		case VOLUME:
			vol = (logical_volume_t *) thing;

			if (vol->flags & VOLFLAG_ACTIVE) {
				vol->flags |= VOLFLAG_NEEDS_DEACTIVATE;
			}

			/* Cancel any pending activate. */
			vol->flags &= ~VOLFLAG_NEEDS_ACTIVATE;
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			obj = (storage_object_t *) thing;

			set_needs_deactivate(obj);
			break;

		default:
			/*
			 * We should not get here since can_deactivate()
			 * verified that the type was a VOLUME, EVMS_OBJECT,
			 * REGION, SEGMENT, or DISK.
			 */
			LOG_ERROR("Cannot activate handle %u of type %#x.\n",
				  handle, type);
			rc = EINVAL;
			break;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

