# -*- coding: utf-8 -*-

# Copyright (c) 2006 - 2008 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the Editor Colours configuration page.
"""

from PyQt4.QtCore import pyqtSignature
from PyQt4.QtGui import QPixmap, QIcon, QColor

from KdeQt import KQColorDialog

from ConfigurationPageBase import ConfigurationPageBase
from Ui_EditorColoursPage import Ui_EditorColoursPage

import Preferences

class EditorColoursPage(ConfigurationPageBase, Ui_EditorColoursPage):
    """
    Class implementing the Editor Colours configuration page.
    """
    def __init__(self):
        """
        Constructor
        """
        ConfigurationPageBase.__init__(self)
        self.setupUi(self)
        self.setObjectName("EditorColoursPage")
        
        self.editorColours = {}
        
        # set initial values
        self.__initEditorColoursColour("CurrentMarker", 
            self.currentLineMarkerButton)
        self.__initEditorColoursColour("ErrorMarker", 
            self.errorMarkerButton)
        self.__initEditorColoursColour("MatchingBrace", 
            self.matchingBracesButton)
        self.__initEditorColoursColour("MatchingBraceBack", 
            self.matchingBracesBackButton)
        self.__initEditorColoursColour("NonmatchingBrace", 
            self.nonmatchingBracesButton)
        self.__initEditorColoursColour("NonmatchingBraceBack", 
            self.nonmatchingBracesBackButton)
        self.__initEditorColoursColour("CallTipsBackground", 
            self.calltipsBackgroundButton)
        self.__initEditorColoursColour("CaretForeground", 
            self.caretForegroundButton)
        self.__initEditorColoursColour("CaretLineBackground", 
            self.caretlineBackgroundButton)
        self.__initEditorColoursColour("SelectionForeground", 
            self.selectionForegroundButton)
        self.__initEditorColoursColour("SelectionBackground", 
            self.selectionBackgroundButton)
        
    def save(self):
        """
        Public slot to save the Editor Colours configuration.
        """
        for key in self.editorColours.keys():
            Preferences.setEditorColour(key, self.editorColours[key])
        
    def __initEditorColoursColour(self, colourstr, button):
        """
        Private method to initialize the colour buttons.
        
        @param colourstr Colour to be set. (string)
        @param button Reference to a QButton to show the colour on.
        """
        colour = QColor(Preferences.getEditorColour(colourstr))
        self.editorColours[colourstr] = colour
        size = button.size()
        pm = QPixmap(size.width()/2, size.height()/2)
        pm.fill(colour)
        button.setIconSize(pm.size())
        button.setIcon(QIcon(pm))
        
    def __selectEditorColour(self, colourstr, button):
        """
        Private method used by the buttons selecting the marker colours.
        
        @param colourstr Colour to be set. (string)
        @param button Reference to a QButton to show the colour on.
        """
        colour = KQColorDialog.getColor(self.editorColours[colourstr])
        if colour.isValid():
            self.editorColours[colourstr] = colour
            size = button.iconSize()
            pm = QPixmap(size.width(), size.height())
            pm.fill(colour)
            button.setIcon(QIcon(pm))
        
    @pyqtSignature("")
    def on_currentLineMarkerButton_clicked(self):
        """
        Private slot to set the colour for the highlight of the current line.
        """
        self.__selectEditorColour("CurrentMarker", 
            self.currentLineMarkerButton)
        
    @pyqtSignature("")
    def on_errorMarkerButton_clicked(self):
        """
        Private slot to set the colour for the highlight of the error line.
        """
        self.__selectEditorColour("ErrorMarker", 
            self.errorMarkerButton)
        
    @pyqtSignature("")
    def on_matchingBracesButton_clicked(self):
        """
        Private slot to set the colour for highlighting matching braces.
        """
        self.__selectEditorColour("MatchingBrace", 
            self.matchingBracesButton)
        
    @pyqtSignature("")
    def on_matchingBracesBackButton_clicked(self):
        """
        Private slot to set the background colour for highlighting matching braces.
        """
        self.__selectEditorColour("MatchingBraceBack", 
            self.matchingBracesBackButton)
        
    @pyqtSignature("")
    def on_nonmatchingBracesButton_clicked(self):
        """
        Private slot to set the colour for highlighting nonmatching braces.
        """
        self.__selectEditorColour("NonmatchingBrace", 
            self.nonmatchingBracesButton)
        
    @pyqtSignature("")
    def on_nonmatchingBracesBackButton_clicked(self):
        """
        Private slot to set the background colour for highlighting nonmatching braces.
        """
        self.__selectEditorColour("NonmatchingBraceBack", 
            self.nonmatchingBracesBackButton)
        
    @pyqtSignature("")
    def on_calltipsBackgroundButton_clicked(self):
        """
        Private slot to set the background colour for calltips.
        """
        self.__selectEditorColour("CallTipsBackground", 
            self.calltipsBackgroundButton)
        
    @pyqtSignature("")
    def on_caretForegroundButton_clicked(self):
        """
        Private slot to set the foreground colour of the caret.
        """
        self.__selectEditorColour("CaretForeground", 
            self.caretForegroundButton)
        
    @pyqtSignature("")
    def on_caretlineBackgroundButton_clicked(self):
        """
        Private slot to set the background colour of the caretline.
        """
        self.__selectEditorColour("CaretLineBackground", 
            self.caretlineBackgroundButton)
        
    @pyqtSignature("")
    def on_selectionForegroundButton_clicked(self):
        """
        Private slot to set the foreground colour of the selection.
        """
        self.__selectEditorColour("SelectionForeground", 
            self.selectionForegroundButton)
        
    @pyqtSignature("")
    def on_selectionBackgroundButton_clicked(self):
        """
        Private slot to set the background colour of the selection.
        """
        self.__selectEditorColour("SelectionBackground", 
            self.selectionBackgroundButton)
    
def create(dlg):
    """
    Module function to create the configuration page.
    
    @param dlg reference to the configuration dialog
    """
    page = EditorColoursPage()
    return page
