import logging

from enthought.traits.api import Bool, HasTraits, Str
from enthought.traits.ui.api import Group, Item, spring
from enthought.traits.has_dynamic_views import DynamicView, HasDynamicViews


# Setup a logger for this module.
logger = logging.getLogger(__name__)


class HasFooView(HasDynamicViews):
    """
    A base class declaring the existence of the 'foo' dynamic view.

    """

    def __init__(self, *args, **kws):
        """
        Constructor.

        Extended to declare our dynamic foo view.

        """

        super(HasFooView, self).__init__(*args, **kws)

        # Declare and add our dynamic view.
        declaration = DynamicView(name = 'foo',
            id = 'enthought.traits.ui.demos.dynamic_views',
            keywords = {
                'buttons' : [ 'OK' ],
                'dock' : 'tab',
                'height' : 0.4,
                'resizable' : True,
                'scrollable' : True,
                'width' : 0.4,
                },
            use_as_default = True,
            )
        self.declare_dynamic_view(declaration)

        return


class MyInfoHandler(HasTraits):
    def object_first_changed(self, info):
        info.object.derived = info.object.first
        return

    def init(self, info):
        return

class BaseFoo(HasFooView):
    """
    A base class that puts some content in the 'foo' dynamic view.

    """

    first = Str('My first name')
    last = Str('My last name')

    # A derived trait set by the handler associated with out dynamic view
    # contribution.
    derived = Str

    ui_person = Group(
        Item(label='On this tab, notice how the sub-handler keeps\n'
            'the derived value equal to the first name.\n\n'
            'On the next tab, change the selection in order to\n'
            'control which tabs are visible when the ui is \n'
            'displayed for the 2nd time.'
            ),
        spring,
        'first', 'last',
        spring,
        'derived',
        label = 'My Info',
        _foo_order = 5,
        _foo_priority = 1,
        _foo_handler = MyInfoHandler(),
        )


class FatherInfoHandler(HasTraits):
    def object_father_first_name_changed(self, info):
        info.object.father_derived = info.object.father_first_name
        return

    def init(self, info):
        return

class DerivedFoo(BaseFoo):
    """
    A derived class that puts additional content in the 'foo' dynamic view.
    Note that the additional content could also have been added via a traits
    category contribution, or even dynamic manipulation of metadata on a UI
    subelement.  The key is what the metadata represents when the view is
    *created*

    """

    knows_mother = Bool(False)
    mother_first_name = Str('My mother\'s first name')
    mother_last_name = Str('My mother\'s last name')

    knows_father = Bool(True)
    father_first_name = Str('My father\'s first name')
    father_last_name = Str('My father\'s last name')
    father_derived = Str

    ui_parents = Group('knows_mother', 'knows_father',
        label = 'Parents?',
        _foo_order = 7,
        _foo_priority = 1,
        )

    ui_mother = Group('mother_first_name', 'mother_last_name',
        label = 'Mother\'s Info',
        #_foo_order=10,
        _foo_priority=1,
        )

    ui_father = Group('father_first_name', 'father_last_name',
        spring,
        'father_derived',
        label = 'Father\'s Info',
        _foo_order=15,
        _foo_priority=1,
        _foo_handler = FatherInfoHandler(),
        )


    def _knows_mother_changed(self, old, new):
        ui_mother = self.trait_view('ui_mother')
        if new:
            logger.debug('Setting foo order on ui_mother')
            ui_mother._foo_order = 10
        elif hasattr(ui_mother, '_foo_order'):
            logger.debug('Clearing foo order on ui_mother')
            del ui_mother._foo_order

        return


    def _knows_father_changed(self, old, new):
        ui_father = self.trait_view('ui_father')
        if new:
            logger.debug('Setting foo order on ui_father')
            ui_father._foo_order = 15
        elif hasattr(ui_father, '_foo_order'):
            logger.debug('Clearing foo order on ui_father')
            del ui_father._foo_order

        return



if __name__ == '__main__':
    f = DerivedFoo()
    f.configure_traits()
    f.configure_traits()


### EOF ######################################################################

