"""
This is the Qt backend that draws into a normal Qt graphics context using the
Image backend.  It detects the OS type and creates the appropriate low-level
support classes to enable Agg to draw into the WX memory buffer.
"""

# These are the symbols that a backend has to define
__all__ = ["GraphicsContext", "Canvas", "CompiledPath",
           "font_metrics_provider"]

from PyQt4.QtGui import QWidget
import sys

class BaseQtCanvas(QWidget):
    def __init__(self, parent=0, flags=0):
        QWidget.__init__(self, parent, flags)
        self.gc = None
        self.clear_color = (1,1,1)
        self.new_gc(size=None)
    
    def do_draw(self, gc):
        """ Method to be implemented by subclasses to actually perform various
        GC drawing commands before the GC is blitted into the screen.
        """
        pasAs

    def paintEvent(self, event):
        # clear the gc
        self.gc.clear(self.clear_color)

        # draw on the GC
        self.do_draw(self.gc)

        # blit the pixel data from the graphics context
        self.blit(event)

    def sizeEvent(self, event):
        width, height = self.size
        if (width != self.gc.width()) or (height != self.gc.height()):
            self.new_gc((width, height))

    def new_gc(self, size):
        """ Creates a new GC of the requested size (or of the widget's current
        size if size is None) and stores it in self.gc.
        """
        raise NotImplementedError

    def blit(self, event):
        """ Blits the pixels in self.gc into the screen buffer. """
        raise NotImplementedError


if sys.platform != "darwin":
    from backend_image import GraphicsContextSystem as GraphicsContext

    # For non-Mac backends, use the CompiledPath object defined in the agg
    # subpackage
    from agg import CompiledPath

    class Canvas(BaseQtCanvas): 
        def new_gc(self, size):
            if size is None:
                size = self.size

            if self.gc is not None:
                del self.gc
                self.gc = None

            width, height = size
            self.gc = GraphicsContext((width, height))
        
        def blit(self, event):
            # Use the PixelMap that the GraphicsContextSystem constructor attached
            # to self.gc to draw from the gc directly into this widget.
            self.gc.pixel_map.draw_to_qtwindow(self, 0, 0)

else:

    from mac import ABCGI
    from mac.ABCGI import CGBitmapContext, CGImage, CGLayerContext, CGMutablePath
    
    # For the Mac backend, use the OS-provided CGMutablePath as our compiled
    # path implementation.
    CompiledPath = CGMutablePath

    class UnflippingCGContext(ABCGI.CGContextInABox):
        """ Vertically flip the context to undo toolkit's flipping.  """

        def __init__(self, *args, **kwds):
            ABCGI.CGContextInABox.__init__(self, *args, **kwds)
            self._begun = False

        def begin(self):
            if self._begun:
                return
            self.save_state()
            self.translate_ctm(0, self.height())
            self.scale_ctm(1.0, -1.0)
            self._begun = True

        def end(self):
            if self._begun:
                self.restore_state()
                self._begun = False

    class GraphicsContext(CGLayerContext):
        def __init__(self, size_or_array, *args, **kwds):
            # Create a tiny base context to spawn the CGLayerContext from.
            bitmap = CGBitmapContext((1,1))
            if isinstance(size_or_array, np.ndarray):
                # Initialize the layer with an image.
                image = CGImage(size_or_array)
                width = image.width
                height = image.height
            else:
                # No initialization.
                image = None
                width, height = size_or_array
            CGLayerContext.__init__(self, bitmap, 
                (width, height))
            if image is not None:
                self.draw_image(image)

    class Canvas(BaseQtCanvas):

        def new_gc(self, size):
            # Create a memory device context
            # self.memDC = ...

            # Wrap it in a CGcontext that doesn't flip the Y coordinate
            #return UnflippingCGContext(self.memDC.macCGHandle(), self.size)

            pass


        def blit(self, event):
            # Draw the memory dc (self.memDC) into our widget's client area
            pass


   
def font_metrics_profider():
    """ Creates an object to be used for querying font metrics.  Typically
    this can just be an empty/dummy graphics context
    """
    return GraphicsContext((1,1))



